/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.awt.Graphics2D;
import java.awt.Transparency;
import java.awt.color.ColorSpace;
import java.awt.image.BufferedImage;
import java.awt.image.ComponentColorModel;
import java.awt.image.DataBuffer;
import java.awt.image.DataBufferByte;
import java.awt.image.DataBufferUShort;
import java.awt.image.Raster;
import java.awt.image.WritableRaster;
import java.io.File;
import java.io.IOException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import javax.imageio.ImageIO;

import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.RenderResolution;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.core.AudioBuffer;
import ch.kuramo.javie.core.MediaFileInput;
import ch.kuramo.javie.core.VideoBuffer;
import ch.kuramo.javie.core.services.VideoRenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;

import com.google.inject.Inject;

public class JavaImageInput implements MediaFileInput {

	private static class Entry {
		private final Size2i size;
		private final Object array;

		private Entry(Size2i size, Object array) {
			this.size = size;
			this.array = array;
		}
	}


	private BufferedImage _srcImage;

	private Size2i _srcSize;

	private VideoBounds _srcBounds;

	private ColorMode _srcColorMode;

	// TODO キャッシュを実装したらそれを使う。
	private final ConcurrentMap<RenderResolution, Entry> _cache = new ConcurrentHashMap<RenderResolution, Entry>();

	@Inject
	private VideoRenderContext _vrContext;

	@Inject
	private VideoRenderSupport _vrSupport;


	public JavaImageInput() {
		super();
	}

	public boolean initialize(File file) {
		try {
			_srcImage = ImageIO.read(file);
			if (_srcImage == null) return false;
		} catch (IOException e) {
			return false;
		}

		_srcSize = new Size2i(_srcImage.getWidth(), _srcImage.getHeight());
		_srcBounds = new VideoBounds(_srcImage.getWidth(), _srcImage.getHeight());

		if (_srcImage.getColorModel().getPixelSize() > 32) {
			_srcColorMode = ColorMode.RGBA16;
		} else {
			_srcColorMode = ColorMode.RGBA8;
		}

		// TODO 大きな画像の場合には時間がかかるので、バックグラウンドスレッドで処理した方が良いかも。
		_cache.put(RenderResolution.FULL, new Entry(_srcSize, toInternalFormat(_srcImage, _srcSize)));

		return true;
	}

	private Object toInternalFormat(BufferedImage srcImage, Size2i scaleTo) {
		// 一度BufferedImage上でdrawImageして、OpenGLの内部フォーマットに合わせる。

		int dataType = (_srcColorMode == ColorMode.RGBA16)
				? DataBuffer.TYPE_USHORT : DataBuffer.TYPE_BYTE;

		WritableRaster wr = Raster.createInterleavedRaster(
				dataType, scaleTo.width, scaleTo.height,
				scaleTo.width*4, 4, new int[] { 2, 1, 0, 3 }, null);
		ComponentColorModel cm = new ComponentColorModel(
				ColorSpace.getInstance(ColorSpace.CS_sRGB), true, true,
				Transparency.TRANSLUCENT, dataType);
		BufferedImage tmpImage = new BufferedImage(cm, wr, true, null);

		Graphics2D g = tmpImage.createGraphics();
		g.drawImage(srcImage, 0, 0, scaleTo.width, scaleTo.height, null);

		DataBuffer dataBuffer = tmpImage.getRaster().getDataBuffer();
		return (dataType == DataBuffer.TYPE_USHORT)
				? ((DataBufferUShort) dataBuffer).getData()
				: ((DataBufferByte) dataBuffer).getData();
	}

	public void dispose() {
		// nothing to do
	}

	public Time getDuration() {
		return null;
	}

	public Time getVideoFrameDuration() {
		return null;
	}

	public VideoBounds getVideoFrameBounds() {
		return _srcBounds;
	}

	public VideoBuffer getVideoFrameImage() {
		RenderResolution resolution = _vrContext.getRenderResolution();

		// 以下、_cache に対する get と put がアトミックでないが (_cache自体はConcurrentMap)、
		// 別スレッドが put で上書きしたとしても、同じ内容で上書きするだけなので支障はない。
		Entry entry = _cache.get(resolution);
		if (entry == null) {
			Size2i scaleTo = resolution.scale(_srcSize);
			entry = new Entry(scaleTo, toInternalFormat(_srcImage, scaleTo));
			_cache.put(resolution, entry);
		}

		ColorMode contextColorMode = _vrContext.getColorMode();
		VideoBuffer videoBuffer = _vrSupport.createVideoBuffer(contextColorMode, entry.size);
		videoBuffer.allocateAsTexture();

		_vrSupport.copyArrayToTexture(entry.array, videoBuffer.getTexture(), _srcColorMode, entry.size);

		return videoBuffer;
	}

	public boolean isVideoAvailable() {
		return true;
	}

	public boolean isAudioAvailable() {
		return false;
	}

	public AudioBuffer getAudioChunk() {
		throw new UnsupportedOperationException("audio is not available");
	}

}
