/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.io.File;

import javax.media.opengl.GL;
import javax.media.opengl.GLContext;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.AudioMode.DataType;
import ch.kuramo.javie.core.AudioBuffer;
import ch.kuramo.javie.core.MediaFileInput;
import ch.kuramo.javie.core.VideoBuffer;
import ch.kuramo.javie.core.services.AudioRenderContext;
import ch.kuramo.javie.core.services.AudioRenderSupport;
import ch.kuramo.javie.core.services.GLGlobal;
import ch.kuramo.javie.core.services.SynchronousTaskThread;
import ch.kuramo.javie.core.services.VideoRenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;
import ch.kuramo.javie.core.services.SynchronousTaskThread.Task;
import ch.kuramo.javie.core.services.SynchronousTaskThread.TaskWithoutResult;

import com.google.inject.Inject;

public class MacOSXQTCoreVideoInput implements MediaFileInput {

	private static final Logger _logger = LoggerFactory.getLogger(MacOSXQTCoreVideoInput.class);


	private GLContext _glContext;

	private int _fboId;

	private long _videoInput;

	private long _audioInput;

	private VideoBounds _bounds;

	private Time _duration;

	private Time _videoFrameDuration;

	private AudioMode _audioMode;

	@Inject
	private SynchronousTaskThread _videoThread;

	@Inject
	private SynchronousTaskThread _audioThread;

	@Inject
	private GLGlobal _glGlobal;

	@Inject
	private VideoRenderContext _vrContext;

	@Inject
	private VideoRenderSupport _vrSupport;

	@Inject
	private AudioRenderContext _arContext;

	@Inject
	private AudioRenderSupport _arSupport;


	public MacOSXQTCoreVideoInput() {
		super();
	}

	public boolean initialize(final File file) {
		_videoThread.start();
		_videoThread.invoke(new TaskWithoutResult() {
			protected void runWithoutResult() {
				initializeVideo(file);
			}
		});
		if (_videoInput == 0) {
			disposeVideo();
		}

		_audioThread.start();
		_audioThread.invoke(new TaskWithoutResult() {
			protected void runWithoutResult() {
				initializeAudio(file);
			}
		});
		if (_audioInput == 0) {
			disposeAudio();
		}

		return (_videoInput != 0 || _audioInput != 0);
	}

	private void initializeVideo(File file) {
		synchronized (_glGlobal.getLockObject()) {
			_glContext = _glGlobal.createContext();
			_glContext.makeCurrent();
		}

		//_glContext.setGL(new DebugGL(_glContext.getGL()));
		GL gl = _glContext.getGL();

		int[] fboId = new int[1];
		gl.glGenBuffers(1, fboId, 0);
		_fboId = fboId[0];

		gl.glBindFramebufferEXT(GL.GL_FRAMEBUFFER_EXT, _fboId);


		long pixelFormat = MacOSXOpenGLPixelFormat.createPixelFormat(_glContext);
		long[] result = openVideo(file.getAbsolutePath(), pixelFormat);
		MacOSXOpenGLPixelFormat.deletePixelFormat(pixelFormat);

		if (result != null) {
			_videoInput = result[0];
			_bounds = new VideoBounds((int) result[1], (int) result[2]);
			_duration = new Time(result[4], (int) result[5]);
			_videoFrameDuration = new Time(result[4] / result[3], (int) result[5]);
			_videoThread.setName(getClass().getSimpleName() + " (Video): " + file.getName());
		}
	}

	private void initializeAudio(File file) {
		long[] result = openAudio(file.getAbsolutePath());
		if (result != null) {
			_audioInput = result[0];
			if (_duration == null) {
				_duration = new Time(result[2], (int) result[3]);
			}
			_audioThread.setName(getClass().getSimpleName() + " (Audio): " + file.getName());
		}
	}

	public void dispose() {
		disposeVideo();
		disposeAudio();
	}

	private void disposeVideo() {
		_videoThread.exit(new TaskWithoutResult() {
			protected void runWithoutResult() {
				if (_videoInput != 0) {
					closeVideo(_videoInput);
					_videoInput = 0;
					_bounds = null;
				}

				if (_glContext != null) {
					GL gl = _glContext.getGL();

					gl.glBindFramebufferEXT(GL.GL_FRAMEBUFFER_EXT, 0);
					gl.glDeleteBuffers(1, new int[] { _fboId }, 0);
					_fboId = 0;

					synchronized (_glGlobal.getLockObject()) {
						_glContext.release();
						_glGlobal.destroyContext(_glContext);
					}

					_glContext = null;
				}
			}
		});
	}

	private void disposeAudio() {
		_audioThread.exit(new TaskWithoutResult() {
			protected void runWithoutResult() {
				if (_audioInput != 0) {
					closeAudio(_audioInput);
					_audioInput = 0;
					_audioMode = null;
				}
			}
		});
	}

	public boolean isVideoAvailable() {
		return (_videoInput != 0);
	}

	public boolean isAudioAvailable() {
		return (_audioInput != 0);
	}

	public Time getDuration() {
		return _duration;
	}

	public Time getVideoFrameDuration() {
		return _videoFrameDuration;
	}

	public VideoBounds getVideoFrameBounds() {
		return _bounds;
	}

	public VideoBuffer getVideoFrameImage() {
		if (_videoInput == 0) {
			return null;
		}

		final VideoBounds bounds = _vrContext.getRenderResolution().scale(_bounds);
		VideoBuffer vb = _vrSupport.createVideoBuffer(_vrContext.getColorMode(), bounds);
		vb.allocateAsTexture();

		final Time time = _vrContext.getTime();
		final int texture = vb.getTexture();

		int error = _videoThread.invoke(new Task<Integer>() {
			public Integer run() throws Exception {
				int error = frameImageAtTime(_videoInput, time.timeValue, time.timeScale, texture, bounds.width, bounds.height);
				if (error != 0) {
					_logger.error("frameImageAtTime: error " + error);
				}
				return error;
			}
		});

		if (error == 0) {
			return vb;
		} else {
			vb.dispose();
			return null;
		}
	}

	public AudioBuffer getAudioChunk() {
		if (_audioInput == 0) {
			return null;
		}

		final Time time = _arContext.getTime();
		final AudioBuffer ab = _arSupport.createAudioBuffer();

		int error = _audioThread.invoke(new Task<Integer>() {
			public Integer run() throws Exception {
				AudioMode am = ab.getAudioMode();
				if (am != _audioMode) {
					int error = setAudioDescription(
							_audioInput, am.channels, am.sampleRate, am.sampleSize,
							(am.dataType == DataType.FLOAT));
					if (error != 0) {
						_logger.error("setAudioDescription: error " + error);
						return error;
					}
					_audioMode = am;
				}

				int error = audioChunkFromTime(
						_audioInput, time.timeValue, time.timeScale, ab.getData(), ab.getFrameCount());
				if (error != 0) {
					_logger.error("audioChunkFromTime: error " + error);
				}
				return error;
			}
		});

		if (error == 0) {
			return ab;
		} else {
			ab.dispose();
			return null;
		}
	}


	static { System.loadLibrary("QTCoreVideoInput"); }

	private native long[] openVideo(String filename, long pixelFormat);

	private native long[] openAudio(String filename);

	private native void closeVideo(long videoInput);

	private native void closeAudio(long audioInput);

	private native int frameImageAtTime(
			long videoInput, long timeValue, int timeScale, int texture, int width, int height);

	private native int setAudioDescription(
			long audioInput, int channels, int sampleRate, int sampleSize, boolean floatingPoint);

	private native int audioChunkFromTime(
			long audioInput, long timeValue, int timeScale, Object buffer, int frameCount);

}
