/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.util.List;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.plugin.PIAnimatableValue;
import ch.kuramo.javie.api.plugin.PIAudioBuffer;
import ch.kuramo.javie.api.plugin.PIAudioRenderContext;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.AudioBuffer;
import ch.kuramo.javie.core.AudioEffect;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.services.AudioEffectPipeline;
import ch.kuramo.javie.core.services.AudioRenderContext;
import ch.kuramo.javie.core.services.AudioRenderSupport;

import com.google.inject.Inject;

public class AudioEffectPipelineImpl implements AudioEffectPipeline {

	@Inject
	private AudioRenderContext _arContext;

	@Inject
	private AudioRenderSupport _arSupport;


	public AudioBuffer doAudioEffects(List<Effect> effects, AudioBuffer inputBuffer) {
		PIAudioRenderContextImpl impl = new PIAudioRenderContextImpl(enabledAudioEffectsOnly(effects), inputBuffer);
		PIAudioRenderContextProxy.push(impl);
		try {
			return (AudioBuffer) impl.doPreviousEffect();
		} finally {
			PIAudioRenderContextProxy.pop();
		}
	}

	private List<AudioEffect> enabledAudioEffectsOnly(List<Effect> effects) {
		List<AudioEffect> audioEffects = Util.newList();
		for (Effect e : effects) {
			if (e instanceof AudioEffect && e.isEnabled()) {
				audioEffects.add((AudioEffect) e);
			}
		}
		return audioEffects;
	}


	private class PIAudioRenderContextImpl implements PIAudioRenderContext {

		private List<AudioEffect> effects;

		private final AudioBuffer inputBuffer;


		private PIAudioRenderContextImpl(List<AudioEffect> effects, AudioBuffer inputBuffer) {
			this.effects = effects;
			this.inputBuffer = inputBuffer;
		}

		public PIAudioBuffer doPreviousEffect() {
			if (effects.isEmpty()) {
				return (PIAudioBuffer) inputBuffer;
			}

			final AudioEffect prevEffect = effects.get(effects.size() - 1);

			return executePrevious(new WrappedOperation<PIAudioBuffer>() {
				public PIAudioBuffer execute() {
					PIAudioBuffer buffer = prevEffect.doAudioEffect();
					return (buffer != null) ? buffer : doPreviousEffect();
				}
			});
		}

		private <T> T executePrevious(final WrappedOperation<T> wop) {
			List<AudioEffect> savedEffects = effects;
			try {
				effects = effects.subList(0, effects.size() - 1);
				return _arContext.saveAndExecute(wop);
			} finally {
				effects = savedEffects;
			}
		}

		public Time getTime() {
			return _arContext.getTime();
		}

		public void setTime(Time time) {
			_arContext.setTime(time);
		}

		public <V> V value(PIAnimatableValue<V> avalue) {
			@SuppressWarnings("unchecked")
			AnimatableValue<V> av = (AnimatableValue<V>) avalue;
			return av.value(_arContext);
		}

		public AudioMode getAudioMode() {
			return _arContext.getAudioMode();
		}

		public int getFrameCount() {
			return _arContext.getFrameCount();
		}

		public void setFrameCount(int count) {
			_arContext.setFrameCount(count);
		}

		public PIAudioBuffer createAudioBuffer() {
			return (PIAudioBuffer) _arSupport.createAudioBuffer();
		}

	}

}
