/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.util.LinkedList;

import javax.media.opengl.GL;
import javax.media.opengl.glu.GLU;

import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.RenderResolution;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.plugin.PIAnimatableValue;
import ch.kuramo.javie.api.plugin.PIVideoBuffer;
import ch.kuramo.javie.api.plugin.PIVideoRenderContext;
import ch.kuramo.javie.core.Util;

public class PIVideoRenderContextProxy implements PIVideoRenderContext {

	private static final ThreadLocal<LinkedList<PIVideoRenderContext>>
			_threadLocal = new ThreadLocal<LinkedList<PIVideoRenderContext>>();

	static void push(PIVideoRenderContext impl) {
		getStack(true).addLast(impl);
	}

	static void pop() {
		LinkedList<PIVideoRenderContext> stack = getStack(false);

		stack.removeLast();

		if (stack.isEmpty()) {
			_threadLocal.remove();
		}
	}

	private static LinkedList<PIVideoRenderContext> getStack(boolean create) {
		LinkedList<PIVideoRenderContext> stack = _threadLocal.get();
		if (stack == null) {
			if (create) {
				stack = Util.newLinkedList();
				_threadLocal.set(stack);
			} else {
				throw new IllegalStateException("no stack is found");
			}
		} else if (stack.isEmpty()) {
			// popメソッドにおいて、スタックが空になったときにはスレッドローカルからスタックを取り除いているので、
			// ここで空のスタックが返されることは無い。
			throw new IllegalStateException("stack is empty");
		}
		return stack;
	}

	private PIVideoRenderContext getImpl() {
		return getStack(false).getLast();
	}

	public VideoBounds getPreviousBounds() {
		return getImpl().getPreviousBounds();
	}

	public PIVideoBuffer doPreviousEffect() {
		return getImpl().doPreviousEffect();
	}

	public Time getTime() {
		return getImpl().getTime();
	}

	public void setTime(Time time) {
		getImpl().setTime(time);
	}

	public <V> V value(PIAnimatableValue<V> avalue) {
		return getImpl().value(avalue);
	}

	public GL getGL() {
		return getImpl().getGL();
	}

	public GLU getGLU() {
		return getImpl().getGLU();
	}

	public RenderResolution getRenderResolution() {
		return getImpl().getRenderResolution();
	}

	public ColorMode getColorMode() {
		return getImpl().getColorMode();
	}

	public PIVideoBuffer createVideoBuffer(VideoBounds bounds) {
		return getImpl().createVideoBuffer(bounds);
	}

}
