/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import net.arnx.jsonic.JSON;
import net.arnx.jsonic.JSONException;
import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.core.AbstractAnimatableEnum;
import ch.kuramo.javie.core.AbstractAnimatableValue;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectDescriptor;
import ch.kuramo.javie.core.Interpolation;
import ch.kuramo.javie.core.Keyframe;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.annotations.ProjectElement;
import ch.kuramo.javie.core.services.EffectRegistry;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectElementFactory;

import com.google.inject.Inject;
import com.google.inject.Injector;

public class ProjectDecoderImpl implements ProjectDecoder {

	@Inject
	private ProjectElementFactory _elementFactory;

	@Inject
	private EffectRegistry _effectRegistry;

	@Inject
	private Injector _injector;


	public Project decode(CharSequence s, File baseDir) throws IOException, ProjectDecodeException {
		try {
			DecoderJSON json = new DecoderJSON(baseDir);
			Project p = json.parse(s, Project.class);
			p.afterDecode();
			return p;
		} catch (JSONException e) {
			if (e.getCause() instanceof ProjectDecodeException) {
				throw (ProjectDecodeException) e.getCause();
			} else {
				throw e;
			}
		}
	}

	public <T> T decodeElement(CharSequence s, Class<T> clazz) throws ProjectDecodeException {
		try {
			DecoderJSON json = new DecoderJSON();
			return json.parse(s, clazz);
		} catch (JSONException e) {
			if (e.getCause() instanceof ProjectDecodeException) {
				throw (ProjectDecodeException) e.getCause();
			} else {
				throw e;
			}
		}
	}

	private class DecoderJSON extends JSON {

		private final URI _baseURI;

		private DecoderJSON(File baseDir) throws IOException {
			super();
			_baseURI = baseDir.getCanonicalFile().toURI();
		}

		private DecoderJSON() {
			super();
			_baseURI = null;
		}

		@Override
		protected <T> T create(Context context, Class<? extends T> c) throws Exception {
			if (c.getAnnotation(ProjectElement.class) != null || Effect.class.isAssignableFrom(c)) {
				return _injector.getInstance(c);
			} else {
				return super.create(context, c);
			}
		}

		@Override
		protected <T> T postparse(final Context context, final Object value, final Class<? extends T> c, final Type type) throws Exception {
			
			if (value instanceof Map<?, ?>) {
				Map<?, ?> map = (Map<?, ?>) value;

				if (Keyframe.class.isAssignableFrom(c) && type instanceof ParameterizedType) {
					return c.cast(createKeyframe(context, map, (ParameterizedType) type));

				} else if ((c.getSuperclass() == AbstractAnimatableValue.class || c.getSuperclass() == AbstractAnimatableEnum.class)
						&& c.getGenericSuperclass() instanceof ParameterizedType) {

					return c.cast(createAnimatableValue(context, map, c,
							(Class<?>) ((ParameterizedType) c.getGenericSuperclass()).getActualTypeArguments()[0]));

				} else {
					String elementType = (String) map.get("TYPE");
					if (elementType != null) {
						Class<T> clazz = _elementFactory.getClass(elementType);
						if (clazz == null) {
							EffectDescriptor ed = _effectRegistry.getEffectDescriptor(elementType);
							if (ed != null) {
								@SuppressWarnings("unchecked")
								Class<T> effectClass = (Class<T>) ed.getEffectClass();
								clazz = effectClass;
							}
						}
						if (clazz == null) {
							throw new ProjectDecodeException("no such project element found: type=" + elementType);
						}
						return super.postparse(context, value, clazz, clazz);
					}
				}

			} else if (value instanceof String && File.class.isAssignableFrom(c)) {
				return c.cast(createAbsoluteFile((String) value));

			} else if (value instanceof List<?>) {
				List<?> list = (List<?>) value;

				if (Color.class.isAssignableFrom(c)) {
					return c.cast(createColor(context, list));

				} else if (Size2i.class.isAssignableFrom(c)) {
					return c.cast(createSize2i(context, list));

				} else if (Vec2d.class.isAssignableFrom(c)) {
					return c.cast(createVec2d(context, list));

				} else if (Vec3d.class.isAssignableFrom(c)) {
					return c.cast(createVec3d(context, list));

				} else if (Time.class.isAssignableFrom(c)) {
					return c.cast(createTime(context, list));

				}

			} else if (value instanceof String && c.isEnum()) {
				@SuppressWarnings("unchecked")
				Enum<?> e = Enum.valueOf((Class<? extends Enum>) c, (String) value);
				return c.cast(e);
			}

			return super.postparse(context, value, c, type);
		}

		private Keyframe<?> createKeyframe(Context context, Map<?, ?> map, ParameterizedType ptype) throws Exception {
			Time time = context.convert("time", map.get("time"), Time.class);
			Object value = context.convert("value", map.get("value"), ptype.getActualTypeArguments()[0]);
			Interpolation interpolation = context.convert("interpolation", map.get("interpolation"), Interpolation.class);

			@SuppressWarnings("unchecked")
			Keyframe<?> keyframe = new Keyframe(time, value, interpolation);
			return keyframe;
		}

		private Object createAnimatableValue(
				Context context, Map<?, ?> map, Class<?> clazz, Class<?> actualTypeArg) throws Exception {

			Constructor<?> cstr = clazz.getConstructor(actualTypeArg, Collection.class, String.class);

			Object staticValue = context.convert("staticValue", map.get("staticValue"), actualTypeArg);

			@SuppressWarnings("unchecked")
			Collection<Keyframe<?>> keyframes = (Collection) context.convert(
					"keyframes", map.get("keyframes"), cstr.getGenericParameterTypes()[1]);

			String expression = context.convert("expression", map.get("expression"), String.class);

			return cstr.newInstance(staticValue, keyframes, expression);
		}

		private File createAbsoluteFile(String uriAsString) throws ProjectDecodeException {
			try {
				URI uri = new URI(uriAsString);
				if (_baseURI != null) {
					uri = _baseURI.resolve(uri);
				}
				return new File(uri);
			} catch (URISyntaxException e) {
				throw new ProjectDecodeException(e);
			}
		}

		private Color createColor(Context context, List<?> list) throws Exception {
			return new Color(
					context.convert(0, list.get(0), double.class),
					context.convert(1, list.get(1), double.class),
					context.convert(2, list.get(2), double.class),
					context.convert(3, list.get(3), double.class));
		}

		private Size2i createSize2i(Context context, List<?> list) throws Exception {
			return new Size2i(
					context.convert(0, list.get(0), int.class),
					context.convert(1, list.get(1), int.class));			
		}

		private Vec2d createVec2d(Context context, List<?> list) throws Exception {
			return new Vec2d(
					context.convert(0, list.get(0), double.class),
					context.convert(1, list.get(1), double.class));
		}

		private Vec3d createVec3d(Context context, List<?> list) throws Exception {
			return new Vec3d(
					context.convert(0, list.get(0), double.class),
					context.convert(1, list.get(1), double.class),
					context.convert(2, list.get(2), double.class));
		}

		private Time createTime(Context context, List<?> list) throws Exception {
			return new Time(
					context.convert(0, list.get(0), long.class),
					context.convert(1, list.get(1), int.class));
		}

	}

}
