/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.media.opengl.GL;
import javax.media.opengl.GLContext;
import javax.media.opengl.Threading;
import javax.media.opengl.glu.GLU;

import org.scannotation.AnnotationDB;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.annotations.GLProgram;
import ch.kuramo.javie.api.annotations.GLShader;
import ch.kuramo.javie.api.plugin.PIShaderProgram;
import ch.kuramo.javie.api.plugin.PIShaderRegistry;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.GLGlobal;
import ch.kuramo.javie.core.services.ShaderBuildException;
import ch.kuramo.javie.core.services.ShaderProgram;
import ch.kuramo.javie.core.services.ShaderRegistry;

import com.google.inject.Inject;

public class ShaderRegistryImpl implements ShaderRegistry, PIShaderRegistry {

	private static final Logger _logger = LoggerFactory.getLogger(ShaderRegistryImpl.class);


	private final Map<String, Integer> _shaders = Util.newMap();

	private final Map<String, ShaderProgramImpl> _programs = Util.newMap();

	@Inject
	private GLGlobal _glGlobal;


	private int getShader(String name) {
		Integer shader = _shaders.get(name);
		return (shader != null) ? shader : 0;
	}

	private int getShader(Class<?> clazz, String name) {
		return getShader(fqfn(clazz, name));
	}

	private ShaderProgramImpl getProgram(String name) {
		return _programs.get(name);
	}

	public ShaderProgramImpl getProgram(Class<?> clazz, String name) {
		return getProgram(fqfn(clazz, name));
	}

	private void registerShader(Field field, int shader) throws ShaderBuildException {
		String name = fqfn(field);
		if (_shaders.containsKey(name)) {
			throw new ShaderBuildException(field, "Same name shader already exists.");
		}
		_shaders.put(name, shader);
	}

	private void registerProgram(Field field, int program, Map<String, Uniform> uniforms) throws ShaderBuildException {
		String name = fqfn(field);
		if (_programs.containsKey(name)) {
			throw new ShaderBuildException(field, "Same name program already exists.");
		}
		_programs.put(name, new ShaderProgramImpl(name, program, uniforms));
	}

	private String fqfn(Class<?> clazz, String name) {
		return clazz.getName() + "." + name;
	}

	private String fqfn(Field field) {
		return fqfn(field.getDeclaringClass(), field.getName());
	}

	private String[] getSource(Field field) throws ShaderBuildException {
		Object fieldValue;
		try {
			fieldValue = field.get(null);
		} catch (IllegalAccessException e) {
			throw new ShaderBuildException(field, e);
		}

		String[] source;

		Class<?> fieldType = field.getType();

		if (fieldType == String[].class) {
			source = (String[]) fieldValue;

		} else if (fieldType == String.class) {

			InputStream in = field.getDeclaringClass().getResourceAsStream((String) fieldValue);
			if (in == null) {
				throw new ShaderBuildException(field, "no such resource found: " + fieldValue);
			}

			try {
				List<String> lines = new ArrayList<String>();
				BufferedReader br = new BufferedReader(new InputStreamReader(in, "UTF-8"));
				try {
					String line;
					while ((line = br.readLine()) != null) {
						lines.add(line);
					}
				} finally {
					br.close();
				}

				source = lines.toArray(new String[lines.size()]);

			} catch (IOException e) {
				throw new ShaderBuildException(field, e);
			}

		} else {
			throw new ShaderBuildException(field,
					"Only String or String[] is allowed to the field type of GLShader.");
		}

		if (source == null || source.length == 0) {
			throw new ShaderBuildException(field, "Source is null or empty.");
		}

		return source;
	}

	private int compileShader(GL gl, int type, String[] source, Field field) throws ShaderBuildException {
		int shader = gl.glCreateShader(type);
		if (shader == 0) {
			throw new ShaderBuildException(field, "glCreateShader failed.");
		}

		gl.glShaderSource(shader, source.length, source, null, 0);
		gl.glCompileShader(shader);

		int[] status = new int[1];
		gl.glGetShaderiv(shader, GL.GL_COMPILE_STATUS, status, 0);
		if (status[0] == GL.GL_TRUE) {
			return shader;
		}

		int[] length = new int[1];
		gl.glGetShaderiv(shader, GL.GL_INFO_LOG_LENGTH, length, 0);

		byte[] infoLogBuf = new byte[length[0]];
		gl.glGetShaderInfoLog(shader, length[0], length, 0, infoLogBuf, 0);

		String infoLog;
		try {
			infoLog = new String(infoLogBuf, 0, length[0], "ISO-8859-1");
		} catch (UnsupportedEncodingException e) {
			infoLog = new String(infoLogBuf, 0, length[0]);
		}

		// 実行環境によってコンパイルに失敗するシェーダがあるかもしれないので、ログに書き出し実行を継続する。
		// 例外を投げてしまうと以降のシェーダがビルドされない。
		_logger.error(String.format("[%s] compile failed: %s", fqfn(field), infoLog));

		gl.glDeleteShader(shader);
		return 0;
	}

	private int linkProgram(GL gl, int shader, String[] otherShaderNames, Field field) throws ShaderBuildException {
		int program = gl.glCreateProgram();
		if (program == 0) {
			throw new ShaderBuildException(field, "glCreateProgram failed.");
		}

		gl.glAttachShader(program, shader);

		for (String otherShaderName : otherShaderNames) {
			int otherShader;
			if (otherShaderName.indexOf('.') == -1) {
				otherShader = getShader(field.getDeclaringClass(), otherShaderName);
			} else {
				otherShader = getShader(otherShaderName);
			}
			if (otherShader == 0) {
				// 次のふたつの場合が考えられる。
				//   * GLProgramアノテーションのattachの値が間違っている。
				//   * あるシェーダのコンパイルが失敗していて、それをアタッチしようとしている。
				// 前者はシェーダ開発時だけの問題だが、後者は環境依存でも起きるかもしれないので、
				// 例外は投げずに以降のシェーダのビルドを継続する。
				_logger.error(String.format("[%s] no such shader found: %s", fqfn(field), otherShaderName));

				gl.glDeleteProgram(program);
				return 0;
			}
			gl.glAttachShader(program, otherShader);
		}

		gl.glLinkProgram(program);

		int[] status = new int[1];
		gl.glGetProgramiv(program, GL.GL_LINK_STATUS, status, 0);
		if (status[0] == GL.GL_TRUE) {
			return program;
		}

		int[] length = new int[1];
		gl.glGetProgramiv(program, GL.GL_INFO_LOG_LENGTH, length, 0);

		byte[] infoLogBuf = new byte[length[0]];
		gl.glGetProgramInfoLog(program, length[0], length, 0, infoLogBuf, 0);

		String infoLog;
		try {
			infoLog = new String(infoLogBuf, 0, length[0], "ISO-8859-1");
		} catch (UnsupportedEncodingException e) {
			infoLog = new String(infoLogBuf, 0, length[0]);
		}

		// 実行環境によってリンクに失敗するシェーダがあるかもしれないので、ログに書き出し実行を継続する。
		// 例外を投げてしまうと以降のシェーダがビルドされない。
		_logger.error(String.format("[%s] link failed: %s", fqfn(field), infoLog));

		gl.glDeleteProgram(program);
		return 0;
	}

	private Map<String, Uniform> getUniform(GL gl, int program, Field field) {
		Map<String, Uniform> map = new HashMap<String, Uniform>();

		int[] int3 = new int[3];

		gl.glGetProgramiv(program, GL.GL_ACTIVE_UNIFORMS, int3, 0);
		int activeUniforms = int3[0];

		gl.glGetProgramiv(program, GL.GL_ACTIVE_UNIFORM_MAX_LENGTH, int3, 0);
		int bufSize = int3[0];
		byte[] nameBuf = new byte[bufSize];

		for (int i = 0; i < activeUniforms; ++i) {
			// int3[0] : length
			// int3[1] : size
			// int3[2] : type
			gl.glGetActiveUniform(program, i, bufSize, int3, 0, int3, 1, int3, 2, nameBuf, 0);

			int error = gl.glGetError();
			if (error != GL.GL_NO_ERROR) {
				_logger.error(String.format("[%s] glGetActiveUniform failed: %s", fqfn(field), new GLU().gluErrorString(error)));
			}

			String name;
			try {
				name = new String(nameBuf, 0, int3[0], "ISO-8859-1");
			} catch (UnsupportedEncodingException e) {
				name = new String(nameBuf, 0, int3[0]);
			}

			int location = gl.glGetUniformLocation(program, name);

			map.put(name, new Uniform(location, int3[1], int3[2]));
		}

		return map;
	}

	private void buildShader(final Field field, final GLShader shaderAnno, final GLProgram programAnno) throws ShaderBuildException {
		final int type;
		switch (shaderAnno.value()) {
			case VERTEX_SHADER:
				type = GL.GL_VERTEX_SHADER;
				break;
			case FRAGMENT_SHADER:
				type = GL.GL_FRAGMENT_SHADER;
				break;
			default:
				throw new IllegalArgumentException("unsupported ShaderType: " + shaderAnno.value());
		}

		final String[] source = getSource(field);

		final ShaderBuildException[] exception = new ShaderBuildException[1];

		Threading.invokeOnOpenGLThread(new Runnable() {
			public void run() {
				GLContext context = _glGlobal.getSharedContext();
				context.makeCurrent();
				try {
					GL gl = context.getGL();

					int shader = compileShader(gl, type, source, field);
					if (shader == 0) {
						return;
					}

					if (programAnno != null) {
						int program = linkProgram(gl, shader, programAnno.attach(), field);
						if (program == 0) {
							return;
						}

						Map<String, Uniform> uniforms = getUniform(gl, program, field);

						registerProgram(field, program, uniforms);

					} else {
						registerShader(field, shader);
					}
				} catch (ShaderBuildException e) {
					exception[0] = e;
				} finally {
					context.release();
				}
			}
		});

		if (exception[0] != null) {
			throw exception[0];
		}
	}

	public void buildShaders(AnnotationDB db, ClassLoader cl) throws ShaderBuildException {
		Set<String> classNames = new HashSet<String>();
		Util.addAll(db.getAnnotationIndex().get(GLShader.class.getName()), classNames);
		Util.addAll(db.getAnnotationIndex().get(GLProgram.class.getName()), classNames);	// GLShader無しにGLProgramを使っていないかを検出するため。

		for (String className : classNames) {
			Class<?> clazz;
			try {
				clazz = Class.forName(className, true, cl);
			} catch (ClassNotFoundException e) {
				// 引数の AnnotationDB と ClassLoader の対応が正しければ必ず見つかるはず。
				throw new IllegalArgumentException(e);
			}

			for (Field field : clazz.getFields()) {
				GLShader shaderAnno = null;
				GLProgram programAnno = null;
				for (Annotation fieldAnno : field.getAnnotations()) {
					if (fieldAnno instanceof GLShader) {
						shaderAnno = (GLShader) fieldAnno;
					} else if (fieldAnno instanceof GLProgram) {
						programAnno = (GLProgram) fieldAnno;
					}
				}

				if (shaderAnno == null) {
					if (programAnno != null) {
						throw new ShaderBuildException(field, "GLProgram without GLShader");
					}
					continue;
				}

				buildShader(field, shaderAnno, programAnno);
			}
		}
	}

}

class Uniform {
	final int location;
//	final int size;
//	final int type;

	Uniform(int location, int size, int type) {
		this.location = location;
//		this.size = size;
//		this.type = type;
	}
}

class ShaderProgramImpl implements ShaderProgram, PIShaderProgram {
	final String name;
	final int program;
	final Map<String, Uniform> uniforms;

	ShaderProgramImpl(String name, int program, Map<String, Uniform> uniforms) {
		this.name = name;
		this.program = program;
		this.uniforms = Collections.unmodifiableMap(uniforms);
	}

	public String getName() {
		return name;
	}

	public int getProgram() {
		return program;
	}

	public int getUniformLocation(String uniformName) {
		Uniform uniform = uniforms.get(uniformName);
		if (uniform == null && uniformName.endsWith("[0]")) {
			// nVidiaの場合 glGetActiveUniform が返すuniform名に[0]が付いていない。
			uniform = uniforms.get(uniformName.substring(0, uniformName.length()-3));
		}
		return (uniform != null) ? uniform.location : -1;
	}

//	public int getUniformSize(String uniformName) {
//		Uniform uniform = uniforms.get(uniformName);
//		return (uniform != null) ? uniform.size : -1;
//	}
//
//	public int getUniformType(String uniformName) {
//		Uniform uniform = uniforms.get(uniformName);
//		return (uniform != null) ? uniform.type : -1;
//	}
}
