/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.util.List;

import javax.media.opengl.GL;
import javax.media.opengl.glu.GLU;

import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.RenderResolution;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.plugin.PIAnimatableValue;
import ch.kuramo.javie.api.plugin.PIVideoBuffer;
import ch.kuramo.javie.api.plugin.PIVideoRenderContext;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.VideoBuffer;
import ch.kuramo.javie.core.VideoEffect;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.services.VideoEffectPipeline;
import ch.kuramo.javie.core.services.VideoRenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;

import com.google.inject.Inject;

public class VideoEffectPipelineImpl implements VideoEffectPipeline {

	@Inject
	private VideoRenderContext _vrContext;

	@Inject
	private VideoRenderSupport _vrSupport;


	public VideoBounds getVideoBounds(List<Effect> effects, VideoBounds inputBounds) {
		PIVideoRenderContextImpl impl = new PIVideoRenderContextImpl(enabledVideoEffectsOnly(effects), inputBounds);
		PIVideoRenderContextProxy.push(impl);
		try {
			return impl.getPreviousBounds();
		} finally {
			PIVideoRenderContextProxy.pop();
		}
	}

	public VideoBuffer doVideoEffects(List<Effect> effects, VideoBuffer inputBuffer) {
		PIVideoRenderContextImpl impl = new PIVideoRenderContextImpl(enabledVideoEffectsOnly(effects), inputBuffer);
		PIVideoRenderContextProxy.push(impl);
		try {
			return (VideoBuffer) impl.doPreviousEffect();
		} finally {
			PIVideoRenderContextProxy.pop();
		}
	}

	private List<VideoEffect> enabledVideoEffectsOnly(List<Effect> effects) {
		List<VideoEffect> videoEffects = Util.newList();
		for (Effect e : effects) {
			if (e instanceof VideoEffect && e.isEnabled()) {
				videoEffects.add((VideoEffect) e);
			}
		}
		return videoEffects;
	}


	private class PIVideoRenderContextImpl implements PIVideoRenderContext {

		private List<VideoEffect> effects;

		private final VideoBounds inputBounds;

		private final VideoBuffer inputBuffer;


		private PIVideoRenderContextImpl(List<VideoEffect> effects, VideoBounds inputBounds) {
			this.effects = effects;
			this.inputBounds = inputBounds;
			this.inputBuffer = null;
		}

		private PIVideoRenderContextImpl(List<VideoEffect> effects, VideoBuffer inputBuffer) {
			this.effects = effects;
			this.inputBounds = null;
			this.inputBuffer = inputBuffer;
		}

		public VideoBounds getPreviousBounds() {
			if (inputBounds == null) {
				throw new IllegalStateException();
			}

			if (effects.isEmpty()) {
				return inputBounds;
			}

			final VideoEffect prevEffect = effects.get(effects.size() - 1);

			return executePrevious(new WrappedOperation<VideoBounds>() {
				public VideoBounds execute() {
					VideoBounds bounds = prevEffect.getVideoBounds();
					return (bounds != null) ? bounds : getPreviousBounds();
				}
			});
		}
		
		public PIVideoBuffer doPreviousEffect() {
			if (inputBuffer == null) {
				throw new IllegalStateException();
			}

			if (effects.isEmpty()) {
				return (PIVideoBuffer) inputBuffer;
			}

			final VideoEffect prevEffect = effects.get(effects.size() - 1);

			return executePrevious(new WrappedOperation<PIVideoBuffer>() {
				public PIVideoBuffer execute() {
					PIVideoBuffer buffer = prevEffect.doVideoEffect();
					return (buffer != null) ? buffer : doPreviousEffect();
				}
			});
		}

		private <T> T executePrevious(final WrappedOperation<T> wop) {
			List<VideoEffect> savedEffects = effects;
			try {

				effects = effects.subList(0, effects.size() - 1);

				return _vrContext.saveAndExecute(new WrappedOperation<T>() {
					public T execute() {
						return _vrSupport.pushMatrixAndExecute(wop);
					}
				});

			} finally {
				effects = savedEffects;
			}
		}

		public Time getTime() {
			return _vrContext.getTime();
		}

		public void setTime(Time time) {
			_vrContext.setTime(time);
		}

		public <V> V value(PIAnimatableValue<V> avalue) {
			@SuppressWarnings("unchecked")
			AnimatableValue<V> av = (AnimatableValue<V>) avalue;
			return av.value(_vrContext);
		}

		public GL getGL() {
			return _vrContext.getGL();
		}

		public GLU getGLU() {
			return _vrContext.getGLU();
		}

		public RenderResolution getRenderResolution() {
			return _vrContext.getRenderResolution();
		}

		public ColorMode getColorMode() {
			return _vrContext.getColorMode();
		}

		public PIVideoBuffer createVideoBuffer(VideoBounds bounds) {
			return (PIVideoBuffer) _vrSupport.createVideoBuffer(_vrContext.getColorMode(), bounds);
		}

	}

}
