/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.nio.FloatBuffer;
import java.nio.ShortBuffer;
import java.util.Arrays;
import java.util.Collection;

import javax.media.opengl.GL;
import javax.media.opengl.glu.GLU;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.core.BlendMode;
import ch.kuramo.javie.core.DepthBuffer;
import ch.kuramo.javie.core.VideoBuffer;
import ch.kuramo.javie.core.VideoLayerBuffer;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.internal.DepthBufferImpl;
import ch.kuramo.javie.core.internal.VideoBufferImpl;
import ch.kuramo.javie.core.services.ArrayPools;
import ch.kuramo.javie.core.services.GLGlobal;
import ch.kuramo.javie.core.services.ShaderProgram;
import ch.kuramo.javie.core.services.ShaderRegistry;
import ch.kuramo.javie.core.services.VideoRenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;
import ch.kuramo.javie.core.shaders.BlendModeShaders;

import com.google.inject.Inject;
import com.google.inject.Injector;

import ftgl.FTGL;
import ftgl.FTGLfont;

public class VideoRenderSupportImpl implements VideoRenderSupport {

	private static final Logger _logger = LoggerFactory.getLogger(VideoRenderSupportImpl.class);

	private static final float[] FLOAT0000 = new float[] { 0, 0, 0, 0 };


	@Inject
	private Injector _injector;

	@Inject
	private GLGlobal _glGlobal;

	@Inject
	private VideoRenderContext _vrContext;

	@Inject
	private ShaderRegistry _shaders;

	@Inject
	private ArrayPools _arrayPools;


	public VideoBuffer createVideoBuffer(ColorMode colorMode, Size2i size) {
		VideoBufferImpl vb = new VideoBufferImpl(colorMode, size);
		_injector.injectMembers(vb);
		return vb;
	}

	public VideoBuffer createVideoBuffer(ColorMode colorMode, VideoBounds bounds) {
		VideoBufferImpl vb = new VideoBufferImpl(colorMode, bounds);
		_injector.injectMembers(vb);
		return vb;
	}

	public DepthBuffer createDepthBuffer(Size2i size) {
		DepthBufferImpl db = new DepthBufferImpl(size);
		_injector.injectMembers(db);
		return db;
	}

	public int createTexture(ColorMode colorMode, Size2i textureSize) {
		GL gl = _vrContext.getGL();

		int[] tex = new int[1];
		gl.glGenTextures(1, tex, 0);
		gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, tex[0]);

		gl.glTexParameteri(GL.GL_TEXTURE_RECTANGLE_EXT, GL.GL_TEXTURE_MIN_FILTER, GL.GL_LINEAR);
		gl.glTexParameteri(GL.GL_TEXTURE_RECTANGLE_EXT, GL.GL_TEXTURE_MAG_FILTER, GL.GL_LINEAR);
		gl.glTexParameteri(GL.GL_TEXTURE_RECTANGLE_EXT, GL.GL_TEXTURE_WRAP_S, GL.GL_CLAMP_TO_BORDER);
		gl.glTexParameteri(GL.GL_TEXTURE_RECTANGLE_EXT, GL.GL_TEXTURE_WRAP_T, GL.GL_CLAMP_TO_BORDER);
		gl.glTexParameterfv(GL.GL_TEXTURE_RECTANGLE_EXT, GL.GL_TEXTURE_BORDER_COLOR, FLOAT0000, 0);
		gl.glTexImage2D(GL.GL_TEXTURE_RECTANGLE_EXT, 0, colorMode.glInternalFormat,
				textureSize.width, textureSize.height, 0, GL.GL_BGRA, colorMode.glDataType, null);

		gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, 0);

		return tex[0];
	}

	public int createDepthTexture(Size2i textureSize) {
		GL gl = _vrContext.getGL();

		int[] tex = new int[1];
		gl.glGenTextures(1, tex, 0);
		gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, tex[0]);

		gl.glTexParameteri(GL.GL_TEXTURE_RECTANGLE_EXT, GL.GL_TEXTURE_MIN_FILTER, GL.GL_NEAREST);
		gl.glTexParameteri(GL.GL_TEXTURE_RECTANGLE_EXT, GL.GL_TEXTURE_MAG_FILTER, GL.GL_NEAREST);
		gl.glTexParameteri(GL.GL_TEXTURE_RECTANGLE_EXT, GL.GL_TEXTURE_WRAP_S, GL.GL_CLAMP_TO_BORDER);
		gl.glTexParameteri(GL.GL_TEXTURE_RECTANGLE_EXT, GL.GL_TEXTURE_WRAP_T, GL.GL_CLAMP_TO_BORDER);
		gl.glTexParameterfv(GL.GL_TEXTURE_RECTANGLE_EXT, GL.GL_TEXTURE_BORDER_COLOR, FLOAT0000, 0);
		gl.glTexImage2D(GL.GL_TEXTURE_RECTANGLE_EXT, 0, GL.GL_DEPTH_COMPONENT32,
				textureSize.width, textureSize.height, 0, GL.GL_DEPTH_COMPONENT, GL.GL_FLOAT, null);

		gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, 0);

		return tex[0];
	}

	public void deleteTexture(int texture) {
		GL gl = _vrContext.getGL();
		gl.glDeleteTextures(1, new int[] { texture }, 0);
	}

	public Object createArray(ColorMode colorMode, Size2i imageSize) {
		int arraySize = imageSize.width * imageSize.height * 4;

		switch (colorMode) {
			case RGBA8:
				return _arrayPools.getByteArray(arraySize);
			case RGBA16:
				return _arrayPools.getShortArray(arraySize);
			case RGBA16_FLOAT:
			case RGBA32_FLOAT:
				return _arrayPools.getFloatArray(arraySize);
		}

		// Never reaches here!
		throw new UnsupportedOperationException("ColorMode: " + colorMode.name());
	}

	public void releaseArray(Object array) {
		_arrayPools.put(array);
	}

	public void copyArrayToTexture(Object array, int texture, ColorMode colorMode, Size2i imageSize) {
		synchronized (_glGlobal.getLockObject()) {

			GL gl = _vrContext.getGL();

			int[] pbo = new int[1];
			gl.glGenBuffers(1, pbo, 0);
			gl.glBindBuffer(GL.GL_PIXEL_UNPACK_BUFFER, pbo[0]);
			gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, texture);

			int imageBytes = imageSize.width * imageSize.height * colorMode.javaPixelBytes;

			// 配列 -> PBO
			gl.glBufferData(GL.GL_PIXEL_UNPACK_BUFFER, imageBytes, toNioBuffer(array), GL.GL_STREAM_DRAW);

			// PBO -> テクスチャ
			gl.glTexSubImage2D(GL.GL_TEXTURE_RECTANGLE_EXT, 0, 0, 0, imageSize.width, imageSize.height,
					GL.GL_BGRA, colorMode.glDataType, 0);

			gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, 0);
			gl.glBindBuffer(GL.GL_PIXEL_UNPACK_BUFFER, 0);
			gl.glDeleteBuffers(1, pbo, 0);

		}
	}

	private Buffer toNioBuffer(Object array) {
		if (array instanceof byte[]) {
			return ByteBuffer.wrap((byte[]) array);
			
		} else if (array instanceof short[]) {
			return ShortBuffer.wrap((short[]) array);

		} else if (array instanceof float[]) {
			return FloatBuffer.wrap((float[]) array);

		} else {
			throw new IllegalArgumentException(
					"not array or unsupported array type: "
					+ array.getClass().getName());
		}
	}

	public void copyTextureToArray(int texture, Object array, ColorMode colorMode, Size2i imageSize) {
		synchronized (_glGlobal.getLockObject()) {

			GL gl = _vrContext.getGL();

			int[] pbo = new int[1];
			gl.glGenBuffers(1, pbo, 0);
			gl.glBindBuffer(GL.GL_PIXEL_PACK_BUFFER, pbo[0]);
			gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, texture);

			int imageBytes = imageSize.width * imageSize.height * colorMode.javaPixelBytes;

			// PBOの領域を確保
			gl.glBufferData(GL.GL_PIXEL_PACK_BUFFER, imageBytes, null, GL.GL_STREAM_READ);

			// テクスチャをFBOに割当てる
			gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
					GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, texture, 0);

			// テクスチャ -> PBO
			gl.glReadBuffer(GL.GL_COLOR_ATTACHMENT0_EXT);
			gl.glReadPixels(0, 0, imageSize.width, imageSize.height, GL.GL_BGRA, colorMode.glDataType, 0);

			// PBO -> 配列
			ByteBuffer mappedPBO = gl.glMapBuffer(GL.GL_PIXEL_PACK_BUFFER, GL.GL_READ_ONLY);
			copyPBOToArray(mappedPBO, array);
			gl.glUnmapBuffer(GL.GL_PIXEL_PACK_BUFFER);

			gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
					GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, 0, 0);

			gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, 0);
			gl.glBindBuffer(GL.GL_PIXEL_PACK_BUFFER, 0);
			gl.glDeleteBuffers(1, pbo, 0);

		}
	}

	private void copyPBOToArray(ByteBuffer mappedPBO, Object array) {
		if (array instanceof byte[]) {
			mappedPBO.get((byte[]) array, 0, mappedPBO.capacity());
			
		} else if (array instanceof short[]) {
			mappedPBO.asShortBuffer().get((short[]) array, 0, mappedPBO.capacity() / 2);

		} else if (array instanceof float[]) {
			mappedPBO.asFloatBuffer().get((float[]) array, 0, mappedPBO.capacity() / 4);

		} else {
			throw new IllegalArgumentException(
					"not array or unsupported array type: "
					+ array.getClass().getName());
		}
	}

	public void clear(VideoBuffer videoBuffer) {
		if (videoBuffer.isTexture()) {
			fill(videoBuffer.getTexture(), Color.COLORLESS_TRANSPARENT);

		} else if (videoBuffer.isArray()) {
			Size2i size = videoBuffer.getImageSize();
			clear(videoBuffer.getArray(), size.width * size.height * 4);

		} else {
			throw new IllegalStateException("neither texture nor array exist.");
		}
	}

	private void clear(Object array, int length) {
		if (array instanceof byte[]) {
			Arrays.fill((byte[]) array, 0, length, (byte) 0);

		} else if (array instanceof short[]) {
			Arrays.fill((short[]) array, 0, length, (short) 0);

		} else if (array instanceof float[]) {
			Arrays.fill((float[]) array, 0, length, 0f);

		} else {
			throw new IllegalArgumentException(
					"not array or unsupported array type: "
					+ array.getClass().getName());
		}
	}

	public void fill(VideoBuffer videoBuffer, Color color) {
		// 配列として確保されている場合
		if (videoBuffer.isArray()) {
			fill(videoBuffer.getArray(), videoBuffer.getImageSize(), color);
			return;
		}

		if (!videoBuffer.isAllocated()) {
			videoBuffer.allocateAsTexture();
		}

		fill(videoBuffer.getTexture(), color);
	}

	private void fill(int texture, Color color) {
		GL gl = _vrContext.getGL();

		gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
				GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, texture, 0);
		gl.glDrawBuffer(GL.GL_COLOR_ATTACHMENT0_EXT);

		float a = (float)color.a;
		float r = (float)color.r * a; 
		float g = (float)color.g * a; 
		float b = (float)color.b * a; 
		gl.glClearColor(r, g, b, a);
		gl.glClear(GL.GL_COLOR_BUFFER_BIT);

		gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
				GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, 0, 0);
	}

	private void fill(Object array, Size2i imageSize, Color color) {
		// TODO
		throw new UnsupportedOperationException("not implemented");
	}

	public void fillRectCR(
			VideoBuffer resultBuffer, DepthBuffer depthBuffer,
			final double[] mvMatrix, final double[] prjMatrix, final double opacity,
			final Size2i size, final Color color) {

		final int resultTexture = resultBuffer.getTexture();
		final Size2i resultImageSize = resultBuffer.getImageSize();

		final int depthTexture = (depthBuffer != null) ? depthBuffer.getTexture() : 0;


		pushMatrixAndExecute(new WrappedOperation<Object>() {
			public Object execute() {
				GL gl = _vrContext.getGL();

				gl.glViewport(0, 0, resultImageSize.width, resultImageSize.height);
				gl.glMatrixMode(GL.GL_PROJECTION);
				gl.glLoadMatrixd(prjMatrix, 0);
				gl.glMatrixMode(GL.GL_MODELVIEW);
				gl.glLoadMatrixd(mvMatrix, 0);


				gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
						GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, resultTexture, 0);
				gl.glDrawBuffer(GL.GL_COLOR_ATTACHMENT0_EXT);

				if (depthTexture != 0) {
					gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
							GL.GL_DEPTH_ATTACHMENT_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, depthTexture, 0);
					gl.glEnable(GL.GL_DEPTH_TEST);
					gl.glClearDepth(1.0);
					gl.glClear(GL.GL_DEPTH_BUFFER_BIT);
				}

				gl.glBegin(GL.GL_QUADS);

				float a = (float)(color.a * opacity / 100.0);
				float r = (float)color.r * a;
				float g = (float)color.g * a;
				float b = (float)color.b * a;
				gl.glColor4f(r, g, b, a);

				gl.glVertex2i(0, 0);
				gl.glVertex2i(size.width, 0);
				gl.glVertex2i(size.width, size.height);
				gl.glVertex2i(0, size.height);

				gl.glEnd();

				if (depthTexture != 0) {
					gl.glDisable(GL.GL_DEPTH_TEST);
					gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
							GL.GL_DEPTH_ATTACHMENT_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, 0, 0);
				}

				gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
						GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, 0, 0);

				return null;
			}
		});
	}

	private void ortho2D(GL gl, GLU glu, Size2i imageSize) {
		ortho2D(gl, glu, imageSize.width, imageSize.height);
	}

	private void ortho2D(GL gl, GLU glu, int width, int height) {
		gl.glViewport(0, 0, width, height);

		gl.glMatrixMode(GL.GL_PROJECTION);
		gl.glLoadIdentity();
		glu.gluOrtho2D(0, width, 0, height);

		gl.glMatrixMode(GL.GL_MODELVIEW);
		gl.glLoadIdentity();
	}

	public void blend(VideoBuffer srcBuffer, VideoBuffer dstBuffer, VideoBuffer resultBuffer, final BlendMode blendMode) {

		final int srcTexture = srcBuffer.getTexture();
		final int dstTexture = dstBuffer.getTexture();
		final int resultTexture = resultBuffer.getTexture();
		final Size2i imageSize = resultBuffer.getImageSize();

		pushMatrixAndExecute(new WrappedOperation<Object>() {
			public Object execute() {
				GL gl = _vrContext.getGL();
				GLU glu = _vrContext.getGLU();

				ortho2D(gl, glu, imageSize);

				// レンダリング先を resultTexture に設定
				gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
						GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, resultTexture, 0);
				gl.glDrawBuffer(GL.GL_COLOR_ATTACHMENT0_EXT);

				// レンダリングのソースとなるテクスチャユニットの設定
				gl.glActiveTexture(GL.GL_TEXTURE0);
				gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, srcTexture);
				gl.glActiveTexture(GL.GL_TEXTURE1);
				gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, dstTexture);

				ShaderProgram program = _shaders.getProgram(BlendModeShaders.class, blendMode.name());
				synchronized (program) {
					gl.glUseProgram(program.getProgram());
					gl.glUniform1i(program.getUniformLocation("texSrc"), 0);
					gl.glUniform1i(program.getUniformLocation("texDst"), 1);

					gl.glBegin(GL.GL_QUADS);
					gl.glTexCoord2f(0, 0);
					gl.glVertex2f(0, 0);
					gl.glTexCoord2f(imageSize.width, 0);
					gl.glVertex2f(imageSize.width, 0);
					gl.glTexCoord2f(imageSize.width, imageSize.height);
					gl.glVertex2f(imageSize.width, imageSize.height);
					gl.glTexCoord2f(0, imageSize.height);
					gl.glVertex2f(0, imageSize.height);
					gl.glEnd();

					gl.glFlush();
					gl.glUseProgram(0);
				}

				gl.glActiveTexture(GL.GL_TEXTURE1);
				gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, 0);
				gl.glActiveTexture(GL.GL_TEXTURE0);
				gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, 0);

				gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
						GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, 0, 0);

				return null;
			}
		});
	}

	public void blend(final Collection<VideoLayerBuffer> srcBuffers, final VideoBuffer dstBuffer, VideoBuffer resultBuffer) {

		final int resultTexture = resultBuffer.getTexture();
		final Size2i imageSize = resultBuffer.getImageSize();

		pushMatrixAndExecute(new WrappedOperation<Object>() {
			public Object execute() {
				GL gl = _vrContext.getGL();
				GLU glu = _vrContext.getGLU();

				ortho2D(gl, glu, imageSize);

				// TODO テクスチャユニットの最大数を実行時に取得する。
				//		また、その場合はシェーダのソースも実行時に修正する必要がある。
				int nTexUnits = 16;
				int maxLayers = (nTexUnits - 1) / 2;

				int nLayers = srcBuffers.size();
				if (nLayers > maxLayers) {
					_logger.warn(String.format("number of layers in a inersection group exceeds limit: "
							+ "limit=%d, layers=%d", maxLayers, nLayers));
					nLayers = maxLayers;
				}

				int[] blendMode = new int[maxLayers];

				// レンダリング先を resultTexture に設定
				gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
						GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, resultTexture, 0);
				gl.glDrawBuffer(GL.GL_COLOR_ATTACHMENT0_EXT);

				// テクスチャユニットの設定
				gl.glActiveTexture(GL.GL_TEXTURE0);
				gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, dstBuffer.getTexture());

				int i = 0;
				for (VideoLayerBuffer vlb : srcBuffers) {
					gl.glActiveTexture(GL.GL_TEXTURE0 + i + 1);
					gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, vlb.getVideoBuffer().getTexture());
					gl.glActiveTexture(GL.GL_TEXTURE0 + i + 1 + maxLayers);
					gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, vlb.getDepthBuffer().getTexture());
					blendMode[i] = vlb.getBlendMode().ordinal();
					++i;
					if (i >= nLayers) break;
				}

				ShaderProgram program = _shaders.getProgram(BlendModeShaders.class, "INTERSECTING_BLEND");
				synchronized (program) {
					gl.glUseProgram(program.getProgram());
					gl.glUniform1i(program.getUniformLocation("nLayers"), nLayers);
					gl.glUniform1i(program.getUniformLocation("texDst"), 0);
					for (i = 0; i < maxLayers; ++i) {
						gl.glUniform1i(program.getUniformLocation("texSrc" + i), i + 1);
						gl.glUniform1i(program.getUniformLocation("texDep" + i), i + 1 + maxLayers);
					}
					gl.glUniform1iv(program.getUniformLocation("blendMode[0]"), maxLayers, blendMode, 0);

					gl.glBegin(GL.GL_QUADS);
					gl.glTexCoord2f(0, 0);
					gl.glVertex2f(0, 0);
					gl.glTexCoord2f(imageSize.width, 0);
					gl.glVertex2f(imageSize.width, 0);
					gl.glTexCoord2f(imageSize.width, imageSize.height);
					gl.glVertex2f(imageSize.width, imageSize.height);
					gl.glTexCoord2f(0, imageSize.height);
					gl.glVertex2f(0, imageSize.height);
					gl.glEnd();

					gl.glFlush();
					gl.glUseProgram(0);
				}

				for (int j = 0, n = maxLayers * 2 + 1; j < n; ++j) {
					gl.glActiveTexture(GL.GL_TEXTURE0 + j);
					gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, 0);
				}

				gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
						GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, 0, 0);

				return null;
			}
		});
	}

	public <T> T pushMatrixAndExecute(WrappedOperation<T> operation) {
		GL gl = _vrContext.getGL();

		int[] viewport = new int[4];
		gl.glGetIntegerv(GL.GL_VIEWPORT, viewport, 0);

		int[] matrixMode = new int[1];
		gl.glGetIntegerv(GL.GL_MATRIX_MODE, matrixMode, 0);

		gl.glMatrixMode(GL.GL_PROJECTION);
		gl.glPushMatrix();

		gl.glMatrixMode(GL.GL_MODELVIEW);
		gl.glPushMatrix();

		try {

			return operation.execute();

		} finally {
			gl.glMatrixMode(GL.GL_MODELVIEW);
			gl.glPopMatrix();
			
			gl.glMatrixMode(GL.GL_PROJECTION);
			gl.glPopMatrix();

			gl.glMatrixMode(matrixMode[0]);

			gl.glViewport(viewport[0], viewport[1], viewport[2], viewport[3]);
		}
	}

	public void multModelViewMatrix(
			Vec3d anchorPoint, Vec3d scale, Vec3d orientation, Vec3d rotate, Vec3d position) {

		GL gl = _vrContext.getGL();
		gl.glMatrixMode(GL.GL_MODELVIEW);

		// 位置
		gl.glTranslatef((float) position.x, (float) position.y, (float) position.z);

		// 方向
		gl.glRotatef((float) orientation.x, 1, 0, 0);
		gl.glRotatef((float) orientation.y, 0, 1, 0);
		gl.glRotatef((float) orientation.z, 0, 0, 1);

		// 回転
		gl.glRotatef((float) rotate.x, 1, 0, 0);
		gl.glRotatef((float) rotate.y, 0, 1, 0);
		gl.glRotatef((float) rotate.z, 0, 0, 1);

		// 拡大縮小
		gl.glScalef((float) (scale.x/100.0), (float) (scale.y/100.0), (float) (scale.z/100.0));

		// アンカーポイント
		gl.glTranslatef((float) -anchorPoint.x, (float) -anchorPoint.y, (float) -anchorPoint.z);
	}

	public void multModelViewMatrixInverse(
			Vec3d anchorPoint, Vec3d scale, Vec3d orientation, Vec3d rotate, Vec3d position) {

		GL gl = _vrContext.getGL();
		gl.glMatrixMode(GL.GL_MODELVIEW);

		// アンカーポイント
		gl.glTranslatef((float) anchorPoint.x, (float) anchorPoint.y, (float) anchorPoint.z);

		// 拡大縮小
		gl.glScalef((float) (100.0/scale.x), (float) (100.0/scale.y), (float) (100.0/scale.z));

		// 回転
		gl.glRotatef((float) -rotate.z, 0, 0, 1);
		gl.glRotatef((float) -rotate.y, 0, 1, 0);
		gl.glRotatef((float) -rotate.x, 1, 0, 0);

		// 方向
		gl.glRotatef((float) -orientation.z, 0, 0, 1);
		gl.glRotatef((float) -orientation.y, 0, 1, 0);
		gl.glRotatef((float) -orientation.x, 1, 0, 0);

		// 位置
		gl.glTranslatef((float) -position.x, (float) -position.y, (float) -position.z);
	}

	public void multCameraMatrix(
			Vec3d orientation, Vec3d rotate, Vec3d position, Vec3d pointOfInterest, boolean flipZ) {

		double dx = pointOfInterest.x - position.x;
		double dy = pointOfInterest.y - position.y;
		double dz = pointOfInterest.z - position.z;

		float _1 = flipZ ? -1 : 1;

		GL gl = _vrContext.getGL();
		gl.glMatrixMode(GL.GL_MODELVIEW);

		// 回転
		gl.glRotatef((float) -rotate.z,  0,  0,  1);
		gl.glRotatef((float) -rotate.y,  0, _1,  0);
		gl.glRotatef((float) -rotate.x, _1,  0,  0);

		// 方向
		gl.glRotatef((float) -orientation.z,  0,  0,  1);
		gl.glRotatef((float) -orientation.y,  0, _1,  0);
		gl.glRotatef((float) -orientation.x, _1,  0,  0);

		// 目標点
		gl.glRotatef((float) Math.toDegrees(Math.atan2(dy, Math.sqrt(dx*dx+dz*dz))), _1, 0, 0);
		gl.glRotatef((float) Math.toDegrees(-Math.atan2(dx, dz)), 0, _1, 0);

		// 位置
		gl.glTranslatef((float) -position.x, (float) -position.y, (float) -position.z * _1);

		if (flipZ) {
			gl.glScalef(1, 1, -1);
		}
	}

	public void multCameraMatrixInverse(
			final Vec3d orientation, final Vec3d rotate, final Vec3d position, final Vec3d pointOfInterest) {

		double dx = pointOfInterest.x - position.x;
		double dy = pointOfInterest.y - position.y;
		double dz = pointOfInterest.z - position.z;

		GL gl = _vrContext.getGL();
		gl.glMatrixMode(GL.GL_MODELVIEW);

		// 位置
		gl.glTranslatef((float) position.x, (float) position.y, (float) position.z);

		// 目標点
		gl.glRotatef((float) Math.toDegrees(Math.atan2(dx, dz)), 0, 1, 0);
		gl.glRotatef((float) Math.toDegrees(-Math.atan2(dy, Math.sqrt(dx*dx+dz*dz))), 1, 0, 0);

		// 方向
		gl.glRotatef((float) orientation.x, 1, 0, 0);
		gl.glRotatef((float) orientation.y, 0, 1, 0);
		gl.glRotatef((float) orientation.z, 0, 0, 1);

		// 回転
		gl.glRotatef((float) rotate.x, 1, 0, 0);
		gl.glRotatef((float) rotate.y, 0, 1, 0);
		gl.glRotatef((float) rotate.z, 0, 0, 1);
	}

	public void multModelViewMatrix(double[] mvMatrix) {
		GL gl = _vrContext.getGL();
		gl.glMatrixMode(GL.GL_MODELVIEW);
		gl.glMultMatrixd(mvMatrix, 0);
	}

	public void getMatrix(double[] mvMatrix, double[] prjMatrix) {
		GL gl = _vrContext.getGL();
		if (mvMatrix != null) {
			gl.glGetDoublev(GL.GL_MODELVIEW_MATRIX, mvMatrix, 0);
		}
		if (prjMatrix != null) {
			gl.glGetDoublev(GL.GL_PROJECTION_MATRIX, prjMatrix, 0);
		}
	}

	public void setMatrix(double[] mvMatrix, double[] prjMatrix) {
		GL gl = _vrContext.getGL();
		if (prjMatrix != null) {
			gl.glMatrixMode(GL.GL_PROJECTION);
			gl.glLoadMatrixd(prjMatrix, 0);
		}
		if (mvMatrix != null) {
			gl.glMatrixMode(GL.GL_MODELVIEW);
			gl.glLoadMatrixd(mvMatrix, 0);
		}
	}

	public void resetMatrix() {
		GL gl = _vrContext.getGL();
		gl.glMatrixMode(GL.GL_PROJECTION);
		gl.glLoadIdentity();
		gl.glMatrixMode(GL.GL_MODELVIEW);
		gl.glLoadIdentity();
	}

	public Vec3d project(Vec3d point, double[] mvMatrix, double[] prjMatrix, Size2i viewportSize) {
		int[] viewport = new int[] { 0, 0, viewportSize.width, viewportSize.height };
		double[] result = new double[3];
		_vrContext.getGLU().gluProject(
				point.x, point.y, point.z,
				mvMatrix, 0, prjMatrix, 0, viewport, 0,
				result, 0);
		return new Vec3d(result[0], result[1], result[2]);
	}

	public void transform(
			VideoBuffer srcBuffer, VideoBuffer resultBuffer, DepthBuffer depthBuffer,
			final double[] mvMatrix, final double[] prjMatrix, final double opacity) {

		final int srcTexture = srcBuffer.getTexture();
		final Size2i srcImageSize = srcBuffer.getImageSize();

		final int resultTexture = resultBuffer.getTexture();
		final Size2i resultImageSize = resultBuffer.getImageSize();

		final int depthTexture = (depthBuffer != null) ? depthBuffer.getTexture() : 0;


		pushMatrixAndExecute(new WrappedOperation<Object>() {
			public Object execute() {
				GL gl = _vrContext.getGL();

				gl.glViewport(0, 0, resultImageSize.width, resultImageSize.height);
				gl.glMatrixMode(GL.GL_PROJECTION);
				gl.glLoadMatrixd(prjMatrix, 0);
				gl.glMatrixMode(GL.GL_MODELVIEW);
				gl.glLoadMatrixd(mvMatrix, 0);


				gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
						GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, resultTexture, 0);
				gl.glDrawBuffer(GL.GL_COLOR_ATTACHMENT0_EXT);
				gl.glActiveTexture(GL.GL_TEXTURE0);
				gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, srcTexture);
				gl.glEnable(GL.GL_TEXTURE_RECTANGLE_EXT);

				if (depthTexture != 0) {
					gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
							GL.GL_DEPTH_ATTACHMENT_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, depthTexture, 0);
					gl.glEnable(GL.GL_DEPTH_TEST);
					gl.glClearDepth(1.0);
					gl.glClear(GL.GL_DEPTH_BUFFER_BIT);
				}

				gl.glBegin(GL.GL_QUADS);

				float a = (float)(opacity / 100.0);
				gl.glColor4f(a, a, a, a);

				gl.glTexCoord2f(0, 0);
				gl.glVertex2f(0, 0);

				gl.glTexCoord2f(srcImageSize.width, 0);
				gl.glVertex2f(srcImageSize.width, 0);

				gl.glTexCoord2f(srcImageSize.width, srcImageSize.height);
				gl.glVertex2f(srcImageSize.width, srcImageSize.height);

				gl.glTexCoord2f(0, srcImageSize.height);
				gl.glVertex2f(0, srcImageSize.height);

				gl.glEnd();

				if (depthTexture != 0) {
					gl.glDisable(GL.GL_DEPTH_TEST);
					gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
							GL.GL_DEPTH_ATTACHMENT_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, 0, 0);
				}

				gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, 0);
				gl.glDisable(GL.GL_TEXTURE_RECTANGLE_EXT);
				gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
						GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, 0, 0);

				return null;
			}
		});
	}

	public void scaleAndFlipVertical(VideoBuffer srcBuffer, VideoBuffer resultBuffer, final double scale) {
		final int srcTexture = srcBuffer.getTexture();
		final Size2i srcImageSize = srcBuffer.getImageSize();

		final int resultTexture = resultBuffer.getTexture();
		final Size2i resultImageSize = resultBuffer.getImageSize();

		pushMatrixAndExecute(new WrappedOperation<Object>() {
			public Object execute() {
				GL gl = _vrContext.getGL();
				GLU glu = _vrContext.getGLU();

				gl.glViewport(0, 0, resultImageSize.width, resultImageSize.height);

				gl.glMatrixMode(GL.GL_PROJECTION);
				gl.glLoadIdentity();
				glu.gluOrtho2D(0, resultImageSize.width, resultImageSize.height, 0);

				gl.glMatrixMode(GL.GL_MODELVIEW);
				gl.glLoadIdentity();
				gl.glScalef((float) scale, (float) scale, 1);

				gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
						GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, resultTexture, 0);
				gl.glDrawBuffer(GL.GL_COLOR_ATTACHMENT0_EXT);
				gl.glActiveTexture(GL.GL_TEXTURE0);
				gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, srcTexture);
				gl.glEnable(GL.GL_TEXTURE_RECTANGLE_EXT);

				gl.glBegin(GL.GL_QUADS);
				gl.glColor4f(1, 1, 1, 1);

				gl.glTexCoord2f(0, 0);
				gl.glVertex2f(0, 0);

				gl.glTexCoord2f(srcImageSize.width, 0);
				gl.glVertex2f(srcImageSize.width, 0);

				gl.glTexCoord2f(srcImageSize.width, srcImageSize.height);
				gl.glVertex2f(srcImageSize.width, srcImageSize.height);

				gl.glTexCoord2f(0, srcImageSize.height);
				gl.glVertex2f(0, srcImageSize.height);

				gl.glEnd();

				gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, 0);
				gl.glDisable(GL.GL_TEXTURE_RECTANGLE_EXT);
				gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
						GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, 0, 0);

				return null;
			}
		});
	}

	public void renderText(final FTGLfont font, final Color fillColor, final String text, VideoBuffer resultBuffer) {
		final int resultTexture = resultBuffer.getTexture();
		final VideoBounds resultBounds = resultBuffer.getBounds();

		pushMatrixAndExecute(new WrappedOperation<Object>() {
			public Object execute() {
				GL gl = _vrContext.getGL();
				GLU glu = _vrContext.getGLU();

				ortho2D(gl, glu, resultBounds.width, resultBounds.height);
				gl.glTranslatef((float) -resultBounds.x, (float) -resultBounds.y, 0);

				float scale = (float) _vrContext.getRenderResolution().scale;
				gl.glScalef(scale, -scale, scale);

				gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
						GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, resultTexture, 0);
				gl.glDrawBuffer(GL.GL_COLOR_ATTACHMENT0_EXT);

				// これをしておかないと ftglRenderFont 内の glEnable(GL_TEXTURE_2D) でエラー1282になる。
				gl.glActiveTexture(GL.GL_TEXTURE0);

				float a = (float)fillColor.a;
				float r = (float)fillColor.r * a; 
				float g = (float)fillColor.g * a; 
				float b = (float)fillColor.b * a; 
				gl.glColor4f(r, g, b, a);

				// オリジナルのFTGLは内部でアルファブレンドの設定を行っているが、Javieはそれを無効にした修正版FTGLを使用している。
				// Javieには次のアルファブレンドが必要なため。
				gl.glEnable(GL.GL_BLEND);
				gl.glBlendFuncSeparate(GL.GL_SRC_ALPHA, GL.GL_ZERO, GL.GL_ONE, GL.GL_ZERO);

				FTGL.instance.ftglRenderFont(font, text, FTGL.FTGL_RENDER_ALL);

				gl.glDisable(GL.GL_BLEND);

				gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
						GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, 0, 0);

				return null;
			}
		});
	}

	public void renderText(
			final FTGLfont font, final Color fillColor, final String text,
			VideoBuffer resultBuffer, DepthBuffer depthBuffer,
			final double[] mvMatrix, final double[] prjMatrix, final double opacity) {

		final int resultTexture = resultBuffer.getTexture();
		final Size2i resultImageSize = resultBuffer.getImageSize();

		final int depthTexture = (depthBuffer != null) ? depthBuffer.getTexture() : 0;


		pushMatrixAndExecute(new WrappedOperation<Object>() {
			public Object execute() {
				GL gl = _vrContext.getGL();

				gl.glViewport(0, 0, resultImageSize.width, resultImageSize.height);
				gl.glMatrixMode(GL.GL_PROJECTION);
				gl.glLoadMatrixd(prjMatrix, 0);
				gl.glMatrixMode(GL.GL_MODELVIEW);
				gl.glLoadMatrixd(mvMatrix, 0);

				float scale = (float) _vrContext.getRenderResolution().scale;
				gl.glScalef(scale, -scale, scale);


				gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
						GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, resultTexture, 0);
				gl.glDrawBuffer(GL.GL_COLOR_ATTACHMENT0_EXT);

				// これをしておかないと ftglRenderFont 内の glEnable(GL_TEXTURE_2D) でエラー1282になる。
				gl.glActiveTexture(GL.GL_TEXTURE0);

				if (depthTexture != 0) {
					gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
							GL.GL_DEPTH_ATTACHMENT_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, depthTexture, 0);
					gl.glEnable(GL.GL_DEPTH_TEST);
					gl.glClearDepth(1.0);
					gl.glClear(GL.GL_DEPTH_BUFFER_BIT);
				}

				float a = (float)(fillColor.a * opacity / 100.0);
				float r = (float)fillColor.r * a;
				float g = (float)fillColor.g * a;
				float b = (float)fillColor.b * a;
				gl.glColor4f(r, g, b, a);

				FTGL.instance.ftglRenderFont(font, text, FTGL.FTGL_RENDER_ALL);

				if (depthTexture != 0) {
					gl.glDisable(GL.GL_DEPTH_TEST);
					gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
							GL.GL_DEPTH_ATTACHMENT_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, 0, 0);
				}

				gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
						GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, 0, 0);

				return null;
			}
		});
	}

	public void flush() {
		_vrContext.getGL().glFlush();
	}

}
