/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.shaders;

import ch.kuramo.javie.api.annotations.GLProgram;
import ch.kuramo.javie.api.annotations.GLShader;
import ch.kuramo.javie.api.annotations.GLShader.ShaderType;

public class BlendModeShaders {

	@GLShader(ShaderType.FRAGMENT_SHADER)
	public static final String[] normal_func = {
		"vec4 blend_normal(vec4 src, vec4 dst)",
		"{",
		"	return src + dst * (1.0 - src.a);",
		"}"
	};

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="normal_func")
	public static final String[] NORMAL = {
		"uniform sampler2DRect texSrc;",
		"uniform sampler2DRect texDst;",
		"",
		"vec4 blend_normal(vec4 src, vec4 dst);",
		"",
		"void main(void)",
		"{",
		"	vec2 texCoord = gl_TexCoord[0].st;",
		"	vec4 src = texture2DRect(texSrc, texCoord);",
		"	vec4 dst = texture2DRect(texDst, texCoord);",
		"	gl_FragColor = blend_normal(src, dst);",
		"}"
	};

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram(attach="normal_func")
	public static final String[] INTERSECTING_BLEND = {
		// 以下のような変な書き方をしているのは、普通に書いたらATIで動かなかったから。
		// 少し前のドライバではそうだったけど、今の最新ドライバなら普通に書いても動くかも。
		"const int maxLayers = 7;",
		"",
		"uniform int nLayers;",
		"uniform sampler2DRect texDst;",
		"uniform sampler2DRect texSrc0;",
		"uniform sampler2DRect texSrc1;",
		"uniform sampler2DRect texSrc2;",
		"uniform sampler2DRect texSrc3;",
		"uniform sampler2DRect texSrc4;",
		"uniform sampler2DRect texSrc5;",
		"uniform sampler2DRect texSrc6;",
		"uniform sampler2DRect texDep0;",
		"uniform sampler2DRect texDep1;",
		"uniform sampler2DRect texDep2;",
		"uniform sampler2DRect texDep3;",
		"uniform sampler2DRect texDep4;",
		"uniform sampler2DRect texDep5;",
		"uniform sampler2DRect texDep6;",
		"uniform int blendMode[maxLayers];",
		"",
		"",
		"vec2 texCoord = gl_TexCoord[0].st;",
		"",
		"vec4 src(int i) {",
		"	return ",
		"		(i == 0) ? texture2DRect(texSrc0, texCoord) : ",
		"		(i == 1) ? texture2DRect(texSrc1, texCoord) : ",
		"		(i == 2) ? texture2DRect(texSrc2, texCoord) : ",
		"		(i == 3) ? texture2DRect(texSrc3, texCoord) : ",
		"		(i == 4) ? texture2DRect(texSrc4, texCoord) : ",
		"		(i == 5) ? texture2DRect(texSrc5, texCoord) : ",
		"		(i == 6) ? texture2DRect(texSrc6, texCoord) : ",
		"		vec4(0.0);",
		"}",
		"",
		"float dep0 = -1.0;",
		"float dep1 = -1.0;",
		"float dep2 = -1.0;",
		"float dep3 = -1.0;",
		"float dep4 = -1.0;",
		"float dep5 = -1.0;",
		"float dep6 = -1.0;",
		"",
		"void tex2dep(int i) {",
		"	if (i == 0) dep0 = texture2DRect(texDep0, texCoord).r; else ",
		"	if (i == 1) dep1 = texture2DRect(texDep1, texCoord).r; else ",
		"	if (i == 2) dep2 = texture2DRect(texDep2, texCoord).r; else ",
		"	if (i == 3) dep3 = texture2DRect(texDep3, texCoord).r; else ",
		"	if (i == 4) dep4 = texture2DRect(texDep4, texCoord).r; else ",
		"	if (i == 5) dep5 = texture2DRect(texDep5, texCoord).r; else ",
		"	if (i == 6) dep6 = texture2DRect(texDep6, texCoord).r; else ",
		"	{}",
		"}",
		"",
		"float getDep(int i) {",
		"	return ",
		"		(i == 0) ? dep0 : ",
		"		(i == 1) ? dep1 : ",
		"		(i == 2) ? dep2 : ",
		"		(i == 3) ? dep3 : ",
		"		(i == 4) ? dep4 : ",
		"		(i == 5) ? dep5 : ",
		"		(i == 6) ? dep6 : ",
		"		-1.0;",
		"}",
		"",
		"void setDep(int i, float dep) {",
		"	if (i == 0) dep0 = dep; else ",
		"	if (i == 1) dep1 = dep; else ",
		"	if (i == 2) dep2 = dep; else ",
		"	if (i == 3) dep3 = dep; else ",
		"	if (i == 4) dep4 = dep; else ",
		"	if (i == 5) dep5 = dep; else ",
		"	if (i == 6) dep6 = dep; else ",
		"	{}",
		"}",
		"",
		"",
		"vec4 blend_normal(vec4 src, vec4 dst);",
		"",
		"void main(void)",
		"{",
		// texSrcNのアルファ値が全てゼロ(nzaCount == 0)の場合、texDstの値をそのまま使えばよい。
		// texSrcNのうちひとつだけがゼロ以外のアルファを持つ(nzaCount == 1)場合、そのテクスチャだけをtexDstとブレンドすればよい。
		"	int nzaCount = 0;",
		"	int nzaLayer;",
		"	vec4 nzaColor;",
		"	for (int i = 0; i < nLayers; ++i) {",
		"		vec4 s = src(i);",
		"		if (s.a > 0.0) {",
		"			if (++nzaCount > 1) {",
		"				break;",
		"			}",
		"			nzaLayer = i;",
		"			nzaColor = s;",
		"		}",
		"	}",
		"	if (nzaCount == 0) {",
		"		gl_FragColor = texture2DRect(texDst, texCoord);",
		"	} else if (nzaCount == 1) {",
		"		if (blendMode[nzaLayer] == 0) {",
		"			gl_FragColor = blend_normal(nzaColor, texture2DRect(texDst, texCoord));",
		"		}",
		"	} else {",
		"		int order[maxLayers];",
		"		",
		"		for (int i = 0; i < nLayers; ++i) {",
		"			tex2dep(i);",
		"			order[i] = i;",
		"		}",
		"		",
		"		for (int i = nLayers - 2; i >= 0; --i) {",
		"			float dep = getDep(i);",
		"			int ord = order[i];",
		"			int j;",
		"			for (j = i + 1; j < nLayers && getDep(j) > dep; ++j) {",
		"				setDep(j - 1, getDep(j));",
		"				order[j - 1] = order[j];",
		"			}",
		"			setDep(j - 1, dep);",
		"			order[j - 1] = ord;",
		"		}",
		"		",
		"		vec4 dst = texture2DRect(texDst, texCoord);",
		"		for (int i = 0; i < nLayers; ++i) {",
		"			if (blendMode[i] == 0) {",
		"				dst = blend_normal(src(order[i]), dst);",
		"			}",
		"		}",
		"		gl_FragColor = dst;",
		"	}",
		"}"
	};


	private BlendModeShaders() { }

}
