/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.audio;

import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.plugin.PIAnimatableBoolean;
import ch.kuramo.javie.api.plugin.PIAnimatableDouble;
import ch.kuramo.javie.api.plugin.PIAudioBuffer;
import ch.kuramo.javie.api.plugin.PIAudioRenderContext;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.StereoMixer", category=Effect.AUDIO)
public class StereoMixer {

	@Property("100")
	private PIAnimatableDouble leftLevel;

	@Property("100")
	private PIAnimatableDouble rightLevel;

	@Property("-100")
	private PIAnimatableDouble leftPan;

	@Property("100")
	private PIAnimatableDouble rightPan;

	@Property
	private PIAnimatableBoolean invertPhase;


	private final PIAudioRenderContext context;

	@Inject
	public StereoMixer(PIAudioRenderContext context) {
		this.context = context;
	}

	public PIAudioBuffer doAudioEffect() {
		// TODO 値をアニメートしている場合、サンプル毎に値は異なるため、
		//      バッファ全体を同じ値で処理するのは正しくない。

		// TODO 値の範囲制限は @Property アノテーションで行う。
		double leftLevel = Math.max(0, Math.min(400, context.value(this.leftLevel)));
		double rightLevel = Math.max(0, Math.min(400, context.value(this.rightLevel)));
		double leftPan = Math.max(-100, Math.min(100, context.value(this.leftPan)));
		double rightPan = Math.max(-100, Math.min(100, context.value(this.rightPan)));
		double phase = context.value(this.invertPhase) ? -1.0 : 1.0;

		leftLevel /= 100.0;						// [0, 400] -> [0, 4]
		rightLevel /= 100.0;
		leftPan = (leftPan - 100.0) / -200.0;	// [-100,  100] -> [1, 0]
		rightPan = (rightPan + 100.0) / 200.0;	// [ 100, -100] -> [1, 0]


		PIAudioBuffer ab = context.doPreviousEffect();

		Object data = ab.getData();
		int frameCount = ab.getFrameCount();

		switch (ab.getAudioMode().dataType) {
			case SHORT:
				mixer((short[]) data, frameCount, leftLevel, rightLevel, leftPan, rightPan, phase);
				break;

			case INT:
				mixer((int[]) data, frameCount, leftLevel, rightLevel, leftPan, rightPan, phase);
				break;

			case FLOAT:
				mixer((float[]) data, frameCount, leftLevel, rightLevel, leftPan, rightPan, phase);
				break;
		}

		return ab;
	}

	private void mixer(short[] data, int frameCount,
			double leftLevel, double rightLevel, double leftPan, double rightPan, double phase) {

		for (int i = 0; i < frameCount; ++i) {
			double left  = data[i*2  ] * leftLevel;
			double right = data[i*2+1] * rightLevel;
			data[i*2  ] = clampToShort((left  * leftPan  + right * (1.0 - rightPan)) * phase);
			data[i*2+1] = clampToShort((right * rightPan + left  * (1.0 - leftPan )) * phase);
		}
	}

	private void mixer(int[] data, int frameCount,
			double leftLevel, double rightLevel, double leftPan, double rightPan, double phase) {

		for (int i = 0; i < frameCount; ++i) {
			double left  = data[i*2  ] * leftLevel;
			double right = data[i*2+1] * rightLevel;
			data[i*2  ] = clampToInt((left  * leftPan  + right * (1.0 - rightPan)) * phase);
			data[i*2+1] = clampToInt((right * rightPan + left  * (1.0 - leftPan )) * phase);
		}
	}

	private void mixer(float[] data, int frameCount,
			double leftLevel, double rightLevel, double leftPan, double rightPan, double phase) {

		for (int i = 0; i < frameCount; ++i) {
			double left  = data[i*2  ] * leftLevel;
			double right = data[i*2+1] * rightLevel;
			data[i*2  ] = (float) ((left  * leftPan  + right * (1.0 - rightPan)) * phase);
			data[i*2+1] = (float) ((right * rightPan + left  * (1.0 - leftPan )) * phase);
		}
	}

	private short clampToShort(double d) {
		return (short) Math.min(Math.max(d, Short.MIN_VALUE), Short.MAX_VALUE); 
	}

	private int clampToInt(double d) {
		return (int) Math.min(Math.max(d, Integer.MIN_VALUE), Integer.MAX_VALUE); 
	}

}
