/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.blurSharpen;

import java.util.Arrays;

import javax.media.opengl.GL;
import javax.media.opengl.glu.GLU;

import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.plugin.PIAnimatableBoolean;
import ch.kuramo.javie.api.plugin.PIAnimatableDouble;
import ch.kuramo.javie.api.plugin.PIAnimatableEnum;
import ch.kuramo.javie.api.plugin.PIAnimatableInteger;
import ch.kuramo.javie.api.plugin.PIShaderRegistry;
import ch.kuramo.javie.api.plugin.PIVideoBuffer;
import ch.kuramo.javie.api.plugin.PIVideoRenderContext;
import ch.kuramo.javie.effects.VideoEffectUtil;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.BoxBlur", category=Effect.BLUR_AND_SHARPEN)
public class BoxBlur {

	@Property
	private PIAnimatableDouble blurriness;

	@Property("1")
	private PIAnimatableInteger iterations;

	@Property
	private PIAnimatableEnum<BlurDimensions> blurDimensions;

	@Property
	private PIAnimatableBoolean repeatEdgePixels;

	@Property("true")
	private PIAnimatableBoolean fast;


	private final PIVideoRenderContext context;

	private final PIShaderRegistry shaders;

	@Inject
	public BoxBlur(PIVideoRenderContext context, PIShaderRegistry shaders) {
		this.context = context;
		this.shaders = shaders;
	}

	public VideoBounds getVideoBounds() {
		// TODO 値の範囲制限は @Property アノテーションで行う。
		double blur = Math.max(0, Math.min(500, context.value(blurriness)));
		if (blur == 0) {
			return null;
		}

		if (context.value(this.repeatEdgePixels)) {
			return null;
		}

		// TODO 値の範囲制限は @Property アノテーションで行う。
		int iterations = Math.max(1, Math.min(50, context.value(this.iterations)));

		BlurDimensions dimensions = context.value(blurDimensions);
		boolean horz = (dimensions != BlurDimensions.VERTICAL);
		boolean vert = (dimensions != BlurDimensions.HORIZONTAL);

		// blurが50より大きい場合はfastプロパティの値にかかわらず高速モード
		boolean fast = (blur > 50) || context.value(this.fast);

		// レンダリング解像度に合わせてblurの値を変換
		blur = context.getRenderResolution().scale(blur);


		VideoBounds bounds = context.getPreviousBounds();

		for (int i = 0; i < iterations; ++i) {
			bounds = calcVideoBounds(bounds, blur, horz, vert, fast);
		}

		return bounds;
	}

	private VideoBounds calcVideoBounds(
			VideoBounds bounds, double blur, boolean horz, boolean vert, boolean fast) {

		int sampleRatio = 1;
		if (fast) {
			for (int factor : BlurUtil.getDownSampleFactors(blur)) {
				int hFactor = horz ? factor : 1;
				int vFactor = vert ? factor : 1;
				bounds = new VideoBounds(bounds.x/hFactor, bounds.y/vFactor,
						(bounds.width+hFactor-1)/hFactor, (bounds.height+vFactor-1)/vFactor);
				sampleRatio *= factor;
			}
			if (sampleRatio != 1) {
				blur /= sampleRatio;
				blur -= 1.0/3.0;
			}
		}

		int radius = (int) Math.ceil(blur);


		bounds = VideoEffectUtil.expandBounds(bounds, radius, horz, vert);

		int hRatio = horz ? sampleRatio : 1;
		int vRatio = vert ? sampleRatio : 1;
		return new VideoBounds(bounds.x*hRatio, bounds.y*vRatio,
				bounds.width*hRatio, bounds.height*vRatio);
	}

	public PIVideoBuffer doVideoEffect() {
		// TODO 値の範囲制限は @Property アノテーションで行う。
		double blur = Math.max(0, Math.min(500, context.value(blurriness)));
		if (blur == 0) {
			return null;
		}

		boolean repeatEdgePixels = context.value(this.repeatEdgePixels);

		// TODO 値の範囲制限は @Property アノテーションで行う。
		int iterations = Math.max(1, Math.min(50, context.value(this.iterations)));

		BlurDimensions dimensions = context.value(blurDimensions);
		boolean horz = (dimensions != BlurDimensions.VERTICAL);
		boolean vert = (dimensions != BlurDimensions.HORIZONTAL);

		// blurが50より大きい場合はfastプロパティの値にかかわらず高速モード
		boolean fast = (blur > 50) || context.value(this.fast);

		// レンダリング解像度に合わせてblurの値を変換
		blur = context.getRenderResolution().scale(blur);


		PIVideoBuffer buffer = context.doPreviousEffect();

		for (int i = 0; i < iterations; ++i) {
			buffer = doBoxBlur(buffer, blur, repeatEdgePixels, horz, vert, fast);
		}

		return buffer;
	}

	private PIVideoBuffer doBoxBlur(
			PIVideoBuffer input, double blur, boolean repeatEdgePixels,
			boolean horz, boolean vert, boolean fast) {

		GL gl = context.getGL();
		GLU glu = context.getGLU();

		VideoBounds inputBounds = input.getBounds();

		int sampleRatio = 1;
		if (fast) {
			VideoBounds bounds = inputBounds;
			for (int factor : BlurUtil.getDownSampleFactors(blur)) {
				if (repeatEdgePixels) {
					VideoEffectUtil.setClampToEdge(input, gl);
				}

				int hFactor = horz ? factor : 1;
				int vFactor = vert ? factor : 1;
				bounds = new VideoBounds(bounds.x/hFactor, bounds.y/vFactor,
						(bounds.width+hFactor-1)/hFactor, (bounds.height+vFactor-1)/vFactor);

				PIVideoBuffer buf = context.createVideoBuffer(bounds);
				BlurUtil.doDownSample(input, buf, hFactor, vFactor, gl, glu, shaders);
				input.dispose();
				input = buf;

				sampleRatio *= factor;
			}
			if (sampleRatio != 1) {
				blur /= sampleRatio;
				blur -= 1.0/3.0;		// 理由がわからないが、1/3引いた方がダウンサンプルしない場合の結果に近くなる。
			}
		}

		int radius = (int) Math.ceil(blur);


		float[] kernel = new float[radius*2+1];
		Arrays.fill(kernel, (float)(1/(blur*2+1)));
		if (blur != radius) {
			kernel[0] *= 1-(radius-blur);
			kernel[kernel.length-1] *= 1-(radius-blur);
		}


		PIVideoBuffer buf1 = null, buf2 = null, buf3 = null;
		try {
			if (horz) {
				VideoBounds bounds = input.getBounds();

				if (repeatEdgePixels) {
					VideoEffectUtil.setClampToEdge(input, gl);
				} else {
					bounds = VideoEffectUtil.expandBounds(bounds, radius, true, false);
				}

				buf1 = context.createVideoBuffer(bounds);
				VideoEffectUtil.convolution1D(input, buf1, true, radius, kernel, gl, glu, shaders);
			} else {
				buf1 = input;
				input = null;
			}

			if (vert) {
				VideoBounds bounds = buf1.getBounds();

				if (repeatEdgePixels) {
					VideoEffectUtil.setClampToEdge(buf1, gl);
				} else {
					bounds = VideoEffectUtil.expandBounds(bounds, radius, false, true);
				}

				buf2 = context.createVideoBuffer(bounds);
				VideoEffectUtil.convolution1D(buf1, buf2, false, radius, kernel, gl, glu, shaders);
			} else {
				buf2 = buf1;
				buf1 = null;
			}

			if (sampleRatio != 1) {
				int hRatio = horz ? sampleRatio : 1;
				int vRatio = vert ? sampleRatio : 1;
				VideoBounds bounds;

				if (repeatEdgePixels) {
					VideoEffectUtil.setClampToEdge(buf2, gl);
					bounds = inputBounds;
				} else {
					bounds = buf2.getBounds();
					bounds = new VideoBounds(bounds.x*hRatio, bounds.y*vRatio, bounds.width*hRatio, bounds.height*vRatio);
				}

				buf3 = context.createVideoBuffer(bounds);
				BlurUtil.doUpSample(buf2, buf3, hRatio, vRatio, gl, glu);
			} else {
				buf3 = buf2;
				buf2 = null;
			}

			return buf3;

		} finally {
			if (input != null) {
				input.dispose();
			}
			if (buf1 != null) {
				buf1.dispose();
			}
			if (buf2 != null) {
				buf2.dispose();
			}
		}
	}

}
