package ch.kuramo.javie.effects.colorCorrection;

import javax.media.opengl.GL;
import javax.media.opengl.glu.GLU;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.GLProgram;
import ch.kuramo.javie.api.annotations.GLShader;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.GLShader.ShaderType;
import ch.kuramo.javie.api.plugin.PIAnimatableColor;
import ch.kuramo.javie.api.plugin.PIAnimatableDouble;
import ch.kuramo.javie.api.plugin.PIShaderProgram;
import ch.kuramo.javie.api.plugin.PIShaderRegistry;
import ch.kuramo.javie.api.plugin.PIVideoBuffer;
import ch.kuramo.javie.api.plugin.PIVideoRenderContext;
import ch.kuramo.javie.effects.VideoEffectUtil;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.Tritone", category=Effect.COLOR_CORRECTION)
public class Tritone {

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram
	public static final String[] TRITONE = {
		"uniform sampler2DRect texture;",
		"uniform vec3 highlight;",
		"uniform vec3 midtone;",
		"uniform vec3 shadow;",
		"uniform float blend;",
		"",
		"void main(void)",
		"{",
		"	vec4 src = texture2DRect(texture, gl_TexCoord[0].st);",
		"	float y = dot(src.rgb/src.a, vec3(0.299, 0.587, 0.114));",
		"	vec3 rgb;",
		"	if (y > 0.5) {",
		"		float y2 = (y-0.5)*2.0;",
		"		rgb = highlight*y2 + midtone*(1.0-y2);",
		"	} else {",
		"		float y2 = y*2.0;",
		"		rgb = shadow*(1.0-y2) + midtone*y2;",
		"	}",
		"	float a = src.a*(1.0-blend);",
		"	gl_FragColor = src*blend + vec4(rgb*a, a);",
		"}"
	};


	private final PIVideoRenderContext context;

	private final PIShaderProgram tritoneProgram;

	@Property("1.0,1.0,1.0")
	private PIAnimatableColor highlight;

	@Property("0.499,0.393,0.275")
	private PIAnimatableColor midtone;

	@Property("0.0,0.0,0.0")
	private PIAnimatableColor shadow;

	@Property("0.0")
	private PIAnimatableDouble blendWithOriginal;


	@Inject
	public Tritone(PIVideoRenderContext context, PIShaderRegistry shaders) {
		this.context = context;
		tritoneProgram = shaders.getProgram(Tritone.class, "TRITONE");
	}

	public PIVideoBuffer doVideoEffect() {
		// TODO 値の範囲制限は @Property アノテーションで行う。
		double blend = Math.max(0, Math.min(1, context.value(blendWithOriginal)/100));
		if (blend == 1) {
			return null;
		}

		Color highlight = context.value(this.highlight);
		Color midtone = context.value(this.midtone);
		Color shadow = context.value(this.shadow);

		PIVideoBuffer input = context.doPreviousEffect();
		PIVideoBuffer buf = null;
		try {
			VideoBounds bounds = input.getBounds();
			int w = bounds.width;
			int h = bounds.height;

			buf = context.createVideoBuffer(bounds);

			GL gl = context.getGL();
			GLU glu = context.getGLU();

			VideoEffectUtil.ortho2D(gl, glu, w, h);

			gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
					GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, buf.getTexture(), 0);
			gl.glDrawBuffer(GL.GL_COLOR_ATTACHMENT0_EXT);

			gl.glActiveTexture(GL.GL_TEXTURE0);
			gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, input.getTexture());

			synchronized (tritoneProgram) {
				gl.glUseProgram(tritoneProgram.getProgram());
				gl.glUniform1i(tritoneProgram.getUniformLocation("texture"), 0);
				gl.glUniform3f(tritoneProgram.getUniformLocation("highlight"), (float)highlight.r, (float)highlight.g, (float)highlight.b);
				gl.glUniform3f(tritoneProgram.getUniformLocation("midtone"), (float)midtone.r, (float)midtone.g, (float)midtone.b);
				gl.glUniform3f(tritoneProgram.getUniformLocation("shadow"), (float)shadow.r, (float)shadow.g, (float)shadow.b);
				gl.glUniform1f(tritoneProgram.getUniformLocation("blend"), (float) blend);

				gl.glBegin(GL.GL_QUADS);
				gl.glTexCoord2f(0, 0);
				gl.glVertex2f(0, 0);
				gl.glTexCoord2f(w, 0);
				gl.glVertex2f(w, 0);
				gl.glTexCoord2f(w, h);
				gl.glVertex2f(w, h);
				gl.glTexCoord2f(0, h);
				gl.glVertex2f(0, h);
				gl.glEnd();
	
				gl.glFlush();
				gl.glUseProgram(0);
			}

			gl.glActiveTexture(GL.GL_TEXTURE0);
			gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, 0);

			gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
					GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, 0, 0);

			PIVideoBuffer output = buf;
			buf = null;
			return output;

		} finally {
			input.dispose();
			if (buf != null) buf.dispose();
		}
	}

}
