/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.stylize;

import javax.media.opengl.GL;
import javax.media.opengl.glu.GLU;

import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.GLProgram;
import ch.kuramo.javie.api.annotations.GLShader;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.GLShader.ShaderType;
import ch.kuramo.javie.api.plugin.PIAnimatableBoolean;
import ch.kuramo.javie.api.plugin.PIAnimatableInteger;
import ch.kuramo.javie.api.plugin.PIShaderProgram;
import ch.kuramo.javie.api.plugin.PIShaderRegistry;
import ch.kuramo.javie.api.plugin.PIVideoBuffer;
import ch.kuramo.javie.api.plugin.PIVideoRenderContext;
import ch.kuramo.javie.effects.VideoEffectUtil;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.Mosaic", category=Effect.STYLIZE)
public class Mosaic {

	@GLShader(ShaderType.FRAGMENT_SHADER)
	@GLProgram
	public static final String[] MOSAIC_SHARP_COLORS = {
		"uniform sampler2DRect texture;",
		"uniform float hSize;",
		"uniform float vSize;",
		"",
		"void main(void)",
		"{",
		"	float s = (floor(gl_TexCoord[0].s/hSize)+0.5)*hSize;",
		"	float t = (floor(gl_TexCoord[0].t/vSize)+0.5)*vSize;",
		"	gl_FragColor = texture2DRect(texture, vec2(s, t));",
		"}"
	};


	private final PIVideoRenderContext context;

	private final PIShaderRegistry shaders;

	private final PIShaderProgram sharpColorsProgram;

	@Property("10")
	private PIAnimatableInteger horizontalBlocks;

	@Property("10")
	private PIAnimatableInteger verticalBlocks;

	@Property
	private PIAnimatableBoolean sharpColors;


	@Inject
	public Mosaic(PIVideoRenderContext context, PIShaderRegistry shaders) {
		this.context = context;
		this.shaders = shaders;
		this.sharpColorsProgram = shaders.getProgram(Mosaic.class, "MOSAIC_SHARP_COLORS");
	}

	public PIVideoBuffer doVideoEffect() {
		// TODO 値の範囲制限は @Property アノテーションで行う。
		int hBlocks = Math.max(1, context.value(horizontalBlocks));
		int vBlocks = Math.max(1, context.value(verticalBlocks));
		boolean sharpColors = context.value(this.sharpColors);

		if (sharpColors) {
			return doSharpColorsMosaic(hBlocks, vBlocks);
		} else {
			return doAverageColorsMosaic(hBlocks, vBlocks);
		}
	}

	private PIVideoBuffer doSharpColorsMosaic(int hBlocks, int vBlocks) {
		PIVideoBuffer input = context.doPreviousEffect();
		VideoBounds bounds = input.getBounds();
		int w = bounds.width;
		int h = bounds.height;

		hBlocks = Math.min(hBlocks, w);
		vBlocks = Math.min(vBlocks, h);

		if (hBlocks == w && vBlocks == h) {
			return input;
		}

		PIVideoBuffer buffer = null;
		try {
			buffer = context.createVideoBuffer(bounds);

			GL gl = context.getGL();
			GLU glu = context.getGLU();

			VideoEffectUtil.ortho2D(gl, glu, w, h);

			gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
					GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, buffer.getTexture(), 0);
			gl.glDrawBuffer(GL.GL_COLOR_ATTACHMENT0_EXT);

			gl.glActiveTexture(GL.GL_TEXTURE0);
			gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, input.getTexture());

			synchronized (sharpColorsProgram) {
				gl.glUseProgram(sharpColorsProgram.getProgram());
				gl.glUniform1i(sharpColorsProgram.getUniformLocation("texture"), 0);
				gl.glUniform1f(sharpColorsProgram.getUniformLocation("hSize"), (float)w/hBlocks);
				gl.glUniform1f(sharpColorsProgram.getUniformLocation("vSize"), (float)h/vBlocks);

				gl.glBegin(GL.GL_QUADS);
				gl.glTexCoord2f(0, 0);
				gl.glVertex2f(0, 0);
				gl.glTexCoord2f(w, 0);
				gl.glVertex2f(w, 0);
				gl.glTexCoord2f(w, h);
				gl.glVertex2f(w, h);
				gl.glTexCoord2f(0, h);
				gl.glVertex2f(0, h);
				gl.glEnd();
	
				gl.glFlush();
				gl.glUseProgram(0);
			}

			gl.glActiveTexture(GL.GL_TEXTURE0);
			gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, 0);

			gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
					GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, 0, 0);

			PIVideoBuffer output = buffer;
			buffer = null;
			return output;

		} finally {
			input.dispose();
			if (buffer != null) buffer.dispose();
		}
	}

	private PIVideoBuffer doAverageColorsMosaic(int hBlocks, int vBlocks) {
		PIVideoBuffer input = context.doPreviousEffect();
		PIVideoBuffer buffer = null;
		try {
			VideoBounds bounds = input.getBounds();
			int w = bounds.width;
			int h = bounds.height;

			hBlocks = Math.min(hBlocks, w);
			vBlocks = Math.min(vBlocks, h);

			if (hBlocks != w || vBlocks != h) {
				buffer = context.createVideoBuffer(new VideoBounds(hBlocks, vBlocks));
				average(input, buffer);
				scale(buffer, input);
			}

			PIVideoBuffer output = input;
			input = null;
			return output;

		} finally {
			if (input != null) input.dispose();
			if (buffer != null) buffer.dispose();
		}
	}

	private void average(PIVideoBuffer src, PIVideoBuffer dst) {
		GL gl = context.getGL();
		GLU glu = context.getGLU();

		PIVideoBuffer src0 = src;
		PIVideoBuffer tmpBuf = null;
		try {
			VideoBounds srcBounds = src.getBounds();
			VideoBounds dstBounds = dst.getBounds();
			int hBlocks = dstBounds.width;
			int vBlocks = dstBounds.height;
			int hbs = (srcBounds.width + hBlocks - 1) / hBlocks;
			int vbs = (srcBounds.height + vBlocks - 1) / vBlocks;

			// CONVOLUTIONフィルタのカーネルサイズが縦横それぞれ7までなので
			// ブロックサイズが7より大きい場合は複数回にわけて処理する
			if (hbs > 7 || vbs > 7) {
				tmpBuf = context.createVideoBuffer(new VideoBounds(
						(hbs > 7) ? hBlocks*7 : hBlocks, (vbs > 7) ? vBlocks*7 : vBlocks));
				average(src, tmpBuf);
				src = tmpBuf;

				srcBounds = src.getBounds();
				hbs = (srcBounds.width + hBlocks - 1) / hBlocks;
				vbs = (srcBounds.height + vBlocks - 1) / vBlocks;
			}

			int ksize = hbs * vbs;
			float[] kernel = new float[ksize];
			float[] offset = new float[ksize*2];

			for (int j = 0; j < vbs; ++j) {
				for (int i = 0; i < hbs; ++i) {
					int k = j*hbs+i;
					kernel[k] = 1f/ksize;
					offset[k*2  ] = i - hbs/2;
					offset[k*2+1] = j - vbs/2;
				}
			}

			VideoEffectUtil.setClampToEdge(src, gl);
			VideoEffectUtil.ortho2D(gl, glu, hBlocks, vBlocks);

			gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
					GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, dst.getTexture(), 0);
			gl.glDrawBuffer(GL.GL_COLOR_ATTACHMENT0_EXT);

			gl.glActiveTexture(GL.GL_TEXTURE0);
			gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, src.getTexture());

			PIShaderProgram program = shaders.getProgram(VideoEffectUtil.class, "CONVOLUTION");
			synchronized (program) {
				gl.glUseProgram(program.getProgram());
				gl.glUniform1i(program.getUniformLocation("texture"), 0);
				gl.glUniform1i(program.getUniformLocation("ksize"), ksize);
				gl.glUniform1fv(program.getUniformLocation("kernel[0]"), ksize, kernel, 0);
				gl.glUniform2fv(program.getUniformLocation("offset[0]"), ksize*2, offset, 0);

				gl.glBegin(GL.GL_QUADS);
				gl.glTexCoord2f(0, 0);
				gl.glVertex2f(0, 0);
				gl.glTexCoord2f(srcBounds.width, 0);
				gl.glVertex2f(dstBounds.width, 0);
				gl.glTexCoord2f(srcBounds.width, srcBounds.height);
				gl.glVertex2f(dstBounds.width, dstBounds.height);
				gl.glTexCoord2f(0, srcBounds.height);
				gl.glVertex2f(0, dstBounds.height);
				gl.glEnd();

				gl.glFlush();
				gl.glUseProgram(0);
			}

			gl.glActiveTexture(GL.GL_TEXTURE0);
			gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, 0);

			gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
					GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, 0, 0);

		} finally {
			if (tmpBuf != null) tmpBuf.dispose();
			VideoEffectUtil.setClampToBorder(src0, gl);
		}
	}

	private void scale(PIVideoBuffer src, PIVideoBuffer dst) {
		GL gl = context.getGL();
		GLU glu = context.getGLU();

		VideoEffectUtil.setNearest(src, gl);

		VideoBounds srcBounds = src.getBounds();
		VideoBounds dstBounds = dst.getBounds();

		VideoEffectUtil.ortho2D(gl, glu, dstBounds.width, dstBounds.height);

		gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
				GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, dst.getTexture(), 0);
		gl.glDrawBuffer(GL.GL_COLOR_ATTACHMENT0_EXT);

		gl.glActiveTexture(GL.GL_TEXTURE0);
		gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, src.getTexture());
		gl.glEnable(GL.GL_TEXTURE_RECTANGLE_EXT);

		gl.glBegin(GL.GL_QUADS);
		gl.glColor4f(1, 1, 1, 1);
		gl.glTexCoord2f(0, 0);
		gl.glVertex2f(0, 0);
		gl.glTexCoord2f(srcBounds.width, 0);
		gl.glVertex2f(dstBounds.width, 0);
		gl.glTexCoord2f(srcBounds.width, srcBounds.height);
		gl.glVertex2f(dstBounds.width, dstBounds.height);
		gl.glTexCoord2f(0, srcBounds.height);
		gl.glVertex2f(0, dstBounds.height);
		gl.glEnd();

		gl.glActiveTexture(GL.GL_TEXTURE0);
		gl.glBindTexture(GL.GL_TEXTURE_RECTANGLE_EXT, 0);
		gl.glDisable(GL.GL_TEXTURE_RECTANGLE_EXT);

		gl.glFramebufferTexture2DEXT(GL.GL_FRAMEBUFFER_EXT,
				GL.GL_COLOR_ATTACHMENT0_EXT, GL.GL_TEXTURE_RECTANGLE_EXT, 0, 0);
	}

}
