/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app;

import java.io.File;
import java.util.Set;

import org.eclipse.core.commands.operations.IOperationHistoryListener;
import org.eclipse.core.commands.operations.IUndoableOperation;
import org.eclipse.core.commands.operations.OperationHistoryEvent;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IViewReference;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.application.ActionBarAdvisor;
import org.eclipse.ui.application.IActionBarConfigurer;
import org.eclipse.ui.application.IWorkbenchWindowConfigurer;
import org.eclipse.ui.application.WorkbenchWindowAdvisor;

import ch.kuramo.javie.app.actions.NewProjectAction;
import ch.kuramo.javie.app.project.ModifyLayerCompSettingsOperation;
import ch.kuramo.javie.app.project.ProjectEvent;
import ch.kuramo.javie.app.project.ProjectEventHub;
import ch.kuramo.javie.app.project.ProjectListener;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.app.views.MediaPlayerView;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.MediaItem;

public class ApplicationWorkbenchWindowAdvisor
		extends WorkbenchWindowAdvisor implements ProjectListener, IOperationHistoryListener {

	private static final String DONE_SAVE_CONFIRMATION = "DONE_SAVE_CONFIRMATION";


	private boolean _maximizeShell = true;


	public ApplicationWorkbenchWindowAdvisor(IWorkbenchWindowConfigurer configurer) {
		super(configurer);
	}

	public ActionBarAdvisor createActionBarAdvisor(IActionBarConfigurer configurer) {
		return new ApplicationActionBarAdvisor(configurer);
	}

	public IStatus restoreState(IMemento memento) {
		_maximizeShell = false;
		return Status.OK_STATUS;
	}

	public void preWindowOpen() {
		IWorkbenchWindowConfigurer configurer = getWindowConfigurer();
		configurer.setInitialSize(new Point(800, 600));
		configurer.setShowMenuBar(true);
		configurer.setShowCoolBar(false);
        configurer.setShowStatusLine(true);
        configurer.setShowProgressIndicator(true);
		configurer.setTitle("Javie");
	}

	public void postWindowCreate() {
		PlatformUI.getWorkbench().getOperationSupport()
				.getOperationHistory().addOperationHistoryListener(this);

		ProjectEventHub.getHub().addProjectListener(this);

		Shell shell = getWindowConfigurer().getWindow().getShell();

		if (_maximizeShell) {
			shell.setMaximized(true);
		}

		shell.getDisplay().asyncExec(new Runnable() {
			public void run() {
				NewProjectAction.newProject(getWindowConfigurer().getWindow());
			}
		});
	}

	static boolean saveConfirmationRequired(IWorkbenchWindow window) {
		Object data = window.getShell().getData(DONE_SAVE_CONFIRMATION);
		return !Boolean.TRUE.equals(data);
	}

	public boolean preWindowShellClose() {
		IWorkbenchWindow window = getWindowConfigurer().getWindow();
		ProjectManager pm = ProjectManager.forWorkbenchWindow(window);
		if (pm != null) {
			switch (SaveUtil.saveIfDirty(pm)) {
				case NOT_DIRTY:
				case SAVED:
				case UNSAVED:
					window.getShell().setData(DONE_SAVE_CONFIRMATION, Boolean.TRUE);
					return true;
			}
		}
		return false;
	}

	public void postWindowClose() {
		ProjectManager pm = ProjectManager.forWorkbenchWindow(getWindowConfigurer().getWindow());
		if (pm != null) {
			pm.dispose();
		}

		ProjectEventHub.getHub().removeProjectListener(this);

		PlatformUI.getWorkbench().getOperationSupport()
				.getOperationHistory().removeOperationHistoryListener(this);
	}

	private void updateWindowTitle(ProjectManager pm) {
		File file = pm.getFile();
		boolean dirty = pm.isDirty();

		String title = (file != null) ? file.getName() : ProjectManager.DEFAULT_NAME;
		if (dirty) {
			title += " *";
		}

		Shell shell = getWindowConfigurer().getWindow().getShell();
		shell.setText(title);
		shell.setModified(dirty);
	}

	private void resetWindowTitle() {
		Shell shell = getWindowConfigurer().getWindow().getShell();
		shell.setText("Javie");
		shell.setModified(false);
	}

	private void hideViews(Set<Item> items) {
		IWorkbenchPage page = getWindowConfigurer().getWindow().getActivePage();

		for (Item item : items) {
			if (item instanceof MediaItem) {
				IViewReference viewRef = page.findViewReference(MediaPlayerView.ID, item.getId());
				if (viewRef != null) {
					page.hideView(viewRef);
				}
			}
			if (item instanceof CompositionItem) {
				IViewReference viewRef = page.findViewReference(LayerCompositionView.ID, item.getId());
				if (viewRef != null) {
					page.hideView(viewRef);
				}
			}
		}
	}

	public void handleProjectEvent(ProjectEvent event) {
		ProjectManager pm = event.getProjectManager();
		if (pm.getWorkbenchWindow() != getWindowConfigurer().getWindow()) {
			return;
		}

		switch (event.type) {
			case PROJECT_INITIALIZE:
			case DIRTY_CHANGE:
			case FILE_CHANGE:
				updateWindowTitle(pm);
				break;

			case PROJECT_DISPOSE:
				resetWindowTitle();
				break;

			case ITEMS_REMOVE:
				hideViews(ProjectEvent.getItems(event));
				break;
		}
	}

	public void historyNotification(OperationHistoryEvent event) {
		IWorkbenchWindow window = getWindowConfigurer().getWindow();
		ProjectManager pm = ProjectManager.forWorkbenchWindow(window);
		if (pm == null) {
			// プロジェクトが初期化される前または破棄された後にここに来ることがあるのか？
			return;
		}

		IUndoableOperation op = event.getOperation();

		if (op instanceof ModifyLayerCompSettingsOperation && op.hasContext(pm.getUndoContext())) {
			String compItemId = ((ModifyLayerCompSettingsOperation) op).getCompositionItemId();
			IWorkbenchPage page = window.getActivePage();

			switch (event.getEventType()) {
				case OperationHistoryEvent.ABOUT_TO_EXECUTE:
				case OperationHistoryEvent.ABOUT_TO_REDO:
				case OperationHistoryEvent.ABOUT_TO_UNDO:
					LayerCompositionView compView = UIUtil.findView(page, LayerCompositionView.ID, compItemId);
					if (compView != null) {
						UIUtil.hideView(page, compView);
					}

					MediaPlayerView playerView = UIUtil.findView(page, MediaPlayerView.ID, compItemId);
					if (playerView != null) {
						UIUtil.hideView(page, playerView);
					}
					break;

				case OperationHistoryEvent.DONE:
				case OperationHistoryEvent.REDONE:
				case OperationHistoryEvent.UNDONE:
					UIUtil.showView(page, LayerCompositionView.ID, compItemId, IWorkbenchPage.VIEW_ACTIVATE);
					UIUtil.showView(page, MediaPlayerView.ID, compItemId, IWorkbenchPage.VIEW_VISIBLE);
					break;
			}
		}
	}

}
