/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.List;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaItemLayer;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.Util;

public class ModifyLayerInPointOperation extends ProjectOperation {

	private class Entry {
		final String layerId;
		final Time oldInPoint;
		final Time newInPoint;

		public Entry(String layerId, Time oldInPoint, Time newInPoint) {
			super();
			this.layerId = layerId;
			this.oldInPoint = oldInPoint;
			this.newInPoint = newInPoint;
		}
	}


	private final String _compId;

	private volatile List<Entry> _entries = Util.newList();


	public ModifyLayerInPointOperation(
			ProjectManager projectManager, LayerComposition comp,
			Object[][] layersAndInPoints, Time deltaTime, String relation) {

		super(projectManager, "インポイントの変更", relation);

		_noEffect = true;

		projectManager.checkComposition(comp);
		_compId = comp.getId();

		Time oneFrame = Time.fromFrameNumber(1, comp.getFrameDuration());

		for (Object[] o : layersAndInPoints) {
			Layer layer = (Layer) o[0];
			Time newInPoint = ((Time) o[1]).add(deltaTime);

			Time outPoint = layer.getOutPoint();
			if (!newInPoint.before(outPoint)) {
				newInPoint = outPoint.subtract(oneFrame);
			}

			if (layer instanceof MediaItemLayer && !LayerNature.isTimeRemapEnabled(layer)) {
				MediaInput input = ((MediaItemLayer) layer).getMediaInput();
				if (input != null && input.getDuration() != null) {
					Time startTime = layer.getStartTime();
					if (newInPoint.before(startTime)) {
						newInPoint = startTime;
					}
				}
			}

			Time oldInPoint = layer.getInPoint();
			_entries.add(new Entry(layer.getId(), oldInPoint, newInPoint));

			if (!newInPoint.equals(oldInPoint)) {
				_noEffect = false;
			}
		}
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		modifyInPoint(false, project, pm);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		modifyInPoint(true, project, pm);
		return Status.OK_STATUS;
	}

	private void modifyInPoint(boolean undo, Project project, ProjectManager pm) {
		List<Layer> layers = null;
		if (pm != null) {
			layers = Util.newList();
		}

		LayerComposition comp = project.getComposition(_compId);

		for (Entry e : _entries) {
			Layer layer = comp.getLayer(e.layerId);
			layer.setInPoint(undo ? e.oldInPoint : e.newInPoint);

			if (pm != null) {
				layers.add(layer);
			}
		}

		if (pm != null) {
			pm.fireLayerTimesChange(comp, layers);
		}
	}

	@Override
	protected boolean merge(ProjectOperation operation, IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) {

		if (pm == null) {
			// _entries はシャドウプロジェクトに反映される前に
			// 後続のマージ可能オペレーションで上書きされているかもしれない。
			// しかし、シャドウオペレーションのキューが全て実行された時点での値は一致するので問題ない。
			modifyInPoint(false, project, null);
			return true;
		}

		if (!(operation instanceof ModifyLayerInPointOperation)) {
			return false;
		}

		ModifyLayerInPointOperation newOp = (ModifyLayerInPointOperation) operation;
		if (!newOp._compId.equals(_compId) || newOp._entries.size() != _entries.size()) {
			return false;
		}

		List<Entry> mergedEntries = Util.newList();
		boolean noEffect = true;

		for (int i = 0, n = _entries.size(); i < n; ++i) {
			Entry oldEntry = _entries.get(i);
			Entry newEntry = newOp._entries.get(i);
			if (!newEntry.layerId.equals(oldEntry.layerId)) {
				return false;
			}
			mergedEntries.add(new Entry(oldEntry.layerId, oldEntry.oldInPoint, newEntry.newInPoint));
			if (!newEntry.newInPoint.equals(oldEntry.oldInPoint)) {
				noEffect = false;
			}
		}

		_entries = mergedEntries;
		modifyInPoint(false, project, pm);

		_noEffect = noEffect;

		return true;
	}

}
