/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.List;
import java.util.Set;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.Keyframe;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaLayer;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.PropertyDescriptor;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class ReverseLayerTimeOperation extends ProjectOperation {

	private static class LayerEntry {
		private final String layerId;
		private final Time startTime;
		private final double rate;

		private LayerEntry(String layerId, Time startTime, double rate) {
			this.layerId = layerId;
			this.startTime = startTime;
			this.rate = rate;
		}
	}

	private static class PropertyEntry {
		private final String layerId;
		private final int effectIndex;
		private final String property;
		private final String data;

		public PropertyEntry(String layerId, int effectIndex, String property, String data) {
			this.layerId = layerId;
			this.effectIndex = effectIndex;
			this.property = property;
			this.data = data;
		}
	}


	private final String _compId;

	private final Set<LayerEntry> _layers = Util.newSet();

	private final Set<PropertyEntry> _properties = Util.newSet();

	@Inject
	private ProjectEncoder _encoder;

	@Inject
	private ProjectDecoder _decoder;


	public ReverseLayerTimeOperation(ProjectManager pm, Set<Layer> layers) {
		super(pm, "レイヤーの時間反転");
		InjectorHolder.getInjector().injectMembers(this);

		if (layers.isEmpty()) {
			_compId = null;
			_noEffect = true;
			return;
		}

		LayerComposition comp = pm.checkLayer(layers.iterator().next());
		_compId = comp.getId();

		for (Layer layer : layers) {
			if (pm.checkLayer(layer) != comp) {
				throw new IllegalArgumentException();
			}

			_layers.add(new LayerEntry(layer.getId(), layer.getStartTime(), layer.getRate()));

			for (String property : PropertyUtil.getPropertyNames(layer)) {
				Object o = PropertyUtil.getProperty(layer, property);
				if (o instanceof AnimatableValue<?>) {
					AnimatableValue<?> avalue = (AnimatableValue<?>) o;
					if (avalue.hasKeyframe()) {
						_properties.add(new PropertyEntry(layer.getId(), -1, property, _encoder.encodeElement(avalue)));
					}
				}
			}

			if (layer instanceof EffectableLayer) {
				List<Effect> effects = ((EffectableLayer) layer).getEffects();
				for (int i = 0; i < effects.size(); ++i) {
					Effect e = effects.get(i);
					for (PropertyDescriptor pd : e.getEffectDescriptor().getPropertyDescriptors()) {
						Object o = pd.get(e);
						if (o instanceof AnimatableValue<?>) {
							AnimatableValue<?> avalue = (AnimatableValue<?>) o;
							if (avalue.hasKeyframe()) {
								_properties.add(new PropertyEntry(layer.getId(), i, pd.getName(), _encoder.encodeElement(avalue)));
							}
						}
					}
				}
			}
		}
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		reverseKeyframes(project, pm);
		reverseLayers(project, pm);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		restoreKeyframes(project, pm);
		restoreLayers(project, pm);
		return Status.OK_STATUS;
	}

	private void reverseKeyframes(Project project, ProjectManager pm) {
		LayerComposition comp = project.getComposition(_compId);
		List<Object[]> data = (pm != null) ? Util.<Object[]>newList() : null;

		for (PropertyEntry e : _properties) {
			Layer layer = comp.getLayer(e.layerId);
			AnimatableValue<Object> avalue;
			if (e.effectIndex == -1) {
				avalue = PropertyUtil.getProperty(layer, e.property);
			} else {
				Effect effect = ((EffectableLayer) layer).getEffects().get(e.effectIndex);
				avalue = PropertyUtil.getProperty(effect, e.property);
			}

			Time inPoint = layer.getInPoint();
			Time outPoint = layer.getOutPoint();

			List<Keyframe<Object>> keyframes = Util.newList(avalue.getKeyframes());
			avalue.clearKeyframes(avalue.getStaticValue());
			for (Keyframe<Object> kf : keyframes) {
				avalue.putKeyframe(outPoint.subtract(kf.time.subtract(inPoint)), kf.value, kf.interpolation);
			}

			if (pm != null) {
				data.add(new Object[] { layer, e.effectIndex, e.property, null });
			}
		}

		if (pm != null) {
			pm.fireKeyframesChange(comp, data.toArray(new Object[data.size()][]));
		}
	}

	private void restoreKeyframes(Project project, ProjectManager pm) throws ExecutionException {
		LayerComposition comp = project.getComposition(_compId);
		List<Object[]> data = (pm != null) ? Util.<Object[]>newList() : null;

		for (PropertyEntry e : _properties) {
			Layer layer = comp.getLayer(e.layerId);
			AnimatableValue<Object> avalue;
			if (e.effectIndex == -1) {
				avalue = PropertyUtil.getProperty(layer, e.property);
			} else {
				Effect effect = ((EffectableLayer) layer).getEffects().get(e.effectIndex);
				avalue = PropertyUtil.getProperty(effect, e.property);
			}

			restore(e.data, avalue);

			if (pm != null) {
				data.add(new Object[] { layer, e.effectIndex, e.property, null });
			}
		}

		if (pm != null) {
			pm.fireKeyframesChange(comp, data.toArray(new Object[data.size()][]));
		}
	}

	// TODO ThreeDSwitchOperation 等にほぼ同じメソッドがあるので整理する。
	private <A extends AnimatableValue<V>, V> void restore(String data, A dst) throws ExecutionException {
		try {
			@SuppressWarnings("unchecked")
			A src = _decoder.decodeElement(data, (Class<A>) dst.getClass());

			dst.reset(src.getStaticValue());
			dst.setExpression(src.getExpression());

			for (Keyframe<V> kf : src.getKeyframes()) {
				dst.putKeyframe(kf);
			}

		} catch (ProjectDecodeException e) {
			throw new ExecutionException("error decoding AnimatableValue data", e);
		}
	}

	private void reverseLayers(Project project, ProjectManager pm) {
		LayerComposition comp = project.getComposition(_compId);
		Set<Layer> layers = Util.newSet();

		for (LayerEntry e : _layers) {
			Layer layer = comp.getLayer(e.layerId);

			if (layer instanceof MediaLayer) {
				MediaInput mi = ((MediaLayer) layer).getMediaInput();
				Time duration = mi.getDuration();
				if (duration != null) {
					Time startTime = layer.getStartTime();
					Time inPoint = layer.getInPoint();
					Time outPoint = layer.getOutPoint();
					layer.setStartTime(inPoint.subtract(startTime.add(duration).subtract(outPoint)));
				}
			}

			layer.setRate(-e.rate);
			layers.add(layer);
		}

		if (pm != null) {
			pm.fireLayerTimesChange(comp, layers);
		}
	}

	private void restoreLayers(Project project, ProjectManager pm) {
		LayerComposition comp = project.getComposition(_compId);
		Set<Layer> layers = Util.newSet();

		for (LayerEntry e : _layers) {
			Layer layer = comp.getLayer(e.layerId);
			layer.setStartTime(e.startTime);
			layer.setRate(e.rate);
			layers.add(layer);
		}

		if (pm != null) {
			pm.fireLayerTimesChange(comp, layers);
		}
	}

}
