/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views.layercomp;

import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.widgets.Combo;

import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.app.views.ComboBoxViewerCellEditor;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.core.AnimatableString;
import ch.kuramo.javie.core.TextLayer;
import ch.kuramo.javie.core.services.Font;
import ch.kuramo.javie.core.services.FontList;

import com.google.inject.Inject;

public class TextFontElement extends LayerAnimatableValueElement {

	public TextFontElement(Element parent, TextLayer layer) {
		super(parent, layer, "font", "フォント");
	}

	protected AnimatableValueElementDelegate<?> createDelegate() {
		AnimatableString avalue = PropertyUtil.getProperty(layer, property);
		return new TextFontElementDelegate(this, name, avalue);
	}

}

class TextFontElementDelegate extends AnimatableStringElementDelegate {

	private ComboBoxViewerCellEditor editor;

	@Inject
	private FontList fontList;


	TextFontElementDelegate(
			AnimatableValueElement element, String name, AnimatableString avalue) {

		super(element, name, avalue);
		InjectorHolder.getInjector().injectMembers(this);
	}

	protected void drawValue(GC gc, int x, int y, int height, boolean focused) {
		Font font = fontList.get(value);
		drawValue(gc, x, y, height, focused, (font != null) ? font.fullName : "      ");
	}

	public CellEditor getCellEditor(int columnIndex) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			if (editor == null) {
				editor = new ComboBoxViewerCellEditor(element.viewer.getTree(), SWT.READ_ONLY);
				editor.setContenProvider(ArrayContentProvider.getInstance());
				editor.setLabelProvider(new FontLabelProvider());
				editor.setActivationStyle(ComboBoxViewerCellEditor.DROP_DOWN_ON_PROGRAMMATIC_ACTIVATION);

				Combo combo = (Combo) editor.getControl();
				combo.setVisibleItemCount(35);
				combo.addSelectionListener(new SelectionAdapter() {
					public void widgetSelected(SelectionEvent e) {
						setCellEditorValue(LayerCompositionView.VALUE_COL, editor.getValue());
					}
				});
			}
			editor.setInput(fontList.list());
			return editor;

		} else {
			return super.getCellEditor(columnIndex);
		}
	}

	public Object getCellEditorValue(int columnIndex) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			return fontList.get(value);
		} else {
			return super.getCellEditorValue(columnIndex);
		}
	}

	public void setCellEditorValue(int columnIndex, Object value) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			value = (value != null) ? ((Font) value).psName : "";
		}
		super.setCellEditorValue(columnIndex, value);
	}

	private static class FontLabelProvider extends LabelProvider {
		public String getText(Object element) {
			return ((Font) element).fullName;
		}
	}

}
