/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IAudioBuffer;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.core.AudioBuffer;

import com.google.inject.Inject;

public class AudioBufferImpl implements AudioBuffer, IAudioBuffer {

	private final AudioMode _audioMode;

	private final int _frameCount;

	private final int _dataLength;

	private final int _dataLengthInBytes;

	private IArray<?> _data;

	@Inject
	private IArrayPools _arrayPools;


	public AudioBufferImpl(AudioMode audioMode, int frameCount) {
		_audioMode = audioMode;
		_frameCount = frameCount;
		_dataLength = frameCount * audioMode.channels;
		_dataLengthInBytes = _dataLength * audioMode.sampleSize;
	}

	@Override
	protected void finalize() throws Throwable {
		dispose();
		super.finalize();
	}

	public void dispose() {
		if (_data != null) {
			_data.release();
			_data = null;
		}
	}

	public AudioMode getAudioMode() {
		return _audioMode;
	}

	public int getFrameCount() {
		return _frameCount;
	}

	public int getDataLength() {
		return _dataLength;
	}

	public int getDataLengthInBytes() {
		return _dataLengthInBytes;
	}

	public Object getData() {
		if (_data == null) {
			switch (_audioMode.dataType) {
				case SHORT:
					_data = _arrayPools.getShortArray(_dataLength);
					break;
				case INT:
					_data = _arrayPools.getIntArray(_dataLength);
					break;
				case FLOAT:
					_data = _arrayPools.getFloatArray(_dataLength);
					break;
				default:
					throw new UnsupportedOperationException(
							"unsupported AudioMode.DataType: " + _audioMode.dataType);
			}
		}
		return _data.getArray();
	}

}
