/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import ch.kuramo.javie.api.RenderResolution;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.core.CameraLayer;
import ch.kuramo.javie.core.CollapseTransformation;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.MediaItemLayer;
import ch.kuramo.javie.core.TransformableLayer;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.services.VideoRenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;

class LayerMatrixUtil {

	static void multModelViewMatrix(
			TransformableLayer layer, Vec2d offset, CollapseTransformation ct,
			VideoRenderContext vrContext, VideoRenderSupport vrSupport) {

		LayerMatrixUtil util = new LayerMatrixUtil(vrContext, vrSupport);

		if (ct != null) {
			util.multCollapseTransformation(ct, LayerNature.isThreeD(layer));
		}

		util.multModelViewMatrix(layer, offset, true);
	}

	static void multParentMatrix(
			CameraLayer cameraLayer, VideoRenderContext vrContext, VideoRenderSupport vrSupport) {

		Layer parent = cameraLayer.getParent();
		if (parent != null) {
			new LayerMatrixUtil(vrContext, vrSupport)
					.multModelViewMatrixInverse(parent, null, true);
		}
	}



	private final VideoRenderContext _vrContext;

	private final VideoRenderSupport _vrSupport;


	private LayerMatrixUtil(VideoRenderContext vrContext, VideoRenderSupport vrSupport) {
		super();
		_vrContext = vrContext;
		_vrSupport = vrSupport;
	}

	private void multCollapseTransformation(final CollapseTransformation ct, final boolean threeD) {
		final MediaItemLayer layer = ct.getLayer();

		if (ct.getParent() != null) {
			multCollapseTransformation(ct.getParent(), threeD && LayerNature.isThreeD(layer));
		}

		_vrContext.saveAndExecute(new WrappedOperation<Object>() {
			public Object execute() {
				_vrContext.setTime(ct.getTime());
				multModelViewMatrix(layer, null, threeD);
				return null;
			}
		});
	}

	private void multModelViewMatrix(Layer layer, Vec2d offset, boolean threeD) {
		threeD &= LayerNature.isThreeD(layer);

		Layer parent = layer.getParent();
		if (parent != null) {
			multModelViewMatrix(parent, null, threeD);
		}

		if (layer instanceof TransformableLayer) {
			multTransformableLayerMatrix((TransformableLayer) layer, offset, threeD, false);
		} else if (layer instanceof CameraLayer) {
			multCameraLayerMatrix((CameraLayer) layer, threeD, false);
		}
	}

	private void multModelViewMatrixInverse(Layer layer, Vec2d offset, boolean threeD) {
		threeD &= LayerNature.isThreeD(layer);

		if (layer instanceof TransformableLayer) {
			multTransformableLayerMatrix((TransformableLayer) layer, offset, threeD, true);
		} else if (layer instanceof CameraLayer) {
			multCameraLayerMatrix((CameraLayer) layer, threeD, true);
		}

		Layer parent = layer.getParent();
		if (parent != null) {
			multModelViewMatrixInverse(parent, null, threeD);
		}
	}

	private void multTransformableLayerMatrix(final TransformableLayer tlayer, final Vec2d offset, final boolean threeD, final boolean inverse) {
		// ビデオをサポートしていない場合は計算しても無駄なので何もせずに抜ける。
		// (抜けずに行列計算をしたとしても恒等変換になるので結果に影響は無いはずだが)
		if (!LayerNature.isVideoNature(tlayer)) {
			return;
		}

		Vec3d anchorPoint = tlayer.getAnchorPoint().value(_vrContext);
		if (offset != null) {
			anchorPoint = new Vec3d(anchorPoint.x + offset.x, anchorPoint.y + offset.y, anchorPoint.z);
		}

		Vec3d scale = tlayer.getScale().value(_vrContext);
		Vec3d orientation;
		Vec3d rotation;
		Vec3d position = tlayer.getPosition().value(_vrContext);

		if (threeD) {
			orientation = tlayer.getOrientation().value(_vrContext);
			rotation = new Vec3d(
					tlayer.getRotationX().value(_vrContext),
					tlayer.getRotationY().value(_vrContext),
					tlayer.getRotationZ().value(_vrContext));

		} else {
			anchorPoint = new Vec3d(anchorPoint.x, anchorPoint.y, 0);
			scale = new Vec3d(scale.x, scale.y, 100);
			orientation = new Vec3d(0, 0, 0);
			rotation = new Vec3d(0, 0, tlayer.getRotationZ().value(_vrContext));
			position = new Vec3d(position.x, position.y, 0);
		}

		RenderResolution resolution = _vrContext.getRenderResolution();

		if (inverse) {
			_vrSupport.multModelViewMatrixInverse(
					resolution.scale(anchorPoint), scale, orientation, rotation, resolution.scale(position));
		} else {
			_vrSupport.multModelViewMatrix(
					resolution.scale(anchorPoint), scale, orientation, rotation, resolution.scale(position));
		}
	}

	private void multCameraLayerMatrix(final CameraLayer clayer, final boolean threeD, final boolean inverse) {
		RenderResolution resolution = _vrContext.getRenderResolution();

		Vec3d position = clayer.getPosition().value(_vrContext);
		Vec3d pointOfInterest = clayer.getPointOfInterest().value(_vrContext);

		if (threeD) {
			Vec3d orientation = clayer.getOrientation().value(_vrContext);
			Vec3d rotation = new Vec3d(
					clayer.getRotationX().value(_vrContext),
					clayer.getRotationY().value(_vrContext),
					clayer.getRotationZ().value(_vrContext));

			// This is OK: カメラレイヤーを親にしたときの変換は、カメラ自身の変換の逆変換となる。
			if (inverse) {
				_vrSupport.multCameraMatrix(
						orientation, rotation, resolution.scale(position), resolution.scale(pointOfInterest), false);
			} else {
				_vrSupport.multCameraMatrixInverse(
						orientation, rotation, resolution.scale(position), resolution.scale(pointOfInterest));
			}
		} else {
			// AEでは、2Dレイヤーの親にカメラを指定したとき、目標点の値がアンカーポイントとして解釈されていると思われる動作をする。
			// 実装上便宜的にそうなってるだけなのか、何らかの意味があってそうなっているのか不明だが、
			// とりあえずその動作に合わせるためにこのように以下のようにしている。

			Vec3d anchorPoint = new Vec3d(pointOfInterest.x, pointOfInterest.y, 0);
			Vec3d scale = new Vec3d(100, 100, 100);
			Vec3d orientation = new Vec3d(0, 0, 0);
			Vec3d rotation = new Vec3d(0, 0, clayer.getRotationZ().value(_vrContext));
			position = new Vec3d(position.x, position.y, 0);

			if (inverse) {
				_vrSupport.multModelViewMatrixInverse(
						resolution.scale(anchorPoint), scale, orientation, rotation, resolution.scale(position));
			} else {
				_vrSupport.multModelViewMatrix(
						resolution.scale(anchorPoint), scale, orientation, rotation, resolution.scale(position));
			}
		}
	}

}
