/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.util.Collections;

import net.arnx.jsonic.JSONHint;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.core.Camera;
import ch.kuramo.javie.core.CollapseTransformation;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.ExpressionScope;
import ch.kuramo.javie.core.FrameBlend;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaItem;
import ch.kuramo.javie.core.MediaItemLayer;
import ch.kuramo.javie.core.MotionBlur;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.VideoBuffer;
import ch.kuramo.javie.core.VideoLayerBuffer;
import ch.kuramo.javie.core.VideoLayerComposer;
import ch.kuramo.javie.core.VideoLayerRenderer;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.annotations.ProjectElement;
import ch.kuramo.javie.core.services.VideoEffectPipeline;
import ch.kuramo.javie.core.services.VideoRenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;

import com.google.inject.Inject;

@ProjectElement("mediaItemLayer")
public class MediaItemLayerImpl extends AbstractMediaLayer implements MediaItemLayer {

	private String _itemId;

	private MediaItem _mediaItem;

	private boolean _ctcr;

	private FrameBlend _frameBlend = FrameBlend.NONE;

	@Inject
	private VideoRenderContext _vrContext;

	@Inject
	private VideoRenderSupport _vrSupport;

	@Inject
	private VideoEffectPipeline _vePipeline;


	@Override
	protected void initialize(boolean videoAvailable, boolean audioAvailable) {
		throw new UnsupportedOperationException("Use initialize(MediaItem) method instead.");
	}

	public void initialize(MediaItem mediaItem) {
		MediaInput input = mediaItem.getMediaInput();
		boolean videoAvailable = input.isVideoAvailable();
		boolean audioAvailable = input.isAudioAvailable();

		if (!videoAvailable && !audioAvailable) {
			throw new IllegalArgumentException("No video nor audio is available.");
		}

		super.initialize(videoAvailable, audioAvailable);

		_itemId = mediaItem.getId();
		_mediaItem = mediaItem;

		setName(mediaItem.getName());
	}

	public String getItemId() {
		return _itemId;
	}

	public void setItemId(String itemId) {
		_itemId = itemId;
	}

	@JSONHint(ignore=true)
	public MediaItem getItem() {
		return _mediaItem;
	}

	public boolean isCTCR() {
		return _ctcr;
	}

	public void setCTCR(boolean ctcr) {
		_ctcr = ctcr;
	}

	public FrameBlend getFrameBlend() {
		return _frameBlend;
	}

	public void setFrameBlend(FrameBlend frameBlend) {
		_frameBlend = frameBlend;
	}

	@JSONHint(ignore=true)
	public MediaInput getMediaInput() {
		return _mediaItem.getMediaInput();
	}

	@JSONHint(ignore=true)
	public boolean isPrecompositionLayer() {
		return (getPrecomposition() != null);
	}

	private LayerComposition getPrecomposition() {
		if (_mediaItem instanceof CompositionItem) {
			Composition comp = ((CompositionItem) _mediaItem).getComposition();
			if (comp instanceof LayerComposition) {
				return (LayerComposition) comp;
			}
		}
		return null;
	}

	@Override
	public void afterDecode(Project p, LayerComposition c) throws ProjectDecodeException {
		super.afterDecode(p, c);

		_mediaItem = p.getItem(_itemId);
		if (_mediaItem == null) {
			throw new ProjectDecodeException(
					"no such MediaItem found: id=" + _itemId);
		}
	}

	@Override
	public void prepareExpression(ExpressionScope scope) {
		super.prepareExpression(scope);

		LayerComposition preComp = getPrecomposition();
		if (preComp != null) {
			ExpressionScope preCompScope = scope.createPrecompositionScope(preComp);
			if (preCompScope != null) {
				preComp.prepareExpression(preCompScope);
			}
		}
	}

	@Override
	public void setupVideoRenderer(
			VideoLayerComposer composer, Camera camera, CollapseTransformation ct,
			MotionBlur motinBlur, boolean frameBlendEnabled) {

		if (LayerNature.isCTCR(this)) {
			LayerComposition preComp = getPrecomposition();
			if (preComp != null) {
				setupCollapseTransformation(composer, camera, ct, motinBlur, preComp);
				return;
			}
		}

		super.setupVideoRenderer(composer, camera, ct, motinBlur, frameBlendEnabled);
	}

	private void setupCollapseTransformation(
			final VideoLayerComposer composer, final Camera camera,
			final CollapseTransformation ct, final MotionBlur motionBlur,
			final LayerComposition preComp) {

		final Time time = _vrContext.getTime();

		final double opacity;
		if (ct != null) {
			opacity = getOpacity().value(_vrContext) / 100d * ct.getOpacity();
		} else {
			opacity = getOpacity().value(_vrContext) / 100d;
		}

		final CollapseTransformation ct2 = new CollapseTransformation() {
			public CollapseTransformation getParent()	{ return ct; }
			public MediaItemLayer getLayer()			{ return MediaItemLayerImpl.this; }
			public Time getTime()						{ return time; }
			public double getOpacity()					{ return opacity; }
		};

		if (getEffects().isEmpty()) {
			_vrContext.saveAndExecute(new WrappedOperation<Object>() {
				public Object execute() {
					_vrContext.setTime(calcVideoMediaTime(getMediaInput()));
					preComp.setupCollapseTransformation(composer, camera, ct2, motionBlur);
					return null;
				}
			});
		} else {
			final VideoLayerComposerImpl composer2 = new VideoLayerComposerImpl(
					_vrContext, _vrSupport, _vrContext.getColorMode(),
					camera.getViewportSize(), camera.getPosition());

			_vrContext.saveAndExecute(new WrappedOperation<Object>() {
				public Object execute() {
					_vrContext.setTime(calcVideoMediaTime(getMediaInput()));
					preComp.setupCollapseTransformation(composer2, camera, ct2, motionBlur);
					return null;
				}
			});

			composer.add2D(new VideoLayerRenderer() {
				public VideoLayerBuffer render(boolean withDepthBuffer) {
					VideoBuffer effectedBuffer = null;
					VideoBuffer resultBuffer = null;
					try {
						WrappedOperation<VideoBuffer> inputBufferOperation = new WrappedOperation<VideoBuffer>() {
							public VideoBuffer execute() {
								VideoBuffer vb = composer2.compose();
								_vrSupport.premultiply(vb);
								return vb;
							}
						};

						effectedBuffer = _vePipeline.doVideoEffects(
								isEffectsEnabled() ? getEffects() : Collections.<Effect>emptyList(), inputBufferOperation);

						VideoBounds resultBounds = new VideoBounds(camera.getViewportSize());
						if (effectedBuffer.getBounds().equals(resultBounds)) {
							resultBuffer = effectedBuffer;
							effectedBuffer = null;
						} else {
							resultBuffer = _vrSupport.createVideoBuffer(_vrContext.getColorMode(), resultBounds);
							resultBuffer.allocateAsTexture();
							resultBuffer.clear();
							_vrSupport.copy(effectedBuffer, resultBuffer);
						}

						VideoLayerBuffer vlb = new VideoLayerBufferImpl(resultBuffer, null, getBlendMode(), 1.0);
						resultBuffer = null;
						return vlb;

					} finally {
						if (resultBuffer != null) resultBuffer.dispose();
						if (effectedBuffer != null) effectedBuffer.dispose();
					}
				}
			});
		}
	}

}
