/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.vecmath.Vector3d;

import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.VideoBuffer;
import ch.kuramo.javie.core.VideoLayerBuffer;
import ch.kuramo.javie.core.VideoLayerComposer;
import ch.kuramo.javie.core.VideoLayerRenderer;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.services.VideoRenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;

class VideoLayerComposerImpl implements VideoLayerComposer {

	private static class Entry {
		final VideoLayerRenderer renderer;
		final Vec3d[] vertices;
		final Vec3d normalVec;
		final String intersectionGroup;
		final Set<Entry> intersectionMembers = Util.newSet();

		Entry(VideoLayerRenderer renderer, Vec3d[] vertices, String intersectionGroup) {
			this.renderer = renderer;
			this.vertices = vertices;
			this.intersectionGroup = intersectionGroup;

			Vector3d nv = new Vector3d();
			nv.cross(new Vector3d(vertices[0].x-vertices[1].x, vertices[0].y-vertices[1].y, vertices[0].z-vertices[1].z),
					new Vector3d(vertices[0].x-vertices[2].x, vertices[0].y-vertices[2].y, vertices[0].z-vertices[2].z));
			normalVec = new Vec3d(nv.x, nv.y, nv.z);
		}

		Entry(VideoLayerRenderer renderer) {
			this.renderer = renderer;
			this.vertices = null;
			this.normalVec = null;
			this.intersectionGroup = "";
		}
	}

	private static class Section {
		final List<Entry> entries = Util.newList();
		final Map<String, Entry> intersectionGroupMap = Util.newMap();

		Section() { }
		Section(Entry e) { add(e); }

		void add(Entry e) {
			if (e.intersectionGroup.length() == 0) {
				entries.add(e);
				return;
			}

			Entry ee = intersectionGroupMap.get(e.intersectionGroup);
			if (ee != null) {
				ee.intersectionMembers.add(e);
			} else {
				entries.add(e);
				intersectionGroupMap.put(e.intersectionGroup, e);
			}
		}
	}

	private final Comparator<Entry> _entryComparator = new Comparator<Entry>() {
		private int signum(Vec3d pt, Entry e) {
			return (int) Math.signum(
					  e.normalVec.x * (pt.x - e.vertices[0].x)
					+ e.normalVec.y * (pt.y - e.vertices[0].y)
					+ e.normalVec.z * (pt.z - e.vertices[0].z));
		}

		private int sameSideVertices(Entry o1, Entry o2) {
			int plus = 0, minus = 0;
			for (int i = 1; i < 5; ++i) {
				switch (signum(o1.vertices[i], o2)) {
					case  1: ++plus; break;
					case -1: --minus; break;
				}
			}
			return (plus > -minus) ? plus : minus;
		}

		public int compare(Entry o1, Entry o2) {
			int ssv1 = sameSideVertices(o1, o2);
			if (Math.abs(ssv1) == 4) {
				return Math.signum(ssv1) == signum(_cameraPosition, o2) ? 1 : -1;
			}

			int ssv2 = sameSideVertices(o2, o1);
			if (Math.abs(ssv2) == 4) {
				return Math.signum(ssv2) == signum(_cameraPosition, o1) ? -1 : 1;
			}

			return (int) Math.signum(o2.vertices[0].z - o1.vertices[0].z);
		}
	};

	private final VideoRenderContext _vrContext;

	private final VideoRenderSupport _vrSupport;

	private final ColorMode _colorMode;

	private final Size2i _renderSize;

	private final Vec3d _cameraPosition;

	private final List<Section> _sections = Util.newList();

	private Section _currentSection;


	VideoLayerComposerImpl(
			VideoRenderContext vrContext, VideoRenderSupport vrSupport,
			ColorMode colorMode, Size2i renderSize, Vec3d cameraPosition) {

		_vrContext = vrContext;
		_vrSupport = vrSupport;
		_colorMode = colorMode;
		_renderSize = renderSize;
		_cameraPosition = cameraPosition;
	}

	public void add2D(VideoLayerRenderer renderer) {
		_currentSection = null;
		_sections.add(new Section(new Entry(renderer)));
	}

	public void add3D(VideoLayerRenderer renderer, Vec3d[] vertices, String intersectionGroup) {
		if (vertices.length != 5) {
			throw new IllegalArgumentException("vertices length must be 5: length=" + vertices.length);
		}
		if (_currentSection == null) {
			_currentSection = new Section();
			_sections.add(_currentSection);
		}
		_currentSection.add(new Entry(renderer, vertices, intersectionGroup));
	}

	VideoBuffer compose() {
		VideoBuffer vb = _vrSupport.createVideoBuffer(_colorMode, _renderSize);
		vb.allocateAsTexture();
		vb.clear();

		List<Entry> entries = Util.newList();

		for (Section s : _sections) {
			Collections.sort(s.entries, _entryComparator);
			entries.addAll(s.entries);
		}

		for (final Entry e : entries) {
			VideoLayerBuffer vlb = _vrContext.saveAndExecute(new WrappedOperation<VideoLayerBuffer>() {
				public VideoLayerBuffer execute() {
					return e.renderer.render(!e.intersectionMembers.isEmpty());
				}
			});

			if (!e.intersectionMembers.isEmpty()) {
				List<VideoLayerBuffer> intersectionGroupBuffers = Util.newList();
				intersectionGroupBuffers.add(vlb);

				for (final Entry ee : e.intersectionMembers) {
					vlb = _vrContext.saveAndExecute(new WrappedOperation<VideoLayerBuffer>() {
						public VideoLayerBuffer execute() {
							return ee.renderer.render(true);
						}
					});
					intersectionGroupBuffers.add(vlb);
				}

				vb = blend(intersectionGroupBuffers, vb);

			} else {
				vb = blend(Collections.singleton(vlb), vb);
			}
		}

		return vb;
	}

	private VideoBuffer blend(Collection<VideoLayerBuffer> intersectionGroupBuffers, VideoBuffer vb) {
		VideoBuffer vbNew = _vrSupport.createVideoBuffer(_colorMode, _renderSize);
		vbNew.allocateAsTexture();
		vbNew.clear();

		if (intersectionGroupBuffers.size() == 1) {
			VideoLayerBuffer vlb = intersectionGroupBuffers.iterator().next();
			_vrSupport.blend(vlb.getVideoBuffer(), vb, vbNew, vlb.getBlendMode(), vlb.getOpacity());
		} else {
			_vrSupport.blend(intersectionGroupBuffers, vb, vbNew);
		}

		for (VideoLayerBuffer vlb : intersectionGroupBuffers) {
			vlb.dispose();
		}
		vb.dispose();

		return vbNew;
	}

}
