/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.util.List;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.IAnimatableValue;
import ch.kuramo.javie.api.IAudioBuffer;
import ch.kuramo.javie.api.IObjectArray;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IAudioEffectContext;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.AudioBuffer;
import ch.kuramo.javie.core.AudioEffect;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.services.AudioEffectPipeline;
import ch.kuramo.javie.core.services.AudioRenderContext;
import ch.kuramo.javie.core.services.AudioRenderSupport;

import com.google.inject.Inject;

public class AudioEffectPipelineImpl implements AudioEffectPipeline {

	@Inject
	private AudioRenderContext _arContext;

	@Inject
	private AudioRenderSupport _arSupport;

	@Inject
	private IArrayPools _arrayPools;


	public AudioBuffer doAudioEffects(List<Effect> effects, WrappedOperation<AudioBuffer> inputOperation) {
		AudioEffectContextImpl impl = new AudioEffectContextImpl(enabledAudioEffectsOnly(effects), inputOperation);
		AudioEffectContextProxy.push(impl);
		try {
			return (AudioBuffer) impl.doPreviousEffect();
		} finally {
			AudioEffectContextProxy.pop();
		}
	}

	private List<AudioEffect> enabledAudioEffectsOnly(List<Effect> effects) {
		List<AudioEffect> audioEffects = Util.newList();
		for (Effect e : effects) {
			if (e instanceof AudioEffect && e.isEnabled()) {
				audioEffects.add((AudioEffect) e);
			}
		}
		return audioEffects;
	}


	private class AudioEffectContextImpl implements IAudioEffectContext {

		private List<AudioEffect> effects;

		private final WrappedOperation<AudioBuffer> inputOperation;


		private AudioEffectContextImpl(List<AudioEffect> effects, WrappedOperation<AudioBuffer> inputOperation) {
			this.effects = effects;
			this.inputOperation = inputOperation;
		}

		public IAudioBuffer doPreviousEffect() {
			if (effects.isEmpty()) {
				AudioBuffer ab = _arContext.saveAndExecute(inputOperation);
				if (ab == null) {
					ab = _arSupport.createAudioBuffer();
					_arSupport.clear(ab);
				}
				return (IAudioBuffer) ab;
			}

			final AudioEffect prevEffect = effects.get(effects.size() - 1);

			return executePrevious(new WrappedOperation<IAudioBuffer>() {
				public IAudioBuffer execute() {
					IAudioBuffer buffer = prevEffect.doAudioEffect();
					return (buffer != null) ? buffer : doPreviousEffect();
				}
			});
		}

		private <T> T executePrevious(final WrappedOperation<T> wop) {
			List<AudioEffect> savedEffects = effects;
			try {
				effects = effects.subList(0, effects.size() - 1);
				return _arContext.saveAndExecute(wop);
			} finally {
				effects = savedEffects;
			}
		}

		public Time getTime() {
			return _arContext.getTime();
		}

		public void setTime(Time time) {
			_arContext.setTime(time);
		}

		public <V> V value(IAnimatableValue<V> avalue) {
			@SuppressWarnings("unchecked")
			AnimatableValue<V> av = (AnimatableValue<V>) avalue;
			return av.value(_arContext);
		}

		public AudioMode getAudioMode() {
			return _arContext.getAudioMode();
		}

		public int getFrameCount() {
			return _arContext.getFrameCount();
		}

		public void setFrameCount(int count) {
			_arContext.setFrameCount(count);
		}

		public IAudioBuffer createAudioBuffer() {
			return (IAudioBuffer) _arSupport.createAudioBuffer();
		}

		public <V> IObjectArray<V> values(IAnimatableValue<V> avalue) {
			return values(avalue, _arContext.getEvaluationResolution());
		}

		public <V> IObjectArray<V> values(IAnimatableValue<V> avalue, int evalResolution) {
			@SuppressWarnings("unchecked")
			AnimatableValue<V> av = (AnimatableValue<V>) avalue;

			evalResolution = Math.max(1, Math.min(evalResolution, _arContext.getEvaluationResolution()));

			IObjectArray<V> objArray = _arrayPools.getObjectArray(_arContext.getFrameCount());
			av.values(objArray, _arContext.getAudioMode().sampleDuration, evalResolution, _arContext);
			return objArray;
		}

	}

}
