/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.util.Arrays;

import ch.kuramo.javie.api.IObjectArray;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.core.AudioBuffer;
import ch.kuramo.javie.core.internal.AudioBufferImpl;
import ch.kuramo.javie.core.services.AudioRenderContext;
import ch.kuramo.javie.core.services.AudioRenderSupport;

import com.google.inject.Inject;
import com.google.inject.Injector;

public class AudioRenderSupportImpl implements AudioRenderSupport {

	@Inject
	private AudioRenderContext _arContext;

	@Inject
	private Injector _injector;


	public AudioBuffer createAudioBuffer() {
		return createAudioBuffer(_arContext.getFrameCount());
	}

	public AudioBuffer createAudioBuffer(int frameCount) {
		AudioBufferImpl ab = new AudioBufferImpl(_arContext.getAudioMode(), frameCount);
		_injector.injectMembers(ab);
		return ab;
	}

	public AudioBuffer mix(AudioBuffer ab1, AudioBuffer ab2) {
		if (ab1.getAudioMode() != ab2.getAudioMode()) {
			throw new IllegalArgumentException("AudioMode of AudioBuffer arguments must match");
		}

		if (ab1.getFrameCount() > ab2.getFrameCount()) {
			AudioBuffer tmp = ab1;
			ab1 = ab2;
			ab2 = tmp;
		}

		Object data1 = ab1.getData();
		Object data2 = ab2.getData();
		int dataLen = ab1.getDataLength();

		switch (ab1.getAudioMode().dataType) {
			case SHORT:
				mix((short[]) data1, (short[]) data2, dataLen);
				break;

			case INT:
				mix((int[]) data1, (int[]) data2, dataLen);
				break;

			case FLOAT:
				mix((float[]) data1, (float[]) data2, dataLen);
				break;

			default:
				throw new IllegalArgumentException(
						"unsupported AudioMode.DataType: " + ab1.getAudioMode().dataType);
		}

		return ab1;
	}

	private void mix(short[] data1, short[] data2, int dataLen) {
		for (int i = 0; i < dataLen; ++i) {
			int v1 = data1[i];
			int v2 = data2[i];
			data1[i] = (short) Math.min(Math.max(v1 + v2, Short.MIN_VALUE), Short.MAX_VALUE);
		}
	}

	private void mix(int[] data1, int[] data2, int dataLen) {
		for (int i = 0; i < dataLen; ++i) {
			long v1 = data1[i];
			long v2 = data2[i];
			data1[i] = (int) Math.min(Math.max(v1 + v2, Integer.MIN_VALUE), Integer.MAX_VALUE);
		}
	}

	private void mix(float[] data1, float[] data2, int dataLen) {
		for (int i = 0; i < dataLen; ++i) {
			data1[i] += data2[i];
		}
	}

	public void amplify(AudioBuffer ab, IObjectArray<Vec2d> amplitude) {
		if (ab.getAudioMode().channels != 2) {
			throw new IllegalArgumentException("channels: " + ab.getAudioMode().channels);
		}

		Object data = ab.getData();
		int frameCount = ab.getFrameCount();

		switch (ab.getAudioMode().dataType) {
			case SHORT:
				amplify((short[]) data, frameCount, amplitude);
				break;

			case INT:
				amplify((int[]) data, frameCount, amplitude);
				break;

			case FLOAT:
				amplify((float[]) data, frameCount, amplitude);
				break;

			default:
				throw new IllegalArgumentException(
						"unsupported AudioMode.DataType: " + ab.getAudioMode().dataType);
		}
	}

	private void amplify(short[] data, int frameCount, IObjectArray<Vec2d> amplitude) {
		Object[] array = amplitude.getArray();
		for (int i = 0; i < frameCount; ++i) {
			Vec2d amp = (Vec2d) array[i];
			double leftAmp = Math.pow(10d, amp.x/20d);
			double rightAmp = Math.pow(10d, amp.y/20d);
			data[i*2  ] = (short) Math.min(Math.max(data[i*2  ]*leftAmp , Short.MIN_VALUE), Short.MAX_VALUE);
			data[i*2+1] = (short) Math.min(Math.max(data[i*2+1]*rightAmp, Short.MIN_VALUE), Short.MAX_VALUE);
		}
	}

	private void amplify(int[] data, int frameCount, IObjectArray<Vec2d> amplitude) {
		Object[] array = amplitude.getArray();
		for (int i = 0; i < frameCount; ++i) {
			Vec2d amp = (Vec2d) array[i];
			double leftAmp = Math.pow(10d, amp.x/20d);
			double rightAmp = Math.pow(10d, amp.y/20d);
			data[i*2  ] = (int) Math.min(Math.max(data[i*2  ]*leftAmp , Integer.MIN_VALUE), Integer.MAX_VALUE);
			data[i*2+1] = (int) Math.min(Math.max(data[i*2+1]*rightAmp, Integer.MIN_VALUE), Integer.MAX_VALUE);
		}
	}

	private void amplify(float[] data, int frameCount, IObjectArray<Vec2d> amplitude) {
		Object[] array = amplitude.getArray();
		for (int i = 0; i < frameCount; ++i) {
			Vec2d amp = (Vec2d) array[i];
			double leftAmp = Math.pow(10d, amp.x/20d);
			double rightAmp = Math.pow(10d, amp.y/20d);
			data[i*2  ] *= leftAmp;
			data[i*2+1] *= rightAmp;
		}
	}

	public void clear(AudioBuffer ab) {
		clear(ab, 0, ab.getFrameCount());
	}

	public void clear(AudioBuffer ab, int frameOffset, int frameCount) {
		Object data = ab.getData();
		int channels = ab.getAudioMode().channels;

		switch (ab.getAudioMode().dataType) {
			case SHORT:
				Arrays.fill((short[])data, frameOffset*channels, (frameOffset+frameCount)*channels, (short)0);
				break;

			case INT:
				Arrays.fill((int[])data, frameOffset*channels, (frameOffset+frameCount)*channels, 0);
				break;

			case FLOAT:
				Arrays.fill((float[])data, frameOffset*channels, (frameOffset+frameCount)*channels, 0);
				break;

			default:
				throw new IllegalArgumentException(
						"unsupported AudioMode.DataType: " + ab.getAudioMode().dataType);
		}
	}

}
