/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.io.File;
import java.util.Map;
import java.util.Set;

import javassist.CannotCompileException;
import javassist.ClassPool;
import javassist.CtClass;
import javassist.CtNewMethod;
import javassist.LoaderClassPath;
import javassist.NotFoundException;

import org.scannotation.AnnotationDB;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.core.CameraLayer;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.FileItem;
import ch.kuramo.javie.core.Folder;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaItem;
import ch.kuramo.javie.core.MediaItemLayer;
import ch.kuramo.javie.core.NullLayer;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.SolidColorItem;
import ch.kuramo.javie.core.TextLayer;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.annotations.ProjectElement;
import ch.kuramo.javie.core.internal.CameraLayerImpl;
import ch.kuramo.javie.core.internal.CompositionItemImpl;
import ch.kuramo.javie.core.internal.FolderImpl;
import ch.kuramo.javie.core.internal.LayerCompositionImpl;
import ch.kuramo.javie.core.internal.MediaFileItem;
import ch.kuramo.javie.core.internal.MediaItemLayerImpl;
import ch.kuramo.javie.core.internal.NullLayerImpl;
import ch.kuramo.javie.core.internal.ProjectImpl;
import ch.kuramo.javie.core.internal.SolidColorItemImpl;
import ch.kuramo.javie.core.internal.TextLayerImpl;
import ch.kuramo.javie.core.services.ProjectElementFactory;

import com.google.inject.Inject;
import com.google.inject.Injector;

public class ProjectElementFactoryImpl implements ProjectElementFactory {

	private final Map<String, Class<?>> _registry = Util.newMap();

	@Inject
	private Injector _injector;


	public void searchClasses(AnnotationDB db, ClassLoader cl) {
		Set<String> classNames = db.getAnnotationIndex().get(ProjectElement.class.getName());
		if (classNames == null) {
			return;
		}

		ModifiedClassLoader modifiedClassLoader = new ModifiedClassLoader();
		Map<String, String> tmpMap = Util.newMap();

		ClassPool pool = new ClassPool();
		pool.appendClassPath(new LoaderClassPath(cl));

		for (String className : classNames) {
			CtClass cc;
			try {
				cc = pool.get(className);
			} catch (NotFoundException e) {
				// 引数の AnnotationDB と ClassLoader の対応が正しければ必ず見つかるはず。
				throw new IllegalArgumentException(e);
			}

			for (Object anno : cc.getAvailableAnnotations()) {
				if (anno instanceof ProjectElement) {
					String type = ((ProjectElement) anno).value();
					if (type.equals("")) {
						type = className;
					}
					modifyClass(cc, type);
					modifiedClassLoader.add(cc);
					tmpMap.put(type, className);
					break;
				}
			}
		}

		modifiedClassLoader.loadAll(cl);

		try {
			for (Map.Entry<String, String> e : tmpMap.entrySet()) {
				_registry.put(e.getKey(), Class.forName(e.getValue(), true, cl));
			}
		} catch (ClassNotFoundException ex) {
			// 上のloadAllでロードできているので、この例外は発生しない。
			throw new Error(ex);
		}
	}

	private void modifyClass(CtClass cc, String type) {
		try {
			String src = String.format(
					"public String getTYPE() { return \"%s\"; }",
					type.replaceAll("\"", "\\\\\""));
			cc.addMethod(CtNewMethod.make(src, cc));
		} catch (CannotCompileException e) {
			throw new JavieRuntimeException(e);
		}
	}

	@SuppressWarnings("unchecked")
	public <T> Class<T> getClass(String type) {
		return (Class<T>) _registry.get(type);
	}

	public Project newProject() {
		ProjectImpl project = _injector.getInstance(ProjectImpl.class);
		project.initialize();
		return project;
	}

	public Folder newFolder() {
		FolderImpl folder = _injector.getInstance(FolderImpl.class);
		folder.initialize();
		return folder;
	}

	public FileItem newFileItem(File file) {
		MediaFileItem item = _injector.getInstance(MediaFileItem.class);
		item.initialize(file);
		if (item.getMediaInput() != null) {
			return item;
		}

		// 他にFileItemの実装を作ったら、ここに入れる。

		return null;
	}

	public SolidColorItem newSolidColorItem(Color color, Size2i size) {
		SolidColorItemImpl item = _injector.getInstance(SolidColorItemImpl.class);
		item.initialize(color, size);
		return item;
	}

	public CompositionItem newCompositionItem(Composition composition) {
		CompositionItemImpl item = _injector.getInstance(CompositionItemImpl.class);
		item.initialize(composition);
		return item;
	}

	public LayerComposition newLayerComposition(
			ColorMode colorMode, Size2i size, Time frameDuration, Time duration) {

		LayerCompositionImpl layerComp = _injector.getInstance(LayerCompositionImpl.class);
		layerComp.initialize(colorMode, size, frameDuration, duration);
		return layerComp;
	}

	public CameraLayer newCameraLayer() {
		CameraLayerImpl cameraLayer = _injector.getInstance(CameraLayerImpl.class);
		cameraLayer.initialize();
		return cameraLayer;
	}

	public MediaItemLayer newMediaItemLayer(MediaItem mediaItem) {
		MediaInput input = mediaItem.getMediaInput();
		if (input == null) {
			// TODO 例外投げる？
			return null;
		}

		MediaItemLayerImpl layer = _injector.getInstance(MediaItemLayerImpl.class);
		layer.initialize(mediaItem);
		return layer;
	}

	public NullLayer newNullLayer() {
		NullLayerImpl nullLayer = _injector.getInstance(NullLayerImpl.class);
		nullLayer.initialize();
		return nullLayer;
	}

	public TextLayer newTextLayer() {
		TextLayerImpl textLayer = _injector.getInstance(TextLayerImpl.class);
		textLayer.initialize();
		return textLayer;
	}

}
