/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.shaders;

import java.util.Arrays;
import java.util.List;

import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.GLGlobal;

import com.google.inject.Inject;

public class AccumulationShaders {

	@ShaderSource
	public final String[] accumulation;


	private String[] createAccumulationSource(int maxUnits) {
		List<String> source = Util.newList();

		/********************
		uniform sampler2DRect texSrc0;
		...
		uniform sampler2DRect texSrcN;		// N = maxUnits - 1
		
		vec2 texCoord = gl_TexCoord[0].st;
		********************/

		for (int i = 0; i < maxUnits; ++i) {
			source.add(String.format("uniform sampler2DRect texSrc%d;", i));
		}

		source.add("vec2 texCoord = gl_TexCoord[0].st;");

		/********************
		vec4 getSrc(int i) {
			return
				(i == 0) ? texture2DRect(texSrc0, texCoord) :
				...
				(i == N) ? texture2DRect(texSrcN, texCoord) :
				vec4(0.0);
		}
		********************/

		source.add("vec4 getSrc(int i) {");
		source.add("	return");
		for (int i = 0; i < maxUnits; ++i) {
			source.add(String.format("		(i == %1$d) ? texture2DRect(texSrc%1$d, texCoord) :", i));
		}
		source.add("		vec4(0.0);");
		source.add("}");


		source.addAll(Arrays.asList(new String[] {
				String.format("uniform float weights[%d];", maxUnits),
				"uniform int nUnits;",
				"",
				"void main(void)",
				"{",
				"	vec4 accum = vec4(0.0);",
				"	for (int i = 0; i < nUnits; ++i) {",
				"		accum += weights[i]*getSrc(i);",
				"	}",
				"	gl_FragColor = accum;",
				"}"
		}));

		return source.toArray(new String[source.size()]);
	}

	@Inject
	public AccumulationShaders(GLGlobal glGlobal) {
		accumulation = createAccumulationSource(glGlobal.getMaxTextureImageUnits());
	}

}
