/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.distort;

import javax.media.opengl.GL2;

import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.effects.VideoEffectUtil;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.PolarCoordinates", category=Effect.DISTORT)
public class PolarCoordinates {

	@ShaderSource
	public static final String[] RECT_TO_POLAR = {
		"uniform sampler2DRect texture;",
		"uniform vec2 center;",
		"uniform float radians_per_px;",
		"uniform float interp;",
		"",
		"void main(void)",
		"{",
		"	float sRect = gl_TexCoord[0].s;",
		"	float tRect = gl_TexCoord[0].t;",
		"",
		"	float sPolar = atan(sRect-center.s, tRect-center.t) / radians_per_px + center.s;",
		"	float tPolar = distance(gl_TexCoord[0].st, center) * 2.0;",
		"",
		"	float s = interp*sPolar + (1.0-interp)*sRect;",
		"	float t = interp*tPolar + (1.0-interp)*tRect;",
		"	gl_FragColor = texture2DRect(texture, vec2(s, t));",
		"}"
	};

	@ShaderSource
	public static final String[] POLAR_TO_RECT = {
		"uniform sampler2DRect texture;",
		"uniform vec2 center;",
		"uniform float radians_per_px;",
		"uniform float maxRadius_div_h;",
		"uniform float interp;",
		"",
		"void main(void)",
		"{",
		"	float sPolar = gl_TexCoord[0].s;",
		"	float tPolar = gl_TexCoord[0].t;",
		"",
		"	float theta = (sPolar - center.s) * radians_per_px;",
		"	float sRect = maxRadius_div_h*tPolar*cos(theta) + center.s;",
		"	float tRect = maxRadius_div_h*tPolar*sin(theta) + center.t;",
		"",
		"	float s = interp*sRect + (1.0-interp)*sPolar;",
		"	float t = interp*tRect + (1.0-interp)*tPolar;",
		"	gl_FragColor = texture2DRect(texture, vec2(s, t));",
		"}"
	};


	public enum TypeOfConversion { RECT_TO_POLAR, POLAR_TO_RECT }


	private final IVideoEffectContext context;

	private final IShaderProgram rectToPolarProgram;

	private final IShaderProgram polarToRectProgram;

	@Property(min="0", max="100")
	private IAnimatableDouble interpolation;

	@Property
	private IAnimatableEnum<TypeOfConversion> typeOfConversion;


	@Inject
	public PolarCoordinates(IVideoEffectContext context, IShaderRegistry shaders) {
		this.context = context;
		rectToPolarProgram = shaders.getProgram(PolarCoordinates.class, "RECT_TO_POLAR");
		polarToRectProgram = shaders.getProgram(PolarCoordinates.class, "POLAR_TO_RECT");
	}

	public IVideoBuffer doVideoEffect() {
		final double interp = context.value(this.interpolation)/100;
		if (interp == 0) {
			return null;
		}

		final TypeOfConversion type = context.value(this.typeOfConversion);
		final IShaderProgram program = (type == TypeOfConversion.RECT_TO_POLAR)
				? rectToPolarProgram : polarToRectProgram;

		IVideoBuffer input = null;
		IVideoBuffer buffer = null;
		try {
			input = context.doPreviousEffect();
			VideoBounds bounds = input.getBounds();
			if (bounds.isEmpty()) {
				IVideoBuffer output = input;
				input = null;
				return output;
			}

			final int w = bounds.width;
			final int h = bounds.height;

			buffer = context.createVideoBuffer(bounds);

			final GL2 gl = context.getGL().getGL2();

			VideoEffectUtil.ortho2D(gl, context.getGLU(), w, h);

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, buffer.getTexture(), 0);
			gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, input.getTexture());

			program.useProgram(new Runnable() {
				public void run() {
					gl.glUniform1i(program.getUniformLocation("texture"), 0);
					gl.glUniform2f(program.getUniformLocation("center"), w/2f, h/2f);
					gl.glUniform1f(program.getUniformLocation("radians_per_px"), (float)Math.toRadians(360.0/w));
					if (type == TypeOfConversion.POLAR_TO_RECT) {
						gl.glUniform1f(program.getUniformLocation("maxRadius_div_h"), (float)(Math.sqrt(w*w+h*h)*0.5/h));
					}
					gl.glUniform1f(program.getUniformLocation("interp"), (float)interp);

					gl.glBegin(GL2.GL_QUADS);
					gl.glTexCoord2f(0, 0);
					gl.glVertex2f(0, 0);
					gl.glTexCoord2f(w, 0);
					gl.glVertex2f(w, 0);
					gl.glTexCoord2f(w, h);
					gl.glVertex2f(w, h);
					gl.glTexCoord2f(0, h);
					gl.glVertex2f(0, h);
					gl.glEnd();
				}
			});

			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

			IVideoBuffer output = buffer;
			buffer = null;
			return output;

		} finally {
			if (input != null) input.dispose();
			if (buffer != null) buffer.dispose();
		}
	}

}
