/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.transition;

import java.nio.FloatBuffer;
import java.util.Arrays;

import javax.media.opengl.GL2;
import javax.media.opengl.glu.GLU;

import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.effects.VideoEffectUtil;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.LinearWipe", category=Effect.TRANSITION)
public class LinearWipe {

	private final IVideoEffectContext context;

	private final IArrayPools arrayPools;

	@Property(min="0", max="100")
	private IAnimatableDouble transitionCompletion;

	@Property("90")
	private IAnimatableDouble wipeAngle;

	@Property(min="0", max="32000")
	private IAnimatableDouble feather;


	@Inject
	public LinearWipe(IVideoEffectContext context, IArrayPools arrayPools) {
		this.context = context;
		this.arrayPools = arrayPools;
	}

	public IVideoBuffer doVideoEffect() {
		double completion = context.value(transitionCompletion)/100;
		if (completion == 0) {
			return null;
		} else if (completion == 1) {
			// TODO context.doPreviousEffect()をしなくてもVideoBoundsを取得できる方法が必要。
			//      context.getPreviousBounds()をここで呼ぶとIllegalStateExceptionが発生する。
			IVideoBuffer vb = context.doPreviousEffect();
			VideoEffectUtil.clearTexture(vb, context.getGL().getGL2());
			return vb;
		}

		double wipeAngle = context.value(this.wipeAngle) % 360;
		if (wipeAngle < 0) wipeAngle += 360;

		double feather = context.value(this.feather);
		feather = context.getRenderResolution().scale(feather);


		IVideoBuffer input = context.doPreviousEffect();
		VideoBounds bounds = input.getBounds();
		if (bounds.isEmpty()) {
			return input;
		}

		int w = bounds.width;
		int h = bounds.height;

		double theta = Math.toRadians(wipeAngle);
		double wipeLen = h*Math.abs(Math.cos(theta))+w*Math.abs(Math.sin(theta));
		double wipeWidth = w*Math.abs(Math.cos(theta))+h*Math.abs(Math.sin(theta));

		GL2 gl = context.getGL().getGL2();
		GLU glu = context.getGLU();

		VideoEffectUtil.ortho2D(gl, glu, w, h);

		gl.glMatrixMode(GL2.GL_MODELVIEW);
		gl.glTranslatef(w/2f, h/2f, 0);
		gl.glRotatef((float)wipeAngle, 0, 0, 1);
		gl.glScalef((float)(wipeWidth/w), (float)(wipeLen/h), 1);
		gl.glTranslatef(-w/2f, -h/2f, 0);

		int[] wipeTex = new int[1];
		gl.glGenTextures(1, wipeTex, 0);
		try {
			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, input.getTexture(), 0);
			gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

			gl.glActiveTexture(GL2.GL_TEXTURE0);
			gl.glBindTexture(GL2.GL_TEXTURE_1D, wipeTex[0]);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_MIN_FILTER, GL2.GL_LINEAR);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_MAG_FILTER, GL2.GL_LINEAR);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_WRAP_S, GL2.GL_CLAMP_TO_EDGE);
			gl.glTexParameteri(GL2.GL_TEXTURE_1D, GL2.GL_TEXTURE_WRAP_T, GL2.GL_CLAMP_TO_EDGE);
			prepareWipeTexture((int)Math.round(wipeLen), (float)completion, (float)feather, gl);
			gl.glEnable(GL2.GL_TEXTURE_1D);

			gl.glEnable(GL2.GL_BLEND);
			gl.glBlendFuncSeparate(GL2.GL_ZERO, GL2.GL_SRC_ALPHA, GL2.GL_ZERO, GL2.GL_SRC_ALPHA);

			gl.glBegin(GL2.GL_QUADS);
			gl.glTexCoord1f(1);
			gl.glVertex2f(0, 0);
			gl.glTexCoord1f(1);
			gl.glVertex2f(w, 0);
			gl.glTexCoord1f(0);
			gl.glVertex2f(w, h);
			gl.glTexCoord1f(0);
			gl.glVertex2f(0, h);
			gl.glEnd();

			gl.glDisable(GL2.GL_BLEND);
			gl.glDisable(GL2.GL_TEXTURE_1D);
			gl.glBindTexture(GL2.GL_TEXTURE_1D, 0);

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);
		} finally {
			gl.glDeleteTextures(1, wipeTex, 0);
		}

		return input;
	}

	private void prepareWipeTexture(int wipeLen, float completion, float feather, GL2 gl) {
		int intFeather = (int) Math.ceil(feather);
		int tmpDataLen = wipeLen + 4*intFeather;

		float completionPoint = intFeather + (wipeLen+2*intFeather) * completion;
		int completionIndex = (int) completionPoint;

		IArray<float[]> tmpArray = arrayPools.getFloatArray(tmpDataLen);
		IArray<float[]> wipeArray = null;
		IArray<float[]> kernelArray = null;

		try {
			float[] tmpData = tmpArray.getArray();
			FloatBuffer buffer;

			if (completionIndex+intFeather == tmpDataLen) {
				// ArrayPoolsから取得した配列はゼロクリアされていないのでクリアする必要がある。
				Arrays.fill(tmpData, 0, tmpDataLen, 0f);
				buffer = FloatBuffer.wrap(tmpData, 0, tmpDataLen);

			} else if (tmpDataLen == wipeLen) {
				Arrays.fill(tmpData, 0, completionIndex, 0f);
				tmpData[completionIndex] = 1 - (completionPoint - completionIndex);
				Arrays.fill(tmpData, completionIndex+1, tmpDataLen, 1f);
				buffer = FloatBuffer.wrap(tmpData, 0, tmpDataLen);

			} else {
				int kernelLen = intFeather*2+1;
				float[] kernel = (kernelArray = arrayPools.getFloatArray(kernelLen)).getArray();

				float sigma = feather / 2.5f;
				float sigmaSquare = sigma * sigma;
				float ksum = 0;
				for (int i = 1; i <= intFeather; ++i) {
					ksum += 2 * (kernel[intFeather-i] = (float) Math.exp(-i * i / (2 * sigmaSquare)));
				}
				kernel[intFeather] = 1 / (++ksum);
				for (int i = 1; i <= intFeather; ++i) {
					kernel[intFeather+i] = (kernel[intFeather-i] /= ksum);
				}

				float fractionAtCompletionIndex = 1 - (completionPoint - completionIndex);

				Arrays.fill(tmpData, 0, completionIndex-intFeather, 0f);
				tmpData[completionIndex-intFeather] = fractionAtCompletionIndex*kernel[0];

				float sum = 0;
				for (int k = 1; k < kernelLen; ++k) {
					sum += kernel[k];
					tmpData[completionIndex-intFeather+k] = sum + fractionAtCompletionIndex*kernel[k-1];
				}

				Arrays.fill(tmpData, completionIndex+intFeather+1, tmpDataLen, 1f);

				float[] wipeData = (wipeArray = arrayPools.getFloatArray(wipeLen)).getArray();
				System.arraycopy(tmpData, 2*intFeather, wipeData, 0, wipeLen);
				buffer = FloatBuffer.wrap(wipeData, 0, wipeLen);
			}

			switch (context.getColorMode()) {
				case RGBA8:
					gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, GL2.GL_ALPHA8, wipeLen, 0, GL2.GL_ALPHA, GL2.GL_FLOAT, buffer);
					break;

				case RGBA16:
					gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, GL2.GL_ALPHA16, wipeLen, 0, GL2.GL_ALPHA, GL2.GL_FLOAT, buffer);
					break;

				case RGBA16_FLOAT:
					gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, GL2.GL_ALPHA16F, wipeLen, 0, GL2.GL_ALPHA, GL2.GL_FLOAT, buffer);
					break;

				case RGBA32_FLOAT:
					gl.glTexImage1D(GL2.GL_TEXTURE_1D, 0, GL2.GL_ALPHA32F, wipeLen, 0, GL2.GL_ALPHA, GL2.GL_FLOAT, buffer);
					break;

				default:
					throw new RuntimeException("unknown ColorMode: " + context.getColorMode());
			}
		} finally {
			tmpArray.release();
			if (wipeArray != null) wipeArray.release();
			if (kernelArray != null) kernelArray.release();
		}
	}

}
