/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.api;

public final class Time implements Comparable<Time> {

	public final long timeValue;

	public final int timeScale;

	private final double second;

	private final int hashCode;


	public Time(long timeValue, int timeScale) {
		super();
		this.timeValue = timeValue;
		this.timeScale = timeScale;

		second = (double) timeValue / timeScale;
		hashCode = (int) (timeValue * 1000 / timeScale);
	}

	public static Time fromFrameNumber(long frameNumber, Time frameDuration) {
		return new Time(frameNumber * frameDuration.timeValue, frameDuration.timeScale);
	}

	public long toFrameNumber(Time frameDuration) {
		if (this.timeScale == frameDuration.timeScale) {
			return (long) Math.floor((double) this.timeValue / frameDuration.timeValue);
		} else {
			return (long) Math.floor(this.second / frameDuration.second);
		}
	}

	public double toSecond() {
		return second;
	}

	public boolean before(Time o) {
		return (compareTo(o) < 0);
	}

	public boolean after(Time o) {
		return (compareTo(o) > 0);
	}

	public int compareTo(Time o) {
		if (this.timeScale == o.timeScale) {
			long diff = this.timeValue - o.timeValue;
			return (diff > 0) ? 1 : (diff < 0) ? -1 : 0;
		} else {
			double diff = this.second - o.second;
			return (diff > 0) ? 1 : (diff < 0) ? -1 : 0;
		}
	}

	public Time add(Time addend) {
		if (addend.timeScale == timeScale) {
			return new Time(timeValue + addend.timeValue, timeScale);
		} else {
			return new Time((long) Math.floor((this.second + addend.second)*timeScale), timeScale);
		}
	}

	public Time subtract(Time subtrahend) {
		if (subtrahend.timeScale == timeScale) {
			return new Time(timeValue - subtrahend.timeValue, timeScale);
		} else {
			return new Time((long) Math.floor((this.second - subtrahend.second)*timeScale), timeScale);
		}
	}

	@Override
	public int hashCode() {
		return hashCode;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;

		return (compareTo((Time) obj) == 0);
	}

	@Override
	public String toString() {
		return String.valueOf(second);
	}

}
