/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.actions;

import java.io.File;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchWindow;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.CommandIds;
import ch.kuramo.javie.app.UIUtil;
import ch.kuramo.javie.app.project.NewImageSequenceOperation;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.views.ProjectView;
import ch.kuramo.javie.app.widgets.FrameRateComboUtil;
import ch.kuramo.javie.app.widgets.GridBuilder;
import ch.kuramo.javie.core.Folder;
import ch.kuramo.javie.core.FrameDuration;

public class NewImageSequenceAction extends Action {

	private final IWorkbenchWindow window;

	public NewImageSequenceAction(IWorkbenchWindow window) {
		super("イメージシーケンス...");
		this.window = window;

		setId(CommandIds.NEW_IMAGE_SEQUENCE);
		setActionDefinitionId(CommandIds.NEW_IMAGE_SEQUENCE);
		//setImageDescriptor(Activator.getImageDescriptor("/icons/new_image_sequence.png"));
	}

	public void run() {
		ProjectManager pm = ProjectManager.forWorkbenchWindow(window);
		if (pm == null) {
			return;
		}

		File file = showFileDialog();
		if (file == null) {
			return;
		}

		Time frameDuration = showFrameDurationDialog();
		if (frameDuration == null) {
			return;
		}

		ProjectView projectView = UIUtil.showView(window.getActivePage(), ProjectView.ID);

		Folder parent = projectView.getItemTreeInsertionPoint();
		pm.postOperation(new NewImageSequenceOperation(pm, parent, file, frameDuration));
	}

	private File showFileDialog() {
		String[] filterNames = new String[] {
				"サポートされているすべてのファイル",
				"BMP (*.bmp)", "PNG (*.png)", "GIF (*.gif)", "JPEG (*.jpg,*.jpeg)", //"TIFF (*.tif,*.tiff)",
				"すべてのファイル (*)"
		};
		String[] filterExtensions = new String[] {
				"*.bmp;*.png;*.gif;*.jpg;*.jpeg", //;*.tif;*.tiff",
				"*.bmp", "*.png", "*.gif", "*.jpg;*.jpeg", //"*.tif;*.tiff",
				"*"
		};

		String platform = SWT.getPlatform();
		if (platform.equals("win32") || platform.equals("wpf")) {
			filterNames[filterNames.length-1] = "すべてのファイル (*.*)";
			filterExtensions[filterExtensions.length-1] = "*.*";
		}

		FileDialog dialog = new FileDialog(window.getShell(), SWT.OPEN | SWT.SHEET);
		dialog.setFilterNames(filterNames);
		dialog.setFilterExtensions(filterExtensions);

		if (dialog.open() == null) {
			return null;
		}

		return new File(dialog.getFilterPath(), dialog.getFileName());
	}

	private Time showFrameDurationDialog() {
		final String[] fpsStr = new String[1];

		Dialog dialog = new Dialog(window.getShell()) {
			private Combo fpsCombo;

			protected Point getInitialSize() {
				return new Point(300, 120);
			}

			protected void configureShell(Shell newShell) {
				super.configureShell(newShell);
				newShell.setText("イメージシーケンスの読み込み");
			}

			protected Control createDialogArea(Composite parent) {
				Composite composite = (Composite) super.createDialogArea(parent);

				GridBuilder gb = new GridBuilder(composite, 16, true);

							gb.hSpan(4).hAlign(SWT.RIGHT).hGrab().label(SWT.NULL, "フレームレート:");
				fpsCombo =	gb.hSpan(3).hAlign(SWT.FILL).hGrab().combo(SWT.NULL, 0,
									FrameRateComboUtil.getComboItems(),
									FrameRateComboUtil.toComboItem(FrameDuration.FPS_29_97));
							gb.hSpan(9).hGrab().label(SWT.NULL, "フレーム／秒");

				fpsCombo.addModifyListener(new ModifyListener() {
					public void modifyText(ModifyEvent e) { updateOkButton(); }
				});
				fpsCombo.addSelectionListener(new SelectionListener() {
					public void widgetSelected(SelectionEvent e) { updateOkButton(); }
					public void widgetDefaultSelected(SelectionEvent e) { updateOkButton(); }
				});

				Composite grid = gb.getComposite();
				grid.setTabList(gb.getTabList());

				return composite;
			}

			private void updateOkButton() {
				Button okButton = getButton(IDialogConstants.OK_ID);
				okButton.setEnabled(isValidFps());
			}

			private boolean isValidFps() {
				String fpsStr = fpsCombo.getText().trim();
				double fps;
				try {
					fps = Double.parseDouble(fpsStr);
				} catch (NumberFormatException e) {
					fps = 0;
				}
				return (fps >= 1);
			}

			protected void okPressed() {
				if (isValidFps()) {
					fpsStr[0] = fpsCombo.getText().trim();
					super.okPressed();
				}
			}
		};

		if (dialog.open() == IDialogConstants.OK_ID) {
			return FrameRateComboUtil.toFrameDuration(fpsStr[0]);
		} else {
			return null;
		}
	}

}
