/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.actions;

import java.io.File;
import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.ui.IWorkbenchWindow;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.app.CommandIds;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.output.JavaImageSequenceOutput;
import ch.kuramo.javie.core.output.ProgressMonitor;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class SequenceOutputAction extends Action {

	private static File defaultFolder;


	private final LayerCompositionView view;

	@Inject
	private ProjectEncoder encoder;

	@Inject
	private ProjectDecoder decoder;


	public SequenceOutputAction(LayerCompositionView view) {
		super("イメージシーケンス...");
		InjectorHolder.getInjector().injectMembers(this);

		this.view = view;

		setId(CommandIds.SEQUENCE_OUTPUT);
		setActionDefinitionId(CommandIds.SEQUENCE_OUTPUT);
		//setImageDescriptor(Activator.getImageDescriptor("/icons/sequence_output.png"));
	}

	public void run() {
		ProjectManager pm = ProjectManager.forWorkbenchWindow(getWindow());
		if (pm == null) {
			return;
		}

		if (defaultFolder == null) {
			File file = pm.getFile();
			if (file != null) {
				defaultFolder = file.getParentFile();
			}
		}

		CompositionItem compItem = view.getCompositionItem();
		File file = showSaveDialog(defaultFolder, compItem.getName());
		if (file != null) {
			defaultFolder = file.getParentFile();
			doOutput(pm.getProject(), compItem.getId(), file);
		}
	}

	private IWorkbenchWindow getWindow() {
		return view.getSite().getWorkbenchWindow();
	}

	private File showSaveDialog(File folder, String name) {
		String[] filterNames = new String[] { "All Files (*)" };
		String[] filterExtensions = new String[] { "*" };

		String platform = SWT.getPlatform();
		if (platform.equals("win32") || platform.equals("wpf")) {
			filterNames = new String[] { "All Files (*.*)" };
			filterExtensions = new String[] { "*.*" };
		}

		FileDialog dialog = new FileDialog(getWindow().getShell(), SWT.SAVE | SWT.SHEET);
		dialog.setFilterNames(filterNames);
		dialog.setFilterExtensions(filterExtensions);
		dialog.setFilterPath(folder != null ? folder.getAbsolutePath() : null);
		dialog.setFileName(name);
		dialog.setOverwrite(false);		// This is OK. ここで指定するのはファイル名のプレフィックスなので、上書きの警告を出しても無意味。

		String path = dialog.open();
		return (path != null) ? new File(path) : null;
	}

	private void doOutput(Project project, String compItemId, File file) {
		Project copy = null;
		try {
			copy = decoder.decodeElement(encoder.encodeElement(project), Project.class);
			copy.afterDecode();

			ProgressMonitorDialog dialog = new ProgressMonitorDialog(getWindow().getShell());
			dialog.create();
			dialog.getShell().setText("書き出し");
			dialog.run(true, true, new OutputWithProgress((CompositionItem) copy.getItem(compItemId), file));

		} catch (ProjectDecodeException e) {
			throw new JavieRuntimeException(e);
		} catch (InvocationTargetException e) {
			throw new JavieRuntimeException(e);
		} catch (InterruptedException e) {
			// ユーザーがキャンセルした場合
		} finally {
			if (copy != null) {
				// TODO Project#dispose メソッドを作る。
				for (Item i : copy.getItems()) {
					i.dispose();
				}
			}
		}
	}

	private class OutputWithProgress implements IRunnableWithProgress {

		private final CompositionItem compItem;

		private final File file;


		private OutputWithProgress(CompositionItem compItem, File file) {
			this.compItem = compItem;
			this.file = file;
		}

		public void run(final IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
			JavaImageSequenceOutput seqOut = InjectorHolder.getInjector().getInstance(JavaImageSequenceOutput.class);
			seqOut.doOutput(compItem.getComposition(), AudioMode.STEREO_48KHZ_INT16, file,
					new ProgressMonitor() {
						private int totalWork;
						private int currentWork;

						public void beginTask(int totalWork) {
							monitor.beginTask(String.format("書き出し: %s", compItem.getName()), totalWork);
							this.totalWork = totalWork;
						}

						public void currentWork(int currentWork) {
							if (currentWork > this.currentWork) {
								monitor.worked(currentWork - this.currentWork);
								monitor.subTask(String.format("フレーム: %d/%d", currentWork, totalWork));
								this.currentWork = currentWork;
							}
						}

						public void done() {
							monitor.done();
						}

						public boolean isCanceled() {
							return monitor.isCanceled();
						}
					});

			if (monitor.isCanceled()) {
				throw new InterruptedException();
			}
		}

	}

}
