/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.player;

import org.eclipse.core.runtime.ListenerList;
import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Scale;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.Activator;
import ch.kuramo.javie.app.IntervalTask;
import ch.kuramo.javie.app.project.ProjectEvent;
import ch.kuramo.javie.app.project.ProjectEventHub;
import ch.kuramo.javie.app.project.ProjectListener;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaItem;
import ch.kuramo.javie.core.Project;

public class MediaPlayer extends Composite implements PlayerThreadListener, PlayerLinkListener, ProjectListener {

	private static final Logger _logger = LoggerFactory.getLogger(MediaPlayer.class);


	private static final Image IMG_PLAYER_PLAY;
	private static final Image IMG_PLAYER_PAUSE;
	private static final Image IMG_PLAYER_HEAD;
	private static final Image IMG_PLAYER_PREV;
	private static final Image IMG_PLAYER_NEXT;
	private static final Image IMG_PLAYER_LAST;

	static {
		ImageRegistry reg = Activator.getDefault().getImageRegistry();
		IMG_PLAYER_PLAY = reg.get(Activator.IMG_PLAYER_PLAY);
		IMG_PLAYER_PAUSE = reg.get(Activator.IMG_PLAYER_PAUSE);
		IMG_PLAYER_HEAD = reg.get(Activator.IMG_PLAYER_HEAD);
		IMG_PLAYER_PREV = reg.get(Activator.IMG_PLAYER_PREV);
		IMG_PLAYER_NEXT = reg.get(Activator.IMG_PLAYER_NEXT);
		IMG_PLAYER_LAST = reg.get(Activator.IMG_PLAYER_LAST);
	}


	private Scale _scale;

	private Button _playButton;

	private MediaItem _mediaItem;

	private TimeKeeper _timeKeeper;

	private AudioThread _audioThread;

	private VideoCanvas _videoCanvas;

	private Time _duration;

	private Time _frameDuration;

	private volatile boolean _hidden = true;

	private volatile boolean _dirty;


	public MediaPlayer(Composite parent, int style) {
		super(parent, style);
		setLayout(new FormLayout());

		createControls();
	}

	private void createControls() {
		_scale = new Scale(this, SWT.HORIZONTAL);

		Button headButton = new Button(this, SWT.PUSH | SWT.FLAT);
		headButton.setImage(IMG_PLAYER_HEAD);

		Button prevButton = new Button(this, SWT.PUSH | SWT.FLAT);
		prevButton.setImage(IMG_PLAYER_PREV);

		_playButton = new Button(this, SWT.PUSH | SWT.FLAT);
		_playButton.setImage(IMG_PLAYER_PLAY);

		Button nextButton = new Button(this, SWT.PUSH | SWT.FLAT);
		nextButton.setImage(IMG_PLAYER_NEXT);

		Button lastButton = new Button(this, SWT.PUSH | SWT.FLAT);
		lastButton.setImage(IMG_PLAYER_LAST);


		FormData data = new FormData();
		data.left = new FormAttachment(0, 5);
		data.right = new FormAttachment(100, -5);
		data.top = new FormAttachment(_playButton, -30, SWT.TOP);
		data.bottom = new FormAttachment(_playButton, 0, SWT.TOP);
		_scale.setLayoutData(data);

		data = new FormData();
		data.left = new FormAttachment(_scale, 0, SWT.CENTER);
		data.right = new FormAttachment(_scale, 0, SWT.CENTER);
		data.top = new FormAttachment(100, -30);
		data.bottom = new FormAttachment(100, -5);
		_playButton.setLayoutData(data);

		data = new FormData();
		data.right = new FormAttachment(_playButton, 0, SWT.LEFT);
		data.top = new FormAttachment(_playButton, 0, SWT.TOP);
		data.bottom = new FormAttachment(_playButton, 0, SWT.BOTTOM);
		prevButton.setLayoutData(data);

		data = new FormData();
		data.right = new FormAttachment(prevButton, 0, SWT.LEFT);
		data.top = new FormAttachment(_playButton, 0, SWT.TOP);
		data.bottom = new FormAttachment(_playButton, 0, SWT.BOTTOM);
		headButton.setLayoutData(data);

		data = new FormData();
		data.left = new FormAttachment(_playButton, 0, SWT.RIGHT);
		data.top = new FormAttachment(_playButton, 0, SWT.TOP);
		data.bottom = new FormAttachment(_playButton, 0, SWT.BOTTOM);
		nextButton.setLayoutData(data);

		data = new FormData();
		data.left = new FormAttachment(nextButton, 0, SWT.RIGHT);
		data.top = new FormAttachment(_playButton, 0, SWT.TOP);
		data.bottom = new FormAttachment(_playButton, 0, SWT.BOTTOM);
		lastButton.setLayoutData(data);


		_playButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				if (_timeKeeper != null) {
					if (_timeKeeper.isPaused()) {
						resume();
					} else {
						pause();
					}
				}
			}
		});

		headButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				setFrameNumber(0);
			}
		});

		prevButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				setFrameNumber(_scale.getSelection() - 1);
			}
		});

		nextButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				setFrameNumber(_scale.getSelection() + 1);
			}
		});

		lastButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				setFrameNumber(_scale.getMaximum());
			}
		});
	}

	public void close() {
		if (_audioThread != null) {
			_audioThread.finish();
		}
		if (_videoCanvas != null) {
			_videoCanvas.finish();
		}
		if (_timeKeeper != null) {
			_timeKeeper.pause();
		}

		ProjectEventHub.getHub().removeProjectListener(this);
	}

	public void setMediaItem(MediaItem mediaItem) {
		if (_mediaItem != null) {
			throw new IllegalStateException("MediaItem is already set");
		}
		MediaInput input = mediaItem.getMediaInput();
		if (input == null) {
			throw new IllegalArgumentException("no MediaInput is available");
		}

		_mediaItem = mediaItem;

		if (input.isAudioAvailable()) {
			try {
				_audioThread = new AudioThread(_mediaItem);
			} catch (Exception e) {
				_logger.warn("error creating AudioThread", e);
			}
		}

		_timeKeeper = (_audioThread != null)
				? _audioThread.getTimeKeeper()
				: TimeKeeper.fromSystemTime();

		if (input.isVideoAvailable()) {
			_videoCanvas = new VideoCanvas(this, _mediaItem, _timeKeeper);

			FormData data = new FormData();
			data.left = new FormAttachment(0, 0);
			data.right = new FormAttachment(100, 0);
			data.top = new FormAttachment(0, 0);
			data.bottom = new FormAttachment(_scale, 0, SWT.TOP);

			Control control = _videoCanvas.getControl();
			control.setLayoutData(data);

			layout(new Control[] { control });

			_videoCanvas.addPlayerThreadListener(this);

		} else if (_audioThread != null) {
			_audioThread.addPlayerThreadListener(this);
		}

		_duration = input.getDuration();

		if (_duration != null) {
			if (input.isVideoAvailable()) {
				_frameDuration = input.getVideoFrameDuration();
			} else {
				// TODO これだとデュレーションが12時間25分以上になると問題が発生するはず。
				_frameDuration = new Time(1, 48000);
			}

			int nFrames = (int) _duration.toFrameNumber(_frameDuration);
			_scale.setMaximum(nFrames - 1);
			_scale.setIncrement(1);
			_scale.setPageIncrement(nFrames / 10);
			_scale.addSelectionListener(new SelectionAdapter() {
				public void widgetSelected(SelectionEvent e) {
					setFrameNumber(_scale.getSelection());
				}
			});
		}

		if (_mediaItem instanceof CompositionItem) {
			ProjectEventHub.getHub().addProjectListener(this);
		}

		if (_videoCanvas != null) {
			_videoCanvas.start();
		}
		if (_audioThread != null) {
			_audioThread.start();
		}
	}

	private void setFrameNumber(int frameNumber) {
		if (_timeKeeper == null || _frameDuration == null) {
			return;
		}

		frameNumber = Math.max(0, Math.min(_scale.getMaximum(), frameNumber));
		_scale.setSelection(frameNumber);

		Time time = Time.fromFrameNumber(frameNumber, _frameDuration);
		_timeKeeper.setTime(time);
		if (_audioThread != null && !_timeKeeper.isPaused()) {
			_audioThread.setTime(time);
		}
		if (_videoCanvas != null && _timeKeeper.isPaused()) {
			_videoCanvas.forceRender();
		}
	}

	private void updateScale() {
		if (!_scale.isDisposed() && _frameDuration != null) {
			Time time = _timeKeeper.getTime();
			_scale.setSelection((int) time.toFrameNumber(_frameDuration));
		}
	}

	/* threadRenderイベントを受け取った時、毎回Scaleを更新すると負荷が高くなるので、100ms間隔で更新する */
	private final IntervalTask _scaleUpdateTask = new IntervalTask(getDisplay(), 100) {
		public void run() {
			updateScale();
		}
	};

	public void threadRender(PlayerThreadEvent event) {
		_scaleUpdateTask.schedule();
		notifyPlayerTime(event.time);
	}

	public void threadEndOfDuration(PlayerThreadEvent event) {
		getDisplay().asyncExec(new Runnable() {
			public void run() {
				pause();
			}
		});
	}

	private void resume() {
		_timeKeeper.resume(null);

		if (_audioThread != null) {
			_audioThread.setTime(_timeKeeper.getTime());
			_audioThread.play(true);
		}

		if (_videoCanvas != null) {
			_videoCanvas.play(true);
		}

		_playButton.setImage(IMG_PLAYER_PAUSE);
	}

	private void pause() {
		if (_audioThread != null) {
			_audioThread.play(false);
		}

		if (_videoCanvas != null) {
			_videoCanvas.play(false);
		}

		_timeKeeper.pause();

		// TimeKeeperの時刻が最終フレームの時刻を超えていることがある。その場合は最終フレームの時刻に設定しなおす。
		if (_duration != null) {
			Time maxTime = _duration.subtract(_frameDuration);
			if (_timeKeeper.getTime().after(maxTime)) {
				_timeKeeper.setTime(maxTime);
			}
		}

		// TimeKeeperをpauseした時刻よりも過去のフレームで表示が停止していることがあるので、
		// pauseした時刻のフレームを確実に表示するために一度forceRenderする。
		if (_videoCanvas != null) {
			_videoCanvas.forceRender();
		}

		// Display#asyncExec経由でここに来る場合があるのでdisposeされていないかチェックが必要。
		// (再生終了直前にビューを閉じた場合に起こりうる)
		if (!_playButton.isDisposed()) {
			_playButton.setImage(IMG_PLAYER_PLAY);
		}
	}

	public boolean hasVideo() {
		return (_videoCanvas != null);
	}

	public boolean hasAudio() {
		return (_audioThread != null);
	}

	public Resolution getVideoResolution() {
		return (_videoCanvas != null) ? _videoCanvas.getResolution() : null;
	}

	public void setVideoResolution(Resolution resolution) {
		if (_videoCanvas != null) {
			_videoCanvas.setResolution(resolution);
		}
	}

	public double getVideoZoom() {
		return (_videoCanvas != null) ? _videoCanvas.getZoom() : 0;
	}

	public void setVideoZoom(double zoom) {
		if (_videoCanvas != null) {
			_videoCanvas.setZoom(zoom);
		}
	}

	public boolean isAudioMuted() {
		return (_audioThread != null) ? _audioThread.isMuted() : false;
	}

	public void setAudioMuted(boolean mute) {
		if (_audioThread != null) {
			_audioThread.setMuted(mute);
		}
	}

	public boolean isShowInfo() {
		return (_videoCanvas != null) ? _videoCanvas.isShowInfo() : false;
	}

	public void setShowInfo(boolean showInfo) {
		if (_videoCanvas != null) {
			_videoCanvas.setShowInfo(showInfo);
		}
	}

	public void setHidden(boolean hidden) {
		if (_hidden != hidden) {
			_hidden = hidden;

			if (hidden && !_timeKeeper.isPaused()) {
				pause();
			} else if (!hidden && _dirty) {
				_dirty = false;
				if (_videoCanvas != null) {
					_videoCanvas.forceRender();
				}
			}
		}
	}


	private final ListenerList _playerLinkListeners = new ListenerList();

	public void addPlayerLinkListener(PlayerLinkListener listener) {
		_playerLinkListeners.add(listener);
	}

	public void removePlayerLinkListener(PlayerLinkListener listener) {
		_playerLinkListeners.remove(listener);
	}

	private void notifyPlayerTime(Time time) {
		PlayerLinkEvent event = new PlayerLinkEvent(this, time);
		for (Object l : _playerLinkListeners.getListeners()) {
			((PlayerLinkListener) l).handlePlayerLinkEvent(event);
		}
	}

	public void handlePlayerLinkEvent(PlayerLinkEvent event) {
		Time time = event.time;

		_timeKeeper.setTime(time);
		if (_hidden) {
			_dirty = true;
			notifyPlayerTime(time);
		} else {
			if (_audioThread != null && !_timeKeeper.isPaused()) {
				_audioThread.setTime(time);
			}
			if (_videoCanvas != null && _timeKeeper.isPaused()) {
				_videoCanvas.forceRender();
			}
		}
	}

	public void handleProjectEvent(ProjectEvent event) {
		switch (event.type) {
			case COMPOSITION_SETTINGS_CHANGE:
			case COMPOSITION_PROPERTY_CHANGE:
			case LAYER_PROPERTY_CHANGE:
			case LAYERS_ADD:
			case LAYERS_REMOVE:
			case LAYERS_REORDER:
			case LAYER_EXPRESSION_CHANGE:
			case EFFECT_PROPERTY_CHANGE:
			case EFFECTS_ADD:
			case EFFECTS_REMOVE:
			case EFFECT_EXPRESSION_CHANGE:
			case LAYER_TIMES_CHANGE:
			case KEYFRAMES_CHANGE:
			case LAYER_SLIP_EDIT:
				// これらのイベントはUIスレッド上で送信される。
				if (isSameProject(event.getProjectManager().getProject())) {
					_dirty = true;
				}
				break;

			case SHADOW_OPERATION_EXECUTION:
				// このイベントは ShadowOperationRunner のスレッド上で送信される。
				if (isSameProject(event.getProjectManager().getShadow()) && !_hidden && _dirty) {
					_dirty = false;
					if (_videoCanvas != null && _timeKeeper.isPaused()) {
						_videoCanvas.forceRender();
					}
				}
				break;
		}
	}

	private boolean isSameProject(Project project) {
		String compId = ((CompositionItem) _mediaItem).getComposition().getId();
		return (project != null && project.getComposition(compId) != null);
	}

}
