/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views.layercomp;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.PlatformUI;

import ch.kuramo.javie.app.ImageUtil;
import ch.kuramo.javie.app.project.ModifyLayerBlendModeOperation;
import ch.kuramo.javie.app.project.ModifyLayerParentOperation;
import ch.kuramo.javie.app.project.ModifyLayerTrackMatteOperation;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.project.RenameLayerOperation;
import ch.kuramo.javie.app.views.ComboBoxViewerCellEditor;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.app.widgets.WSWin32;
import ch.kuramo.javie.core.BlendMode;
import ch.kuramo.javie.core.CameraLayer;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.ItemLayer;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.MediaLayer;
import ch.kuramo.javie.core.NullLayer;
import ch.kuramo.javie.core.TextLayer;
import ch.kuramo.javie.core.TrackMatte;
import ch.kuramo.javie.core.TransformableLayer;
import ch.kuramo.javie.core.Util;

public class LayerElement extends Element {

	private static final boolean COCOA = SWT.getPlatform().equals("cocoa");


	public final Layer layer;

	private final LayerComposition composition;

	private final List<Element> children = Util.newList();

	private final Map<Class<? extends Element>, Element> childrenMap = Util.newMap();

	private final SwitchGroup showHideColumnSwitches = new SwitchGroup();

	private final SwitchGroup switchesColumnSwitches = new SwitchGroup();

	private TextCellEditor nameEditor;

	private ComboBoxViewerCellEditor modeEditor;

	private ComboBoxViewerCellEditor trackMatteEditor;

	private ComboBoxViewerCellEditor parentEditor;


	public LayerElement(TreeViewer viewer, Layer layer) {
		super(viewer);
		this.layer = layer;
		this.composition = (LayerComposition) viewer.getData(LayerCompositionView.LAYER_COMPOSITION);

		ProjectManager pm = (ProjectManager) viewer.getData(LayerCompositionView.PROJECT_MANAGER);

		if (LayerNature.isVideoNature(layer)) {
			showHideColumnSwitches.add(new VideoSwitch(pm, layer));
		} else {
			showHideColumnSwitches.addPlaceholder();
		}

		if (LayerNature.isAudioNature(layer)) {
			showHideColumnSwitches.add(new AudioSwitch(pm, layer));
		} else {
			showHideColumnSwitches.addPlaceholder();
		}


		switchesColumnSwitches.add(new ShySwitch(pm, layer));

		if (LayerNature.isCTCRNature(layer)) {
			switchesColumnSwitches.add(new CTCRSwitch(pm, layer));
		} else {
			switchesColumnSwitches.addPlaceholder();
		}

		if (layer instanceof EffectableLayer) {
			switchesColumnSwitches.add(new EffectsSwitch(pm, (EffectableLayer) layer));
		} else {
			switchesColumnSwitches.addPlaceholder();
		}

		if (LayerNature.isFrameBlendNature(layer)) {
			switchesColumnSwitches.add(new FrameBlendSwitch(pm, layer));
		} else {
			switchesColumnSwitches.addPlaceholder();
		}

		if (LayerNature.isMotionBlurNature(layer)) {
			switchesColumnSwitches.add(new MotionBlurSwitch(pm, layer));
		} else {
			switchesColumnSwitches.addPlaceholder();
		}

		if (layer instanceof TransformableLayer && LayerNature.isThreeDNature(layer)) {
			switchesColumnSwitches.add(new ThreeDSwitch(pm, (TransformableLayer) layer));
		} else {
			switchesColumnSwitches.addPlaceholder();
		}


		if (WSWin32.isXPThemed()) {
			showHideColumnSwitches.setMarginLeft(11);
			switchesColumnSwitches.setMarginLeft(11);
		}
	}

	public Image getColumnImage(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				if (layer instanceof ItemLayer) {
					return ImageUtil.getItemIcon(((ItemLayer) layer).getItem());
				}
				if (layer instanceof CameraLayer) {
					return ImageUtil.getCameraIcon();
				}
				if (layer instanceof NullLayer) {
					return ImageUtil.getNullIcon();
				}
				if (layer instanceof TextLayer) {
					return ImageUtil.getTextIcon();
				}

				// TODO その他のレイヤーのアイコン
				return PlatformUI.getWorkbench().getSharedImages().getImage(ISharedImages.IMG_OBJ_ELEMENT);

			default:
				return null;
		}
	}

	public String getColumnText(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL: {
				String name;
				if (layer instanceof ItemLayer) {
					String itemName = ((ItemLayer) layer).getItem().getName();
					if (isLayerNameMode()) {
						name = layer.getName();
						if (name.equals(itemName)) {
							name = String.format("[%s]", name);
						}
					} else {
						name = itemName;
					}
				} else {
					name = layer.getName();
				}
				return name;
			}

			case LayerCompositionView.MODE_COL:
				return (layer instanceof MediaLayer)
						? (COCOA ? "  " : "") + ((MediaLayer) layer).getBlendMode().label() : null;

			case LayerCompositionView.TRACKMATTE_COL:
				if (layer instanceof MediaLayer && getTrackMatteLayer() != null) {
					TrackMatte trackMatte = ((MediaLayer) layer).getTrackMatte();
					String label = trackMatte.label();
					return (COCOA ? "  " : "") + (trackMatte != TrackMatte.NONE ? label : " - ");
				}
				return null;

			case LayerCompositionView.PARENT_COL: {
				String parentName;
				Layer parent = composition.getParentLayer(layer);
				if (parent == null) {
					parentName = " - ";
				} else if (!(parent instanceof ItemLayer) || isLayerNameMode()) {
					parentName = parent.getName();
				} else {
					parentName = ((ItemLayer) parent).getItem().getName();
				}
				return (COCOA ? "  " : "") + parentName;
			}

			default:
				return null;
		}
	}

	private boolean isLayerNameMode() {
		return Boolean.TRUE.equals(viewer.getData(LayerCompositionView.LAYER_NAME_MODE));
	}

	private MediaLayer getTrackMatteLayer() {
		if (layer instanceof MediaLayer) {
			List<Layer> layers = composition.getLayers();
			int layerIndex = layers.indexOf(layer);
			if (layerIndex + 1 < layers.size()) {
				Layer matteLayer = layers.get(layerIndex + 1);
				if (matteLayer instanceof MediaLayer) {
					return (MediaLayer) matteLayer;
				}
			}
		}
		return null;
	}

	private <E extends Element> E getChildElement(Class<E> clazz) {
		@SuppressWarnings("unchecked")
		E element = (E) childrenMap.get(clazz);
		if (element == null) {
			try {
				element = clazz.getConstructor(getClass()).newInstance(this);
			} catch (Exception e) {
				throw new JavieRuntimeException(e);
			}
			childrenMap.put(clazz, element);
		}
		return element;
	}

	private void prepareChildren() {
		children.clear();

		if (LayerNature.isTimeRemapEnabled(layer)) {
			children.add(getChildElement(TimeRemapElement.class));
		}

		if (layer instanceof TextLayer) {
			children.add(getChildElement(TextElement.class));
		}

		if (layer instanceof EffectableLayer && !((EffectableLayer) layer).getEffects().isEmpty()) {
			children.add(getChildElement(EffectsElement.class));
		}

		if (layer instanceof TransformableLayer && LayerNature.isVideoNature(layer)) {
			children.add(getChildElement(TransformElement.class));

		} else if (layer instanceof CameraLayer) {
			children.add(getChildElement(CameraTransformElement.class));
			children.add(getChildElement(CameraOptionsElement.class));
		}

		if (layer instanceof MediaLayer && LayerNature.isAudioNature(layer)) {
			children.add(getChildElement(AudioElement.class));
		}

		childrenMap.values().retainAll(children);
	}

	@Override
	public boolean hasChildren() {
		prepareChildren();
		return (children.size() > 0);
	}

	@Override
	public Element[] getChildren() {
		// getChildren が呼ばれるより前に必ず hasChildren が呼ばれるようではあるが、
		// 念のため prepareChildren を実行しておく。
		prepareChildren();
		return children.toArray(new Element[children.size()]);
	}

	public void paintColumn(Event event) {
		switch (event.index) {
			case LayerCompositionView.SHOWHIDE_COL:
				showHideColumnSwitches.paint(event);
				break;

			case LayerCompositionView.VALUE_COL:
				switchesColumnSwitches.paint(event);
				break;

			case LayerCompositionView.TIMELINE_COL:
				TimelineManager tm = (TimelineManager) viewer.getData(LayerCompositionView.TIMELINE_MANAGER);
				tm.drawLayer(event, layer);
				break;

			default:
				super.paintColumn(event);
				break;
		}
	}

	public void updateCursor(MouseEvent event, int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.TIMELINE_COL:
				TimelineManager tm = (TimelineManager) viewer.getData(LayerCompositionView.TIMELINE_MANAGER);
				tm.updateCursor(event, layer);
				break;

			default:
				super.updateCursor(event, columnIndex);
				break;
		}
	}

	public void mouseDown(MouseEvent event, int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.SHOWHIDE_COL:
				showHideColumnSwitches.mouseDown(event);
				break;

			case LayerCompositionView.VALUE_COL:
				switchesColumnSwitches.mouseDown(event);
				break;

			case LayerCompositionView.TRACKMATTE_COL:
				if (!(layer instanceof MediaLayer) || getTrackMatteLayer() == null) break;
				// fall through
			case LayerCompositionView.MODE_COL:
				if (!(layer instanceof MediaLayer)) break;
				// fall through
			case LayerCompositionView.PARENT_COL:
				// WIN32では選択中の行でしかエディタが起動しないので、プログラム上からエディタを起動する。
				// (おそらくFULL_SELECTIONにすれば選択中でなくても(行のどこでもクリックすれば行が選択されるので、結果的に)エディタは起動するはずだが)
				// COCOAではプログラム上からエディタを起動しないと挙動がおかしくなる。
				if (event.button == 1) {
					viewer.editElement(this, columnIndex);
				}
				break;

			case LayerCompositionView.TIMELINE_COL:
				TimelineManager tm = (TimelineManager) viewer.getData(LayerCompositionView.TIMELINE_MANAGER);
				tm.mouseDown(event, layer);
				break;

			default:
				super.mouseDown(event, columnIndex);
				break;
		}
	}

	public boolean canEdit(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				return Boolean.TRUE.equals(viewer.getData(LayerCompositionView.EDIT_ELEMENT_NAME));

			case LayerCompositionView.MODE_COL:
				return (layer instanceof MediaLayer);

			case LayerCompositionView.TRACKMATTE_COL:
				return (layer instanceof MediaLayer && getTrackMatteLayer() != null);

			case LayerCompositionView.PARENT_COL:
				return true;

			default:
				return false;
		}
	}

	public CellEditor getCellEditor(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				if (nameEditor == null) {
					nameEditor = new TextCellEditor(viewer.getTree(), SWT.SINGLE | SWT.BORDER);
					Control control = nameEditor.getControl();
					control.setBackground(control.getDisplay().getSystemColor(SWT.COLOR_WHITE));
				}
				return nameEditor;

			case LayerCompositionView.MODE_COL:
				if (modeEditor == null) {
					modeEditor = new ComboBoxViewerCellEditor(viewer.getTree(), SWT.READ_ONLY);
					modeEditor.setContenProvider(ArrayContentProvider.getInstance());
					modeEditor.setLabelProvider(new BlendModeLabelProvider());
					modeEditor.setActivationStyle(ComboBoxViewerCellEditor.DROP_DOWN_ON_PROGRAMMATIC_ACTIVATION);

					Combo combo = (Combo) modeEditor.getControl();
					combo.setVisibleItemCount(35);
					combo.addSelectionListener(new SelectionAdapter() {
						public void widgetSelected(SelectionEvent e) {
							setCellEditorValue(LayerCompositionView.MODE_COL, modeEditor.getValue());
						}
					});
				}
				modeEditor.setInput(BlendMode.values());
				return modeEditor;

			case LayerCompositionView.TRACKMATTE_COL:
				if (trackMatteEditor == null) {
					trackMatteEditor = new ComboBoxViewerCellEditor(viewer.getTree(), SWT.READ_ONLY);
					trackMatteEditor.setContenProvider(ArrayContentProvider.getInstance());
					trackMatteEditor.setLabelProvider(new TrackMatteLabelProvider());
					trackMatteEditor.setActivationStyle(ComboBoxViewerCellEditor.DROP_DOWN_ON_PROGRAMMATIC_ACTIVATION);

					((Combo) trackMatteEditor.getControl()).addSelectionListener(new SelectionAdapter() {
						public void widgetSelected(SelectionEvent e) {
							setCellEditorValue(LayerCompositionView.TRACKMATTE_COL, trackMatteEditor.getValue());
						}
					});
				}
				trackMatteEditor.setInput(TrackMatte.values());
				return trackMatteEditor;

			case LayerCompositionView.PARENT_COL:
				if (parentEditor == null) {
					parentEditor = new ComboBoxViewerCellEditor(viewer.getTree(), SWT.READ_ONLY);
					parentEditor.setContenProvider(ArrayContentProvider.getInstance());
					parentEditor.setLabelProvider(new ParentLabelProvider());
					parentEditor.setActivationStyle(ComboBoxViewerCellEditor.DROP_DOWN_ON_PROGRAMMATIC_ACTIVATION);

					((Combo) parentEditor.getControl()).addSelectionListener(new SelectionAdapter() {
						public void widgetSelected(SelectionEvent e) {
							setCellEditorValue(LayerCompositionView.PARENT_COL, parentEditor.getValue());
						}
					});
				}
				parentEditor.setInput(getLayersForParentEditor());
				return parentEditor;

			default:
				return null;
		}
	}

	public Object getCellEditorValue(int columnIndex) {
		switch (columnIndex) {
			case LayerCompositionView.NAME_COL:
				return layer.getName();

			case LayerCompositionView.MODE_COL:
				return ((MediaLayer) layer).getBlendMode();

			case LayerCompositionView.TRACKMATTE_COL:
				return ((MediaLayer) layer).getTrackMatte();

			case LayerCompositionView.PARENT_COL:
				Layer parent = composition.getParentLayer(layer);
				return (parent != null) ? parent : NO_PARENT;

			default:
				return null;
		}
	}

	public void setCellEditorValue(int columnIndex, Object value) {
		ProjectManager pm = (ProjectManager) viewer.getData(LayerCompositionView.PROJECT_MANAGER);

		switch (columnIndex) {
			case LayerCompositionView.NAME_COL: {
				String newName = (String) value;
				if (newName.length() == 0) {
					if (layer instanceof ItemLayer) {
						newName = ((ItemLayer) layer).getItem().getName();
					} else {
						// TODO ItemLayer以外の場合、どうするのがいいか？　以下は暫定
						if (layer instanceof TextLayer) newName = "テキスト";
						else if (layer instanceof NullLayer) newName = "ヌル";
						else if (layer instanceof CameraLayer) newName = "カメラ";
					}
				}
				pm.postOperation(new RenameLayerOperation(pm, layer, newName));
				break;
			}

			case LayerCompositionView.MODE_COL:
				pm.postOperation(new ModifyLayerBlendModeOperation(pm, (MediaLayer) layer, (BlendMode) value));
				break;

			case LayerCompositionView.TRACKMATTE_COL:
				pm.postOperation(new ModifyLayerTrackMatteOperation(pm, (MediaLayer) layer, (TrackMatte) value));
				break;

			case LayerCompositionView.PARENT_COL:
				pm.postOperation(new ModifyLayerParentOperation(pm, layer, (value != NO_PARENT) ? (Layer) value : null));
				break;
		}
	}

	private static class BlendModeLabelProvider extends LabelProvider {
		public String getText(Object element) {
			return ((BlendMode) element).label();
		}
	}

	private class TrackMatteLabelProvider extends LabelProvider {
		public String getText(Object element) {
			TrackMatte trackMatte = (TrackMatte) element;
			if (trackMatte == TrackMatte.NONE) {
				return trackMatte.label();
			} else {
				String name;
				MediaLayer trackMatteLayer = getTrackMatteLayer();
				if (!(trackMatteLayer instanceof ItemLayer) || isLayerNameMode()) {
					name = trackMatteLayer.getName();
				} else {
					name = ((ItemLayer) trackMatteLayer).getItem().getName();
				}
				return String.format("%s - %s", trackMatte.label(), name);
			}
		}
	}

	private Object[] getLayersForParentEditor() {
		List<Object> layers = Util.newList();

		for (Layer l : composition.getLayers()) {
			Layer tmp = l;
			while (tmp != null && tmp != layer) {
				tmp = composition.getParentLayer(tmp);
			}
			if (tmp == null) {
				layers.add(l);
			}
		}

		layers.add(NO_PARENT);
		Collections.reverse(layers);

		return layers.toArray();
	}

	private static final Object NO_PARENT = new Object();

	private class ParentLabelProvider extends LabelProvider {
		public String getText(Object element) {
			if (element == NO_PARENT) {
				return "なし";
			} else if (!(element instanceof ItemLayer) || isLayerNameMode()) {
				return ((Layer) element).getName();
			} else {
				return ((ItemLayer) element).getItem().getName();
			}
		}
	}

}
