/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core;

import java.util.Collection;

import org.mozilla.javascript.Context;
import org.mozilla.javascript.NativeArray;
import org.mozilla.javascript.Scriptable;
import org.mozilla.javascript.Wrapper;

import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.core.exprelems.Vec2dProperty;
import ch.kuramo.javie.core.exprelems.Vec3dProperty;

public class AnimatableVec2d extends ArithmeticalAnimatableValue<Vec2d> implements IAnimatableVec2d {

	public AnimatableVec2d(Vec2d staticValue, Collection<Keyframe<Vec2d>> keyframes, String expression) {
		super(staticValue, keyframes, expression, Vec2d.NEGATIVE_INFINITY, Vec2d.POSITIVE_INFINITY);
	}

	public AnimatableVec2d(Vec2d defaultValue) {
		super(defaultValue, Vec2d.NEGATIVE_INFINITY, Vec2d.POSITIVE_INFINITY);
	}

	public AnimatableVec2d(Vec2d defaultValue, Vec2d minValue, Vec2d maxValue) {
		super(defaultValue, minValue, maxValue);
	}

	@Override
	public Vec2d clamp(Vec2d value) {
		return Vec2d.min(Vec2d.max(value, _minValue), _maxValue);
	}

	public Vec2d jsToJava(Object jsValue) {
		if (jsValue instanceof Wrapper) {
			Object javaObj = ((Wrapper) jsValue).unwrap();

			if (javaObj instanceof Vec2dProperty) {
				javaObj = ((Vec2dProperty) javaObj).getValue();
			} else if (javaObj instanceof Vec3dProperty) {
				javaObj = ((Vec3dProperty) javaObj).getValue();
			}

			if (javaObj instanceof Vec2d) {
				return (Vec2d) javaObj;

			} else if (javaObj instanceof Vec3d) {
				Vec3d v3d = (Vec3d) javaObj;
				return new Vec2d(v3d.x, v3d.y);

			} else if (javaObj instanceof Object[]) {
				Object[] value = (Object[]) javaObj;
				double[] array = new double[2];
				for (int i = 0, n = Math.min(2, value.length); i < n; ++i) {
					array[i] = (Double) Context.jsToJava(value[i], double.class);
				}
				return valueOf(array);
			}

		} else if (jsValue instanceof NativeArray) {
			// 配列表現を用いた場合 (例: [time*10, 200])

			double[] array = (double[]) Context.jsToJava(jsValue, double[].class);
			if (array.length < 2) {
				double[] tmp = array;
				array = new double[2];
				for (int i = 0; i < tmp.length; ++i) {
					array[i] = tmp[i];
				}
			}
			return valueOf(array);

		} else if (jsValue == null) {
			return Vec2d.ZERO;
		}

		double d = (Double) Context.jsToJava(jsValue, double.class);
		return new Vec2d(d, d);
	}

	@Override
	protected double[] toArray(Vec2d value) {
		return new double[] { value.x, value.y };
	}

	@Override
	protected Vec2d valueOf(double[] d) {
		return new Vec2d(d[0], d[1]);
	}

	public Scriptable createExpressionElement(final CoreContext context) {
		Vec2dProperty v2p = new Vec2dProperty() {
			public Vec2d getValue()				{ return value(context); }
			public Vec2d valueAtTime(double t)	{ return AnimatableVec2d.this.valueAtTime(t, context); }
		};
		return context.toNativeJavaObject(v2p, null);
	}

}
