/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.util.Collections;
import java.util.List;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.IObjectArray;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.core.AbstractTransformableLayer;
import ch.kuramo.javie.core.AnimatableDouble;
import ch.kuramo.javie.core.AnimatableString;
import ch.kuramo.javie.core.AnimatableVec2d;
import ch.kuramo.javie.core.AudioBuffer;
import ch.kuramo.javie.core.BlendMode;
import ch.kuramo.javie.core.Camera;
import ch.kuramo.javie.core.CollapseTransformation;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.CoreContext;
import ch.kuramo.javie.core.DepthBuffer;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.ExpressionScope;
import ch.kuramo.javie.core.FrameBlend;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaLayer;
import ch.kuramo.javie.core.MotionBlur;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.TrackMatte;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.VectorMediaInput;
import ch.kuramo.javie.core.VideoBuffer;
import ch.kuramo.javie.core.VideoLayerBuffer;
import ch.kuramo.javie.core.VideoLayerComposer;
import ch.kuramo.javie.core.VideoLayerRenderer;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.services.AudioEffectPipeline;
import ch.kuramo.javie.core.services.AudioRenderSupport;
import ch.kuramo.javie.core.services.GLGlobal;
import ch.kuramo.javie.core.services.RenderContext;
import ch.kuramo.javie.core.services.VideoEffectPipeline;
import ch.kuramo.javie.core.services.VideoRenderSupport;

import com.google.inject.Inject;

public abstract class AbstractMediaLayer extends AbstractTransformableLayer implements MediaLayer {

	private boolean _videoNature;

	private boolean _audioNature;

	private boolean _videoEnabled;

	private boolean _audioEnabled;

	private boolean _effectsEnabled = true;

	private List<Effect> _effects = Util.newList();

	private boolean _motionBlurEnabled = false;

	private TrackMatte _trackMatte = TrackMatte.NONE;

	private AnimatableVec2d _depthBase = new AnimatableVec2d(new Vec2d(0, 0));

	private AnimatableString _intersectionGroup = new AnimatableString("");

	private AnimatableDouble _opacity = new AnimatableDouble(100d, 0d, 100d);

	private BlendMode _blendMode = BlendMode.NORMAL;

	private AnimatableVec2d _audioLevels = new AnimatableVec2d(Vec2d.ZERO, new Vec2d(-48, -48), new Vec2d(48, 48));

	private AnimatableDouble _timeRemap = new AnimatableDouble(0.0);

	@Inject
	private RenderContext _context;

	@Inject
	private VideoRenderSupport _vrSupport;

	@Inject
	private VideoEffectPipeline _vePipeline;

	@Inject
	private GLGlobal _glGlobal;

	@Inject
	private AudioRenderSupport _arSupport;

	@Inject
	private AudioEffectPipeline _aePipeline;

	@Inject
	private IArrayPools _arrayPools;


	@Override
	public void initialize() {
		throw new UnsupportedOperationException("Use initialize(boolean, boolean) method instead.");
	}

	protected void initialize(boolean videoAvailable, boolean audioAvailable) {
		if (!videoAvailable && !audioAvailable) {
			throw new IllegalArgumentException();
		}

		super.initialize();

		if (videoAvailable) {
			_videoNature = true;
			_videoEnabled = true;
		}

		if (audioAvailable) {
			_audioNature = true;
			_audioEnabled = true;
		}
	}

	public boolean isVideoNature() {
		return _videoNature;
	}

	public void setVideoNature(boolean videoNature) {
		_videoNature = videoNature;
	}

	public boolean isAudioNature() {
		return _audioNature;
	}

	public void setAudioNature(boolean audioNature) {
		_audioNature = audioNature;
	}

	public boolean isVideoEnabled() {
		return _videoEnabled;
	}

	public void setVideoEnabled(boolean enabled) {
		_videoEnabled = enabled;
	}

	public boolean isAudioEnabled() {
		return _audioEnabled;
	}

	public void setAudioEnabled(boolean enabled) {
		_audioEnabled = enabled;
	}

	public boolean isEffectsEnabled() {
		return _effectsEnabled;
	}

	public void setEffectsEnabled(boolean effectsEnabled) {
		_effectsEnabled = effectsEnabled;
	}

	public List<Effect> getEffects() {
		return _effects;
	}

	public void setEffects(List<Effect> effects) {
		_effects = effects;
	}

	public boolean isMotionBlurEnabled() {
		return _motionBlurEnabled;
	}

	public void setMotionBlurEnabled(boolean motionBlurEnabled) {
		_motionBlurEnabled = motionBlurEnabled;
	}

	public TrackMatte getTrackMatte() {
		return _trackMatte;
	}

	public void setTrackMatte(TrackMatte trackMatte) {
		_trackMatte = trackMatte;
	}

	public AnimatableVec2d getDepthBase() {
		return _depthBase;
	}

	public void setDepthBase(AnimatableVec2d depthBase) {
		depthBase.copyConfigurationFrom(_depthBase);
		_depthBase = depthBase;
	}

	public AnimatableString getIntersectionGroup() {
		return _intersectionGroup;
	}

	public void setIntersectionGroup(AnimatableString intersectionGroup) {
		intersectionGroup.copyConfigurationFrom(_intersectionGroup);
		_intersectionGroup = intersectionGroup;
	}

	public AnimatableDouble getOpacity() {
		return _opacity;
	}

	public void setOpacity(AnimatableDouble opacity) {
		opacity.copyConfigurationFrom(_opacity);
		_opacity = opacity;
	}

	public BlendMode getBlendMode() {
		return _blendMode;
	}

	public void setBlendMode(BlendMode blendMode) {
		_blendMode = blendMode;
	}

	public AnimatableVec2d getAudioLevels() {
		return _audioLevels;
	}

	public void setAudioLevels(AnimatableVec2d audioLevels) {
		audioLevels.copyConfigurationFrom(_audioLevels);
		_audioLevels = audioLevels;
	}

	public AnimatableDouble getTimeRemap() {
		return _timeRemap;
	}

	public void setTimeRemap(AnimatableDouble timeRemap) {
		timeRemap.copyConfigurationFrom(_timeRemap);
		_timeRemap = timeRemap;
	}

	public boolean isVectorLayer() {
		return (getMediaInput() instanceof VectorMediaInput);
	}

	@Override
	public void prepareExpression(ExpressionScope scope) {
		super.prepareExpression(scope);

		scope.assignTo(_depthBase);
		scope.assignTo(_intersectionGroup);
		scope.assignTo(_opacity);
		scope.assignTo(_audioLevels);
		scope.assignTo(_timeRemap);

		for (Effect effect : _effects) {
			effect.prepareExpression(scope.clone());
		}
	}

	public void setupVideoRenderer(
			VideoLayerComposer composer, Camera camera, CollapseTransformation ct,
			MotionBlur motinBlur, boolean frameBlendEnabled,
			boolean withEffects, boolean withTransformation) {

		// トラックマット適用時やエフェクトからの参照時にこのチェックがあるとまずいので。同じチェックはコンポ側でもしている。
		//if (!LayerNature.isVideoEnabled(this)) {
		//	return;
		//}

		MediaInput input = getMediaInput();
		if (input == null || !input.isVideoAvailable()) {
			return;
		}

		if (isVectorLayer() && LayerNature.isCTCR(this)) {
			setupVideoRendererCR(composer, camera, ct, motinBlur, withEffects);
		} else {
			setupVideoRendererNormal(composer, camera, ct, motinBlur, frameBlendEnabled, withEffects, withTransformation);
		}
	}

	protected Time calcVideoMediaTime(MediaInput input) {
		Time time;

		if (LayerNature.isTimeRemapEnabled(this)) {
			double t0 = _timeRemap.value(_context);

			Time ctxTime = _context.getTime();
			double t1 = _timeRemap.valueAtTime(new Time(ctxTime.timeValue + 1, ctxTime.timeScale), _context);

			Time mediaFrameDuration = input.getVideoFrameDuration();
			int mediaTimeScale = mediaFrameDuration.timeScale;

			time = new Time(Math.round(t0 * mediaTimeScale), mediaTimeScale);

			// 変化量が負の時のときは１フレーム分ずらす。
			if (t0 > t1) {
				time = time.subtract(mediaFrameDuration);
			}

		} else {
			time = _context.getTime().subtract(getStartTime());

			double rate = getRate();
			Time mediaFrameDuration = input.getVideoFrameDuration();

			if (mediaFrameDuration != null) {
				// メディアのタイムスケールにしておかないと他の箇所の計算で誤差が発生することがある。
				int mediaTimeScale = mediaFrameDuration.timeScale;
				time = new Time(Math.round(time.toSecond() * rate * mediaTimeScale), mediaTimeScale);
			} else {
				time = new Time(Math.round(time.timeValue * rate), time.timeScale);
			}

			if (rate < 0.0) {
				Time mediaDuration = input.getDuration();
				if (mediaDuration != null) {
					time = time.add(mediaDuration);

					// 逆再生時は１フレーム分ずらす。
					time = time.subtract(mediaFrameDuration);

				} else {
					mediaDuration = getOutPoint().subtract(getStartTime());
					time = time.add(mediaDuration);
				}
			}
		}

		return time;
	}

	private void setupVideoRendererNormal(
			VideoLayerComposer composer, final Camera camera, final CollapseTransformation ct,
			final MotionBlur motionBlur, final boolean frameBlendEnabled,
			final boolean withEffects, final boolean withTransformation) {

		final MediaInput input = getMediaInput();
		final Resolution resolution = _context.getVideoResolution();

		final WrappedOperation<VideoBounds> inputBoundsOperation = new WrappedOperation<VideoBounds>() {
			public VideoBounds execute() {
				return resolution.scale(input.getVideoFrameBounds());
			}
		};

		final List<Effect> effects = (withEffects && _effectsEnabled) ? _effects : Collections.<Effect>emptyList();

		VideoBounds bounds = _vePipeline.getVideoBounds(effects, inputBoundsOperation);
		final Vec2d offset = new Vec2d(-bounds.x/resolution.scale, -bounds.y/resolution.scale);

		final double[] mvMatrix;
		final double[] prjMatrix;
		if (withTransformation) {
			mvMatrix = new double[16];
			prjMatrix = new double[16];
			createMatrix(camera, ct, offset, mvMatrix, prjMatrix);
		} else {
			mvMatrix = null;
			prjMatrix = null;
		}


		final Time time = _context.getTime();
		final Composition comp = _context.getComposition();

		VideoLayerRenderer r = new VideoLayerRenderer() {
			public VideoLayerBuffer render(boolean withDepthBuffer) {
				// コラップスしているときはここで時刻とコンポを再度設定する必要がある。
				_context.setTime(time);
				_context.setComposition(comp);

				final double opacity;
				if (ct == null) {
					opacity = _opacity.value(_context) / 100d;
				} else {
					opacity = _opacity.value(_context) / 100d * ct.getOpacity();
				}

				VideoBuffer effectedBuffer = null;
				VideoBuffer trnsfrmdBuffer = null;
				DepthBuffer depthBuffer = null;
				try {
					WrappedOperation<VideoBuffer> inputBufferOperation = new WrappedOperation<VideoBuffer>() {
						public VideoBuffer execute() {
							if (frameBlendEnabled) {
								FrameBlend frameBlend = LayerNature.getFrameBlend(AbstractMediaLayer.this);
								if (frameBlend != FrameBlend.NONE) {
									return blendFrames(frameBlend);
								}
							}
							return input.getVideoFrame(calcVideoMediaTime(input));
						}
					};

					effectedBuffer = _vePipeline.doVideoEffects(effects, inputBoundsOperation, inputBufferOperation);

					if (withTransformation) {
						if (motionBlur != null && LayerNature.isMotionBlurEnabled(AbstractMediaLayer.this)) {
							if (withDepthBuffer) {
								depthBuffer = _vrSupport.createDepthBuffer(camera.getViewportSize());
								_vrSupport.fillDepth(depthBuffer, mvMatrix, prjMatrix);
							}

							final VideoBuffer mblurSrcBuf = effectedBuffer;
							MotionBlurSampler sampler = new MotionBlurSampler() {
								public void sample(double[] mvMatrix, double[] prjMatrix, VideoBuffer buffer) {
									_vrSupport.transform(mblurSrcBuf, buffer, null, mvMatrix, prjMatrix);
								}
							};
							trnsfrmdBuffer = renderWithMotionBlur(camera, ct, motionBlur, offset, time, sampler);

						} else {
							trnsfrmdBuffer = _vrSupport.createVideoBuffer(_context.getColorMode(), camera.getViewportSize());
							trnsfrmdBuffer.allocateAsTexture();
							trnsfrmdBuffer.clear();

							if (withDepthBuffer) {
								depthBuffer = _vrSupport.createDepthBuffer(camera.getViewportSize());
								_vrSupport.transform(effectedBuffer, trnsfrmdBuffer, depthBuffer, mvMatrix, prjMatrix);
							} else {
								_vrSupport.transform(effectedBuffer, trnsfrmdBuffer, null, mvMatrix, prjMatrix);
							}
						}
					} else {
						if (withDepthBuffer) {
							throw new IllegalArgumentException("can't create DepthBuffer when withTransformation is false");
						}
						trnsfrmdBuffer = effectedBuffer;
						effectedBuffer = null;
					}

					VideoLayerBuffer vlb = new VideoLayerBufferImpl(trnsfrmdBuffer, depthBuffer,
							_blendMode, opacity, getId().hashCode() + getName().hashCode());
					trnsfrmdBuffer = null;
					depthBuffer = null;
					return vlb;

				} finally {
					if (depthBuffer != null) depthBuffer.dispose();
					if (trnsfrmdBuffer != null) trnsfrmdBuffer.dispose();
					if (effectedBuffer != null) effectedBuffer.dispose();
				}
			}
		};

		if (withTransformation && LayerNature.isThreeD(this)) {
			Vec2d depthBase2d = _depthBase.value(_context);
			Vec3d depthBase = new Vec3d(depthBase2d.x + offset.x, depthBase2d.y + offset.y);
			depthBase = resolution.scale(depthBase);

			Size2i vp = camera.getViewportSize();
			Vec3d[] vertices = new Vec3d[] {
				_vrSupport.project(depthBase, mvMatrix, prjMatrix, vp),
				_vrSupport.project(new Vec3d(bounds.x               , bounds.y                ), mvMatrix, prjMatrix, vp),
				_vrSupport.project(new Vec3d(bounds.x + bounds.width, bounds.y                ), mvMatrix, prjMatrix, vp),
				_vrSupport.project(new Vec3d(bounds.x + bounds.width, bounds.y + bounds.height), mvMatrix, prjMatrix, vp),
				_vrSupport.project(new Vec3d(bounds.x               , bounds.y + bounds.height), mvMatrix, prjMatrix, vp)
			};
			composer.add3D(r, vertices, _intersectionGroup.value(_context));

		} else {
			composer.add2D(r);
		}
	}

	private void setupVideoRendererCR(
			VideoLayerComposer composer, final Camera camera, final CollapseTransformation ct,
			final MotionBlur motionBlur, final boolean withEffects) {

		final VectorMediaInput input = (VectorMediaInput) getMediaInput();
		Resolution resolution = _context.getVideoResolution();

		VideoBounds bounds = resolution.scale(input.getVideoFrameBounds());

		final double[] mvMatrix = new double[16];
		final double[] prjMatrix = new double[16];
		createMatrix(camera, ct, null, mvMatrix, prjMatrix);


		final Time time = _context.getTime();
		final Composition comp = _context.getComposition();

		VideoLayerRenderer r = new VideoLayerRenderer() {
			public VideoLayerBuffer render(boolean withDepthBuffer) {
				// コラップスしているときはここで時刻とコンポを再度設定する必要がある。
				_context.setTime(time);
				_context.setComposition(comp);

				final double opacity;
				if (ct == null) {
					opacity = _opacity.value(_context) / 100d;
				} else {
					opacity = _opacity.value(_context) / 100d * ct.getOpacity();
				}

				VideoBuffer effectedBuffer = null;
				VideoBuffer resultBuffer = null;
				DepthBuffer depthBuffer = null;
				try {
					if (withDepthBuffer) {
						depthBuffer = _vrSupport.createDepthBuffer(camera.getViewportSize());
						_vrSupport.fillDepth(depthBuffer, mvMatrix, prjMatrix);
					}

					WrappedOperation<VideoBounds> inputBoundsOperation = new WrappedOperation<VideoBounds>() {
						public VideoBounds execute() {
							return new VideoBounds(camera.getViewportSize());
						}
					};

					WrappedOperation<VideoBuffer> inputBufferOperation = new WrappedOperation<VideoBuffer>() {
						public VideoBuffer execute() {
							VideoBuffer rasterizedBuffer = null;
							try {
								if (motionBlur != null && LayerNature.isMotionBlurEnabled(AbstractMediaLayer.this)) {
									MotionBlurSampler sampler = new MotionBlurSampler() {
										public void sample(double[] mvMatrix, double[] prjMatrix, VideoBuffer buffer) {
											input.rasterize(buffer, mvMatrix, prjMatrix, calcVideoMediaTime(input));
										}
									};
									rasterizedBuffer = renderWithMotionBlur(camera, ct, motionBlur, null, _context.getTime(), sampler);

								} else {
									rasterizedBuffer = _vrSupport.createVideoBuffer(_context.getColorMode(), camera.getViewportSize());
									rasterizedBuffer.allocateAsTexture();
									rasterizedBuffer.clear();

									input.rasterize(rasterizedBuffer, mvMatrix, prjMatrix, calcVideoMediaTime(input));
								}

								VideoBuffer result = rasterizedBuffer;
								rasterizedBuffer = null;
								return result;

							} finally {
								if (rasterizedBuffer != null) rasterizedBuffer.dispose();
							}
						}
					};

					if (withEffects) {
						effectedBuffer = _vePipeline.doVideoEffects(
								_effectsEnabled ? _effects : Collections.<Effect>emptyList(),
								inputBoundsOperation, inputBufferOperation);

						VideoBounds resultBounds = new VideoBounds(camera.getViewportSize());
						if (effectedBuffer.getBounds().equals(resultBounds)) {
							resultBuffer = effectedBuffer;
							effectedBuffer = null;
						} else {
							resultBuffer = _vrSupport.createVideoBuffer(_context.getColorMode(), resultBounds);
							resultBuffer.allocateAsTexture();
							resultBuffer.clear();
							_vrSupport.copy(effectedBuffer, resultBuffer);
						}
					} else {
						resultBuffer = _context.saveAndExecute(inputBufferOperation);
					}

					VideoLayerBuffer vlb = new VideoLayerBufferImpl(resultBuffer, depthBuffer,
							_blendMode, opacity, getId().hashCode() + getName().hashCode());
					resultBuffer = null;
					depthBuffer = null;
					return vlb;

				} finally {
					if (depthBuffer != null) depthBuffer.dispose();
					if (resultBuffer != null) resultBuffer.dispose();
					if (effectedBuffer != null) effectedBuffer.dispose();
				}
			}
		};

		if (LayerNature.isThreeD(this)) {
			Vec2d depthBase2d = _depthBase.value(_context);
			Vec3d depthBase = new Vec3d(depthBase2d.x, depthBase2d.y);
			depthBase = resolution.scale(depthBase);

			Size2i vp = camera.getViewportSize();
			Vec3d[] vertices = new Vec3d[] {
				_vrSupport.project(depthBase, mvMatrix, prjMatrix, vp),
				_vrSupport.project(new Vec3d(bounds.x               , bounds.y                ), mvMatrix, prjMatrix, vp),
				_vrSupport.project(new Vec3d(bounds.x + bounds.width, bounds.y                ), mvMatrix, prjMatrix, vp),
				_vrSupport.project(new Vec3d(bounds.x + bounds.width, bounds.y + bounds.height), mvMatrix, prjMatrix, vp),
				_vrSupport.project(new Vec3d(bounds.x               , bounds.y + bounds.height), mvMatrix, prjMatrix, vp)
			};
			composer.add3D(r, vertices, _intersectionGroup.value(_context));

		} else {
			composer.add2D(r);
		}
	}

	private void createMatrix(
			final Camera camera, final CollapseTransformation ct, final Vec2d offset,
			final double[] mvMatrix, final double[] prjMatrix) {

		WrappedOperation<Object> operation = new WrappedOperation<Object>() {
			public Object execute() {
				if (camera == null) {
					_vrSupport.resetMatrix();
				} else if (LayerNature.isThreeD(AbstractMediaLayer.this)) {
					_vrSupport.setMatrix(camera.getModelView3D(), camera.getProjection3D());
				} else {
					_vrSupport.setMatrix(camera.getModelView2D(), camera.getProjection2D());
				}

				LayerMatrixUtil.multModelViewMatrix(AbstractMediaLayer.this, offset, ct, _context, _vrSupport);

				_vrSupport.getMatrix(mvMatrix, prjMatrix);
				return null;
			}
		};

		_vrSupport.pushMatrixAndExecute(operation);
	}

	private VideoBuffer blendFrames(FrameBlend frameBlend) {
		if (frameBlend == FrameBlend.NONE) throw new IllegalArgumentException();
		if (frameBlend != FrameBlend.FRAME_MIX) throw new UnsupportedOperationException();

		MediaInput input = getMediaInput();
		Time mediaFrameDuration = input.getVideoFrameDuration();

		Time ctxTime = _context.getTime();
		Time mediaTime0 = calcVideoMediaTime(input);
		_context.setTime(ctxTime.add(_context.getVideoFrameDuration()));
		Time mediaTime1 = calcVideoMediaTime(input);
		_context.setTime(ctxTime);

		Time mediaTimeDiff = mediaTime1.subtract(mediaTime0);
		double fpsRatio = mediaTimeDiff.toSecond() / mediaFrameDuration.toSecond();
		if (fpsRatio == 1) {
			return input.getVideoFrame(ctxTime);
		}

		fpsRatio = Math.abs(fpsRatio);


		List<VideoBuffer> buffers = Util.newList();
		VideoBuffer buf32f = null;
		try {
			Resolution resolution = _context.getVideoResolution();
			VideoBounds bounds = resolution.scale(input.getVideoFrameBounds());

			buf32f = _vrSupport.createVideoBuffer(ColorMode.RGBA32_FLOAT, bounds);
			buf32f.allocateAsTexture();
			buf32f.clear();

			List<Double> weights = Util.newList();
			int maxUnits = _glGlobal.getMaxTextureImageUnits();

			if (fpsRatio > 1) {
				List<Integer> intWeights = Util.newList();
				int w = (int)Math.round(255/fpsRatio);
				intWeights.add(w);
				int rest = 255 - w;

				for (int i = 1; rest > 0; ++i) {
					double p = 1.17741*2*i/fpsRatio;
					double q = Math.exp(-p*p/2)/fpsRatio;
					w = Math.min(rest, 2*(int)Math.round(255*q))/2;
					if (w > 0) {
						intWeights.add(w);
						rest -= 2*w;
					} else if (rest > 0) {
						// 端数は中央のフレームに上乗せしてしまう。
						intWeights.set(0, intWeights.get(0)+rest);
						rest = 0;
					}
				}

				for (int n = intWeights.size(), i = -n+1; i < n; ++i) {
					Time t = mediaTime0.add(new Time(mediaFrameDuration.timeValue*i, mediaFrameDuration.timeScale));

					buffers.add(input.getVideoFrame(t));
					weights.add(intWeights.get(Math.abs(i))/255.0);

					if (buffers.size() >= maxUnits) {
						_vrSupport.accumulate(buffers, weights, buf32f);
						for (VideoBuffer vb : buffers) vb.dispose();
						buffers.clear();
						weights.clear();
					}
				}
			} else {
				long frame = mediaTime0.toFrameNumber(mediaFrameDuration);
				Time t0, t1;
				if (mediaTime0.timeValue >= 0) {
					t0 = Time.fromFrameNumber(frame, mediaFrameDuration);
					t1 = Time.fromFrameNumber(frame+1, mediaFrameDuration);
				} else {
					t0 = Time.fromFrameNumber(frame-1, mediaFrameDuration);
					t1 = Time.fromFrameNumber(frame, mediaFrameDuration);
				}
				long tv0 = mediaTime0.subtract(t0).timeValue;
				long tv1 = t1.subtract(mediaTime0).timeValue;

				buffers.add(input.getVideoFrame(t0));
				weights.add((double)tv1/(tv0+tv1));

				// maxUnitsが1の場合はあるのか？　無いのなら次のifブロックは不要。
				if (maxUnits == 1) {
					_vrSupport.accumulate(buffers, weights, buf32f);
					buffers.get(0).dispose();
					buffers.clear();
					weights.clear();
				}

				buffers.add(input.getVideoFrame(t1));
				weights.add((double)tv0/(tv0+tv1));
			}
			if (!buffers.isEmpty()) {
				_vrSupport.accumulate(buffers, weights, buf32f);
			}

			VideoBuffer resultBuffer;
			if (_context.getColorMode() == ColorMode.RGBA32_FLOAT) {
				resultBuffer = buf32f;
				buf32f = null;
			} else {
				resultBuffer = _vrSupport.createVideoBuffer(_context.getColorMode(), buf32f.getBounds());
				resultBuffer.allocateAsTexture();
				_vrSupport.copy(buf32f, resultBuffer);
			}
			return resultBuffer;
		} finally {
			for (VideoBuffer vb : buffers) vb.dispose();
			if (buf32f != null) buf32f.dispose();
		}
	}

	private interface MotionBlurSampler {
		void sample(double[] mvMatrix, double[] prjMatrix, VideoBuffer buffer);
	}

	private VideoBuffer renderWithMotionBlur(
			Camera camera, CollapseTransformation ct, MotionBlur motionBlur,
			Vec2d offset, Time time, MotionBlurSampler sampler) {

		List<VideoBuffer> buffers = Util.newList();
		VideoBuffer buf = null;
		VideoBuffer buf32f = null;
		try {
			buf32f = _vrSupport.createVideoBuffer(ColorMode.RGBA32_FLOAT, camera.getViewportSize());
			buf32f.allocateAsTexture();
			buf32f.clear();

			double shutterAngle = motionBlur.getShutterAngle();
			double shutterPhase = motionBlur.getShutterPhase();
			int samples = motionBlur.getSamples();
			Time frameDuration = _context.getVideoFrameDuration();
			int maxUnits = _glGlobal.getMaxTextureImageUnits();

			double[] mvMatrix = new double[16];
			double[] prjMatrix = new double[16];

			for (int i = 0; i < samples; ++i) {
				double angle = shutterPhase + shutterAngle*i/samples;
				Time t = time.add(new Time((long)(frameDuration.timeValue*angle/360), frameDuration.timeScale));
				_context.setTime(t);

				createMatrix(camera, ct, offset, mvMatrix, prjMatrix);

				buf = _vrSupport.createVideoBuffer(_context.getColorMode(), camera.getViewportSize());
				buf.allocateAsTexture();
				buf.clear();

				sampler.sample(mvMatrix, prjMatrix, buf);

				buffers.add(buf);
				buf = null;

				if (buffers.size() >= maxUnits) {
					_vrSupport.accumulate(buffers, 1.0/samples, buf32f);
					for (VideoBuffer vb : buffers) vb.dispose();
					buffers.clear();
				}
			}
			if (!buffers.isEmpty()) {
				_vrSupport.accumulate(buffers, 1.0/samples, buf32f);
			}

			VideoBuffer resultBuffer;
			if (_context.getColorMode() == ColorMode.RGBA32_FLOAT) {
				resultBuffer = buf32f;
				buf32f = null;
			} else {
				resultBuffer = _vrSupport.createVideoBuffer(_context.getColorMode(), buf32f.getBounds());
				resultBuffer.allocateAsTexture();
				_vrSupport.copy(buf32f, resultBuffer);
			}
			return resultBuffer;

		} finally {
			for (VideoBuffer vb : buffers) vb.dispose();
			if (buf != null) buf.dispose();
			if (buf32f != null) buf32f.dispose();
		}
	}

	private AudioBuffer getTimeRemappedAudioChunk(MediaInput input) {
		AudioBuffer ab1 = null;
		AudioBuffer ab2 = null;
		IObjectArray<Double> times = null;

		try {
			AudioMode audioMode = _context.getAudioMode();
			int sampleRate = audioMode.sampleRate;

			ab1 = _arSupport.createAudioBuffer();

			times = _arrayPools.getObjectArray(ab1.getFrameCount());
			_timeRemap.values(times, audioMode.sampleDuration, _context.getAudioAnimationRate(), _context);


			// 音声データを取得する範囲を求める。
			double min = Double.POSITIVE_INFINITY, max = Double.NEGATIVE_INFINITY;
			Object[] timeArray = times.getArray();
			for (int i = 0, n = times.getLength(); i < n; ++i) {
				min = Math.min(min, (Double) timeArray[i]);
				max = Math.max(max, (Double) timeArray[i]);
			}
			long timeValueMin = (long) Math.floor(min*sampleRate);
			long timeValueMax = (long) Math.ceil(max*sampleRate);


			// 上で求めた範囲の音声データをMediaInputから取得。
			_context.setAudioFrameCount((int)(timeValueMax - timeValueMin) + 1);

			ab2 = input.getAudioChunk(new Time(timeValueMin, sampleRate));
			if (ab2 == null) return null;


			// タイムリマップの計算値に従ってリサンプルする。
			Object data1 = ab1.getData();
			Object data2 = ab2.getData();
			int n1 = ab1.getFrameCount();

			switch (audioMode.dataType) {
				case SHORT: {
					short[] array1 = (short[]) data1, array2 = (short[]) data2;
					for (int i = 0; i < n1; ++i) {
						int j = (int)(Math.round((Double)timeArray[i]*sampleRate) - timeValueMin);
						array1[i*2  ] = array2[j*2  ];
						array1[i*2+1] = array2[j*2+1];
					}
					break;
				}
				case INT: {
					int[] array1 = (int[]) data1, array2 = (int[]) data2;
					for (int i = 0; i < n1; ++i) {
						int j = (int)(Math.round((Double)timeArray[i]*sampleRate) - timeValueMin);
						array1[i*2  ] = array2[j*2  ];
						array1[i*2+1] = array2[j*2+1];
					}
					break;
				}
				case FLOAT: {
					float[] array1 = (float[]) data1, array2 = (float[]) data2;
					for (int i = 0; i < n1; ++i) {
						int j = (int)(Math.round((Double)timeArray[i]*sampleRate) - timeValueMin);
						array1[i*2  ] = array2[j*2  ];
						array1[i*2+1] = array2[j*2+1];
					}
					break;
				}
				default:
					throw new UnsupportedOperationException(
							"unsupported AudioMode.DataType: " + audioMode.dataType);
			}


			AudioBuffer result = ab1;
			ab1 = null;
			return result;

		} finally {
			if (ab1 != null) ab1.dispose();
			if (ab2 != null) ab2.dispose();
			if (times != null) times.release();
		}
	}

	private AudioBuffer getRateChangedAudioChunk(MediaInput input, Time time, double rate) {
		AudioBuffer ab1 = null;
		AudioBuffer ab2 = null;

		try {
			AudioMode audioMode = _context.getAudioMode();
			int sampleRate = audioMode.sampleRate;

			ab1 = _arSupport.createAudioBuffer();

			long timeValueMin;
			long timeValueMax;
			double duration = input.getDuration().toSecond();

			if (rate >= 0) {
				timeValueMin = (long) Math.floor(time.toSecond() * sampleRate * rate);

				Time t = time.add(new Time(_context.getAudioFrameCount(), sampleRate));
				timeValueMax = (long) Math.ceil(t.toSecond() * sampleRate * rate);

			} else {
				timeValueMax =  (long) Math.ceil(duration * sampleRate + time.toSecond() * sampleRate * rate);

				Time t = time.add(new Time(_context.getAudioFrameCount(), sampleRate));
				timeValueMin = (long) Math.floor(duration * sampleRate + t.toSecond() * sampleRate * rate);
			}

			_context.setAudioFrameCount((int)(timeValueMax - timeValueMin) + 1);

			ab2 = input.getAudioChunk(new Time(timeValueMin, sampleRate));
			if (ab2 == null) return null;


			Object data1 = ab1.getData();
			Object data2 = ab2.getData();
			int n1 = ab1.getFrameCount();

			switch (audioMode.dataType) {
				case SHORT: {
					short[] array1 = (short[]) data1, array2 = (short[]) data2;
					for (int i = 0; i < n1; ++i) {
						int j = (rate >= 0) ? (int)(Math.round((time.toSecond() * sampleRate + i) * rate) - timeValueMin)
											: (int)(Math.round(duration * sampleRate + (time.toSecond() * sampleRate + i) * rate) - timeValueMin); 
						array1[i*2  ] = array2[j*2  ];
						array1[i*2+1] = array2[j*2+1];
					}
					break;
				}
				case INT: {
					int[] array1 = (int[]) data1, array2 = (int[]) data2;
					for (int i = 0; i < n1; ++i) {
						int j = (rate >= 0) ? (int)(Math.round((time.toSecond() * sampleRate + i) * rate) - timeValueMin)
											: (int)(Math.round(duration * sampleRate + (time.toSecond() * sampleRate + i) * rate) - timeValueMin); 
						array1[i*2  ] = array2[j*2  ];
						array1[i*2+1] = array2[j*2+1];
					}
					break;
				}
				case FLOAT: {
					float[] array1 = (float[]) data1, array2 = (float[]) data2;
					for (int i = 0; i < n1; ++i) {
						int j = (rate >= 0) ? (int)(Math.round((time.toSecond() * sampleRate + i) * rate) - timeValueMin)
											: (int)(Math.round(duration * sampleRate + (time.toSecond() * sampleRate + i) * rate) - timeValueMin); 
						array1[i*2  ] = array2[j*2  ];
						array1[i*2+1] = array2[j*2+1];
					}
					break;
				}
				default:
					throw new UnsupportedOperationException(
							"unsupported AudioMode.DataType: " + audioMode.dataType);
			}


			AudioBuffer result = ab1;
			ab1 = null;
			return result;

		} finally {
			if (ab1 != null) ab1.dispose();
			if (ab2 != null) ab2.dispose();
		}
	}

	public AudioBuffer renderAudioChunk(boolean withEffects, boolean withLevelsAndInOut) {
		// エフェクトからの参照時にこのチェックがあるとまずいので。同じチェックはコンポ側でもしている。
		//if (!LayerNature.isAudioEnabled(this)) {
		//	return null;
		//}

		final MediaInput input = getMediaInput();
		if (input == null || !input.isAudioAvailable()) {
			return null;
		}

		WrappedOperation<AudioBuffer> inputOperation = new WrappedOperation<AudioBuffer>() {
			public AudioBuffer execute() {
				if (LayerNature.isTimeRemapEnabled(AbstractMediaLayer.this)) {
					return getTimeRemappedAudioChunk(input);

				} else {
					Time time = _context.getTime().subtract(getStartTime());
					double rate = getRate();
					if (rate == 1.0) {
						return input.getAudioChunk(time);
					} else {
						return getRateChangedAudioChunk(input, time, rate);
					}
				}
			}
		};

		AudioBuffer ab = _aePipeline.doAudioEffects(
				(withEffects && _effectsEnabled) ? _effects : Collections.<Effect>emptyList(), inputOperation);

		if (withLevelsAndInOut) {
			if (_audioLevels.hasKeyframe() || _audioLevels.getExpression() != null
					|| !_audioLevels.getStaticValue().equals(Vec2d.ZERO)) {

				IObjectArray<Vec2d> audioLevels = _arrayPools.getObjectArray(ab.getFrameCount());

				_audioLevels.values(audioLevels, ab.getAudioMode().sampleDuration,
									_context.getAudioAnimationRate(), _context);
				_arSupport.amplify(ab, audioLevels);

				audioLevels.release();
			}

			// インポイントよりも手前とアウトポイント以降はゼロクリアする
			Time time = _context.getTime();
			Time sampleDuration = _context.getAudioMode().sampleDuration;

			int clearFrameCount = (int) getInPoint().subtract(time).toFrameNumber(sampleDuration);
			if (clearFrameCount > 0) {
				_arSupport.clear(ab, 0, clearFrameCount);
			}

			int clearFrameOffset = (int) getOutPoint().subtract(time).toFrameNumber(sampleDuration);
			clearFrameCount = _context.getAudioFrameCount() - clearFrameOffset;
			if (clearFrameCount > 0) {
				_arSupport.clear(ab, clearFrameOffset, clearFrameCount);
			}
		}

		return ab;
	}

	@Override
	public void afterDecode(Project p, LayerComposition c) throws ProjectDecodeException {
		super.afterDecode(p, c);

		for (Effect effect : _effects) {
			effect.afterDecode(p);
		}
	}

	public Object createExpressionElement(CoreContext context) {
		return new MediaLayerExpressionElement(context);
	}

	public class MediaLayerExpressionElement extends TransformableLayerExpressionElement {

		public MediaLayerExpressionElement(CoreContext context) {
			super(context);
		}

		public Object getDepthBase()			{ return elem(_depthBase); }
		public Object getIntersectionGroup()	{ return elem(_intersectionGroup); }
		public Object getOpacity()				{ return elem(_opacity); }
		public Object getAudioLevels()			{ return elem(_audioLevels); }
		public Object getTimeRemap()			{ return LayerNature.isTimeRemapEnabled(AbstractMediaLayer.this) ? elem(_timeRemap) : null; }

		public Object effect(int index) {
			Effect effect = _effects.get(index - 1);
			return context.getExpressionElement(effect);
		}

		public Object effect(String name) {
			for (Effect e : _effects) {
				if (e.getName().equals(name)) {
					return context.getExpressionElement(e);
				}
			}
			return null;
		}
	}

}
