/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import net.arnx.jsonic.JSONHint;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.core.AbstractItem;
import ch.kuramo.javie.core.AudioBuffer;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.VideoBuffer;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.annotations.ProjectElement;
import ch.kuramo.javie.core.services.RenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;

import com.google.inject.Inject;

@ProjectElement("composition")
public class CompositionItemImpl extends AbstractItem implements CompositionItem {

	private String _compositionId;

	private Composition _composition;

	private final CompositionInput _input = new CompositionInput();

	@Inject
	private RenderContext _context;

	@Inject
	private VideoRenderSupport _vrSupport;


	public CompositionItemImpl() {
		super();
	}

	public void initialize(Composition composition) {
		super.initialize();
		_compositionId = composition.getId();
		_composition = composition;
	}

	public void dispose() {
		_composition = null;
	}

	public String getCompositionId() {
		return _compositionId;
	}

	public void setCompositionId(String compositionId) {
		_compositionId = compositionId;
	}

	@JSONHint(ignore=true)
	public Composition getComposition() {
		return _composition;
	}

	@Override
	public void afterDecode(Project p) throws ProjectDecodeException {
		super.afterDecode(p);

		_composition = p.getComposition(_compositionId);
		if (_composition == null) {
			throw new ProjectDecodeException(
					"no such Composition found: id=" + _compositionId);
		}
	}

	@JSONHint(ignore=true)
	public MediaInput getMediaInput() {
		return _input;
	}

	@JSONHint(ignore=true)
	public String getKind() {
		return "コンポジション";
	}


	private class CompositionInput implements MediaInput {

		public void dispose() {
			// noting to do
		}

		public boolean isVideoAvailable() {
			// TODO コンポジションがビデオを持っている場合だけtrueを返すようにする。
			return true;
		}

		public boolean isAudioAvailable() {
			// TODO コンポジションがオーディオを持っている場合だけtrueを返すようにする。
			return true;
		}

		public Time getDuration() {
			return _composition.getDuration();
		}

		public Time getVideoFrameDuration() {
			return _composition.getFrameDuration();
		}

		public VideoBounds getVideoFrameBounds() {
			Size2i size = _composition.getSize();
			return new VideoBounds(size.width, size.height);  // TODO 毎回newするのは無駄
		}

		public VideoBuffer getVideoFrame(final Time mediaTime) {
			VideoBuffer vb = _context.saveAndExecute(new WrappedOperation<VideoBuffer>() {
				public VideoBuffer execute() {
					_context.setTime(mediaTime);
					return _composition.renderVideoFrame();
				}
			});
			_vrSupport.premultiply(vb);
			return vb;
		}

		public AudioBuffer getAudioChunk(final Time mediaTime) {
			return _context.saveAndExecute(new WrappedOperation<AudioBuffer>() {
				public AudioBuffer execute() {
					_context.setTime(mediaTime);
					return _composition.renderAudioChunk();
				}
			});
		}

	}

}
