/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.io.File;

import net.arnx.jsonic.JSONHint;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.core.AbstractItem;
import ch.kuramo.javie.core.ImageSequenceItem;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaInputPlaceholder;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.annotations.ProjectElement;
import ch.kuramo.javie.core.services.MediaFileInputFactory;

import com.google.inject.Inject;
import com.google.inject.Injector;

@ProjectElement("imageSequence")
public class ImageSequenceItemImpl extends AbstractItem implements ImageSequenceItem {

	private File _file;

	private Time _frameDuration;

	private VideoBounds _videoFrameBounds;

	private MediaInput _mediaInput;

	private final MediaFileInputFactory _mediaFileInputFactory;

	private final Injector _injector;

	@Inject
	public ImageSequenceItemImpl(MediaFileInputFactory mediaFileInputFactory, Injector injector) {
		super();
		_mediaFileInputFactory = mediaFileInputFactory;
		_injector = injector;
	}

	public void initialize(File file, Time frameDuration) {
		super.initialize();

		ImageSequenceInput input = _mediaFileInputFactory.get(ImageSequenceInput.class, getId(), file);

		if (input != null) {
			input.setVideoFrameDuration(frameDuration);
			_file = input.getFirstFile();
			_frameDuration = frameDuration;
			_videoFrameBounds = input.isVideoAvailable() ? input.getVideoFrameBounds() : null;
			_mediaInput = input;
			setName(_file.getName());
		}
	}

	@Override
	protected void finalize() throws Throwable {
		dispose();
		super.finalize();
	}

	public void dispose() {
		if (_mediaInput != null) {
			if (!(_mediaInput instanceof MediaInputPlaceholder)) {
				_mediaFileInputFactory.release(getId());
			}
			_mediaInput = null;
		}
	}

	public File getFile() {
		return _file;
	}

	public void setFile(File file) {
		_file = file;
	}

	public Time getFrameDuration() {
		return _frameDuration;
	}

	public void setFrameDuration(Time frameDuration) {
		_frameDuration = frameDuration;

		// TODO
		//if (_mediaInput instanceof ImageSequenceInput) {
		//	((ImageSequenceInput) _mediaInput).setVideoFrameDuration(frameDuration);
		//}
	}

	public VideoBounds getVideoFrameBounds() {
		return _videoFrameBounds;
	}

	public void setVideoFrameBounds(VideoBounds videoFrameBounds) {
		_videoFrameBounds = videoFrameBounds;
	}

	@JSONHint(ignore=true)
	public MediaInput getMediaInput() {
		return _mediaInput;
	}

	@Override
	public void afterDecode(Project p) throws ProjectDecodeException {
		super.afterDecode(p);

		ImageSequenceInput input = _mediaFileInputFactory.get(ImageSequenceInput.class, getId(), _file);

		if (input != null) {
			input.setVideoFrameDuration(_frameDuration);
			_file = input.getFirstFile();
			_videoFrameBounds = input.isVideoAvailable() ? input.getVideoFrameBounds() : null;
			_mediaInput = input;
		} else {
			MediaInputPlaceholderImpl placeholder = _injector.getInstance(MediaInputPlaceholderImpl.class);
			placeholder.setVideoFrameBounds(_videoFrameBounds);
			_mediaInput = placeholder;
		}
	}

	@JSONHint(ignore=true)
	public String getKind() {
		return "イメージシーケンス";
	}

}
