/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.util.List;
import java.util.ListIterator;

import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.core.AbstractComposition;
import ch.kuramo.javie.core.AudioBuffer;
import ch.kuramo.javie.core.Camera;
import ch.kuramo.javie.core.CameraLayer;
import ch.kuramo.javie.core.CollapseTransformation;
import ch.kuramo.javie.core.CoreContext;
import ch.kuramo.javie.core.ExpressionScope;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaLayer;
import ch.kuramo.javie.core.MotionBlur;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.TrackMatte;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.VideoBuffer;
import ch.kuramo.javie.core.VideoLayerBuffer;
import ch.kuramo.javie.core.VideoLayerComposer;
import ch.kuramo.javie.core.VideoLayerRenderer;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.annotations.ProjectElement;
import ch.kuramo.javie.core.services.AudioRenderSupport;
import ch.kuramo.javie.core.services.RenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;

import com.google.inject.Inject;

@ProjectElement("layerComposition")
public class LayerCompositionImpl extends AbstractComposition implements LayerComposition {

	private boolean _shyEnabled = false;

	private boolean _frameBlendEnabled = false;

	private boolean _motionBlurEnabled = false;

	private double _motionBlurShutterAngle = 180;

	private double _motionBlurShutterPhase = -90;

	private int _motionBlurSamples = 16;

	private final MotionBlur _motionBlur = new MotionBlur() {
		public double getShutterAngle()	{ return _motionBlurShutterAngle; }
		public double getShutterPhase()	{ return _motionBlurShutterPhase; }
		public int getSamples()			{ return _motionBlurSamples; }
	};

	private List<Layer> _layers = Util.newList();

	@Inject
	private RenderContext _context;

	@Inject
	private VideoRenderSupport _vrSupport;

	@Inject
	private AudioRenderSupport _arSupport;


	public LayerCompositionImpl() {
		super();
	}

	public boolean isShyEnabled() {
		return _shyEnabled;
	}

	public void setShyEnabled(boolean shyEnabled) {
		_shyEnabled = shyEnabled;
	}

	public boolean isFrameBlendEnabled() {
		return _frameBlendEnabled;
	}

	public void setFrameBlendEnabled(boolean frameBlendEnabled) {
		_frameBlendEnabled = frameBlendEnabled;
	}

	public boolean isMotionBlurEnabled() {
		return _motionBlurEnabled;
	}

	public void setMotionBlurEnabled(boolean motionBlurEnabled) {
		_motionBlurEnabled = motionBlurEnabled;
	}

	public double getMotionBlurShutterAngle() {
		return _motionBlurShutterAngle;
	}

	public void setMotionBlurShutterAngle(double motionBlurShutterAngle) {
		_motionBlurShutterAngle = motionBlurShutterAngle;
	}

	public double getMotionBlurShutterPhase() {
		return _motionBlurShutterPhase;
	}

	public void setMotionBlurShutterPhase(double motionBlurShutterPhase) {
		_motionBlurShutterPhase = motionBlurShutterPhase;
	}

	public int getMotionBlurSamples() {
		return _motionBlurSamples;
	}

	public void setMotionBlurSamples(int motionBlurSamples) {
		_motionBlurSamples = motionBlurSamples;
	}

	public List<Layer> getLayers() {
		return _layers;
	}

	public void setLayers(List<Layer> layers) {
		_layers = layers;
	}

	public Layer getLayer(String id) {
		for (Layer layer : _layers) {
			if (layer.getId().equals(id)) {
				return layer;
			}
		}
		return null;
	}

	public Layer getParentLayer(Layer layer) {
		String parentId = layer.getParentId();
		return (parentId != null) ? getLayer(parentId) : null;
	}

	public void setParentLayer(Layer layer, Layer parent) {
		// 引数のレイヤーと親がどちらもこのコンポジションに属しているかどうかのチェックをすべきか？
		layer.setParentId((parent != null) ? parent.getId() : null);
	}

	public void afterDecode(Project p) throws ProjectDecodeException {
		for (Layer layer : _layers) {
			layer.afterDecode(p, this);
		}
	}

	public void prepareExpression(ExpressionScope scope) {
		scope.putExpressionElement("thisComp", this);

		for (Layer layer : _layers) {
			layer.prepareExpression(scope.clone());
		}
	}

	public VideoBuffer renderVideoFrame() {
		_context.setComposition(this);

		ColorMode colorMode = getColorMode();
		_context.setColorMode(colorMode);

		Time time = _context.getTime();

		if (isFrameDurationPreserved()) {
			Time frameDuration = getFrameDuration();
			time = Time.fromFrameNumber(time.toFrameNumber(frameDuration), frameDuration);
			_context.setTime(time);
			_context.setVideoFrameDuration(frameDuration);
		}

		// 上のレイヤーから(リスト内では末尾から)順に有効なカメラレイヤーを探し、見つかればそれを使う。
		CameraLayer cameraLayer = null;
		for (ListIterator<Layer> it = _layers.listIterator(_layers.size()); it.hasPrevious(); ) {
			Layer layer = it.previous();
			if (layer instanceof CameraLayer && LayerNature.isVideoEnabled(layer)
					&& !layer.getInPoint().after(time) && layer.getOutPoint().after(time)) {
				cameraLayer = (CameraLayer) layer;
				break;
			}
		}

		Size2i renderSize = _context.getVideoResolution().scale(getSize());

		final CameraImpl camera = new CameraImpl(
				_context, _vrSupport, cameraLayer, getSize(), renderSize);

		final VideoLayerComposerImpl composer = new VideoLayerComposerImpl(
				_context, _vrSupport, colorMode, renderSize, camera.getPosition());

		final MotionBlur motionBlur = _motionBlurEnabled ? _motionBlur : null;

		for (int i = 0, n = _layers.size(); i < n; ++i) {
			Layer layer = _layers.get(i);

			if (!LayerNature.isVideoEnabled(layer)) {
				continue;
			}

			if (layer.getInPoint().after(time) || !layer.getOutPoint().after(time)) {
				continue;
			}

			if (layer instanceof MediaLayer) {
				final MediaLayer ml = (MediaLayer) layer;

				if (ml.getTrackMatte() != TrackMatte.NONE && i+1 < n) {
					Layer matteLayer = _layers.get(i+1);
					if (matteLayer instanceof MediaLayer) {
						setupTrackMatteRenderer(ml, (MediaLayer) matteLayer, composer, camera, null, motionBlur);
						continue;
					}
				}

				_context.saveAndExecute(new WrappedOperation<Object>() {
					public Object execute() {
						ml.setupVideoRenderer(composer, camera, null, motionBlur, _frameBlendEnabled, true, true);
						return null;
					}
				});
			}
		}

		return composer.compose();
	}

	public void setupCollapseTransformation(
			final VideoLayerComposer composer, final Camera camera, final CollapseTransformation ct, final MotionBlur motionBlur) {

		_context.setComposition(this);

		Time time = _context.getTime();

		for (int i = 0, n = _layers.size(); i < n; ++i) {
			Layer layer = _layers.get(i);

			if (!LayerNature.isVideoEnabled(layer)) {
				continue;
			}

			if (layer.getInPoint().after(time) || !layer.getOutPoint().after(time)) {
				continue;
			}

			if (layer instanceof MediaLayer) {
				final MediaLayer ml = (MediaLayer) layer;

				if (ml.getTrackMatte() != TrackMatte.NONE && i+1 < n) {
					Layer matteLayer = _layers.get(i+1);
					if (matteLayer instanceof MediaLayer) {
						setupTrackMatteRenderer(ml, (MediaLayer) matteLayer, composer, camera, ct, motionBlur);
						continue;
					}
				}

				_context.saveAndExecute(new WrappedOperation<Object>() {
					public Object execute() {
						ml.setupVideoRenderer(composer, camera, ct, motionBlur, _frameBlendEnabled, true, true);
						return null;
					}
				});
			}
		}
	}

	private void setupTrackMatteRenderer(
			final MediaLayer fillLayer, final MediaLayer matteLayer,
			VideoLayerComposer composer, final Camera camera, final CollapseTransformation ct, final MotionBlur motionBlur) {

		final VideoLayerRenderer[] renderers = new VideoLayerRenderer[2];
		final Vec3d[][] vertices = new Vec3d[1][];
		final String[] intersectionGroup = new String[1];

		_context.saveAndExecute(new WrappedOperation<Object>() {
			public Object execute() {
				fillLayer.setupVideoRenderer(new VideoLayerComposer() {
					public void add2D(VideoLayerRenderer r)							{ renderers[0] = r; }
					public void add3D(VideoLayerRenderer r, Vec3d[] v, String ig)	{ renderers[0] = r; vertices[0] = v; intersectionGroup[0] = ig; }
				}, camera, ct, motionBlur, _frameBlendEnabled, true, true);
				return null;
			}
		});

		_context.saveAndExecute(new WrappedOperation<Object>() {
			public Object execute() {
				matteLayer.setupVideoRenderer(new VideoLayerComposer() {
					public void add2D(VideoLayerRenderer r)							{ renderers[1] = r; }
					public void add3D(VideoLayerRenderer r, Vec3d[] v, String ig)	{ renderers[1] = r; }
				}, camera, ct, motionBlur, _frameBlendEnabled, true, true);
				return null;
			}
		});


		if (renderers[0] == null) {
			return;

		} else if (renderers[1] == null) {
			if (vertices[0] == null) {
				composer.add2D(renderers[0]);
			} else {
				composer.add3D(renderers[0], vertices[0], intersectionGroup[0]);
			}
			return;
		}


		final TrackMatte trackMatte = fillLayer.getTrackMatte();

		VideoLayerRenderer r = new VideoLayerRenderer() {
			public VideoLayerBuffer render(boolean withDepthBuffer) {
				VideoLayerBuffer fillBuffer = null;
				VideoLayerBuffer matteBuffer = null;
				VideoBuffer vbFill = null;
				try {
					fillBuffer = renderers[0].render(withDepthBuffer);
					matteBuffer = renderers[1].render(false);
					vbFill = fillBuffer.getVideoBuffer();

					VideoBuffer resultBuffer = _vrSupport.createVideoBuffer(vbFill.getColorMode(), vbFill.getBounds());
					_vrSupport.trackMatte(matteBuffer.getVideoBuffer(), fillBuffer.getVideoBuffer(), resultBuffer,
							trackMatte, matteBuffer.getOpacity());

					VideoLayerBuffer newBuffer = new VideoLayerBufferImpl(
							resultBuffer, fillBuffer.getDepthBuffer(),
							fillBuffer.getBlendMode(), fillBuffer.getOpacity(), fillBuffer.getDissolveSeed());
					fillBuffer = null;
					return newBuffer;

				} finally {
					if (matteBuffer != null) {
						matteBuffer.dispose();
					}

					// 正常時はfillBufferを破棄してしまうとDepthBufferまで破棄されてしまうので、VideoBufferだけ破棄する。
					// (tryブロックを抜ける直前にfillBufferにnullを代入しているので、ここでfillBufferがnullでないのは例外発生時のみ)
					if (fillBuffer != null) {
						fillBuffer.dispose();
					} else if (vbFill != null) {
						vbFill.dispose();
					}
				}
			}
		};

		if (vertices[0] == null) {
			composer.add2D(r);
		} else {
			composer.add3D(r, vertices[0], intersectionGroup[0]);
		}
	}

	public VideoBuffer getLayerVideoFrame(Layer layer, final boolean withEffects, final boolean withTransformation) {
		if (_context.getComposition() != this) {
			throw new IllegalStateException("this composition is not the context composition.");
		}

		if (layer == null) {
			throw new NullPointerException();
		}

		if (!_layers.contains(layer)) {
			throw new IllegalArgumentException("no such layer found in this LayerComposition");
		}

		if (!LayerNature.isVideoNature(layer)) {
			return null;
		}

		if (layer instanceof MediaLayer) {
			final MediaLayer ml = (MediaLayer) layer;

			MediaInput input = ml.getMediaInput();
			if (input == null) {
				return null;
			}

			VideoBounds bounds = input.getVideoFrameBounds();
			Size2i layerSize = LayerNature.isCTCR(layer) ? getSize() : new Size2i(bounds.width, bounds.height);
			Size2i viewportSize = _context.getVideoResolution().scale(layerSize);

			if (viewportSize.width == 0 || viewportSize.height == 0) {
				VideoBuffer vb = _vrSupport.createVideoBuffer(_context.getColorMode(), viewportSize);
				vb.clear();
				return vb;
			}

			final VideoLayerRenderer[] renderers = new VideoLayerRenderer[1];

			final VideoLayerComposer composer = new VideoLayerComposer() {
				public void add2D(VideoLayerRenderer r)							{ renderers[0] = r; }
				public void add3D(VideoLayerRenderer r, Vec3d[] v, String ig)	{ renderers[0] = r; }
			};

			final CameraImpl camera = new CameraImpl(
					_context, _vrSupport, null, layerSize, viewportSize);

			final MotionBlur motionBlur = _motionBlurEnabled ? _motionBlur : null;

			_context.saveAndExecute(new WrappedOperation<Object>() {
				public Object execute() {
					ml.setupVideoRenderer(
							composer, camera, null, motionBlur, _frameBlendEnabled,
							withEffects, withTransformation);
					return null;
				}
			});

			VideoLayerBuffer vlb = renderers[0].render(false);
			return vlb.getVideoBuffer();
		}

		return null;
	}

	public AudioBuffer renderAudioChunk() {
		_context.setComposition(this);

		if (isFrameDurationPreserved()) {
			_context.setVideoFrameDuration(getFrameDuration());
		}

		AudioBuffer ab = null;
		Time time = _context.getTime();
		Time chunkDuration = Time.fromFrameNumber(
				_context.getAudioFrameCount(), _context.getAudioMode().sampleDuration);

		for (Layer layer : _layers) {
			if (!LayerNature.isAudioEnabled(layer)) {
				continue;
			}

			if (layer.getInPoint().after(time.add(chunkDuration)) || !layer.getOutPoint().after(time)) {
				continue;
			}

			if (layer instanceof MediaLayer) {
				final MediaLayer al = (MediaLayer) layer;

				AudioBuffer alab = _context.saveAndExecute(new WrappedOperation<AudioBuffer>() {
					public AudioBuffer execute() {
						return al.renderAudioChunk(true, true);
					}
				});

				if (alab != null) {
					if (ab == null) {
						ab = alab;
					} else {
						AudioBuffer abOld = ab;
						try {
							ab = _arSupport.mix(ab, alab);
						} finally {
							if (ab != abOld) {
								abOld.dispose();
							}
							if (ab != alab) {
								alab.dispose();
							}
						}
					}
				}
			}
		}

		return ab;
	}

	public AudioBuffer getLayerAudioChunk(Layer layer, final boolean withEffects, final boolean withLevelsAndInOut) {
		if (_context.getComposition() != this) {
			throw new IllegalStateException("this composition is not the context composition.");
		}

		if (layer == null) {
			throw new NullPointerException();
		}

		if (!_layers.contains(layer)) {
			throw new IllegalArgumentException("no such layer found in this LayerComposition");
		}

		if (!LayerNature.isAudioNature(layer)) {
			return null;
		}

		if (layer instanceof MediaLayer) {
			final MediaLayer ml = (MediaLayer) layer;

			MediaInput input = ml.getMediaInput();
			if (input == null) {
				return null;
			}

			return _context.saveAndExecute(new WrappedOperation<AudioBuffer>() {
				public AudioBuffer execute() {
					return ml.renderAudioChunk(withEffects, withLevelsAndInOut);
				}
			});
		}

		return null;
	}

	public Object createExpressionElement(CoreContext context) {
		return new LayerCompositionExpressionElement(context);
	}

	public class LayerCompositionExpressionElement {

		protected final CoreContext context;

		public LayerCompositionExpressionElement(CoreContext context) {
			this.context = context;
		}

		public Object layer(int index) {
			Layer layer = _layers.get(_layers.size() - index);
			return context.getExpressionElement(layer);
		}

		public Object layer(String name) {
			for (ListIterator<Layer> it = _layers.listIterator(_layers.size()); it.hasPrevious(); ) {
				Layer layer = it.previous();
				if (layer.getName().equals(name)) {
					return context.getExpressionElement(layer);
				}
			}
			return null;
		}

		public int getHeight() {
			return getSize().width;
		}

		public int getWidth() {
			return getSize().height;
		}

	}

}
