/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import javax.media.opengl.GL2;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.services.IAntiAliasSupport;
import ch.kuramo.javie.core.services.GLGlobal;
import ch.kuramo.javie.core.services.RenderBufferPool;
import ch.kuramo.javie.core.services.RenderContext;

import com.google.inject.Inject;

public class AntiAliasSupportImpl implements IAntiAliasSupport {

	private final RenderContext _context;

	private final RenderBufferPool _rbPool;

	private final int _samples;

	private int _frameBuffer;

	private int _renderBuffer;

	private int _internalFormat;

	private int _width;

	private int _height;


	@Inject
	public AntiAliasSupportImpl(RenderContext context, RenderBufferPool rbPool, GLGlobal glGlobal) {
		super();
		_context = context;
		_rbPool = rbPool;

		// GL_MAX_SAMPLESが8より大きい場合でも負荷とのバランスを考えて8に制限している。
		// TODO 8より大きいサンプル数も環境設定で指定できるようにする。
		_samples = Math.min(glGlobal.getMaxSamples(), 8);
	}

	void dispose() {
		GL2 gl = _context.getGL().getGL2();

		_internalFormat = 0;
		_width = 0;
		_height = 0;

		if (_renderBuffer != 0) {
			_rbPool.put(_renderBuffer);
			_renderBuffer = 0;
		}

		if (_frameBuffer != 0) {
			gl.glDeleteFramebuffers(1, new int[] { _frameBuffer }, 0);
			_frameBuffer = 0;
		}
	}

	private int getFrameBuffer(GL2 gl) {
		if (_frameBuffer == 0) {
			int[] fb = new int[1];
			gl.glGenFramebuffers(1, fb, 0);
			_frameBuffer = fb[0];
		}
		return _frameBuffer;
	}

	private int getRenderBuffer(GL2 gl) {
		if (_renderBuffer == 0) {
			_renderBuffer = _rbPool.get();
		}
		return _renderBuffer;
	}

	public void antiAlias(int width, int height, Color clearColor, Runnable operation) {
		if (width <= 0 || height <= 0) {
			// TODO 例外を投げた方がいいか？
			//throw new IllegalArgumentException();
			return;
		}

		if (_samples < 2) {
			operation.run();
			return;
		}


		GL2 gl = _context.getGL().getGL2();

		int[] savedFb = new int[1];
		gl.glGetIntegerv(GL2.GL_FRAMEBUFFER_BINDING, savedFb, 0);
		try {
			int fb = getFrameBuffer(gl);
			int rb = getRenderBuffer(gl);

			gl.glBindFramebuffer(GL2.GL_FRAMEBUFFER, fb);
			gl.glBindRenderbuffer(GL2.GL_RENDERBUFFER, rb);

			int internalFormat = _context.getColorMode().glInternalFormat;
			if (internalFormat != _internalFormat || width > _width || height > _height) {
				gl.glRenderbufferStorageMultisample(GL2.GL_RENDERBUFFER, _samples, internalFormat, width, height);
				_internalFormat = internalFormat;
				_width = width;
				_height = height;
			}

			gl.glFramebufferRenderbuffer(GL2.GL_FRAMEBUFFER, GL2.GL_COLOR_ATTACHMENT0, GL2.GL_RENDERBUFFER, rb);

			if (clearColor != null) {
				float a = (float)clearColor.a;
				float r = (float)clearColor.r * a;
				float g = (float)clearColor.g * a;
				float b = (float)clearColor.b * a;
				gl.glClearColor(r, g, b, a);
				gl.glClear(GL2.GL_COLOR_BUFFER_BIT);
			}

			operation.run();

			gl.glBindFramebuffer(GL2.GL_READ_FRAMEBUFFER, fb);
			gl.glBindFramebuffer(GL2.GL_DRAW_FRAMEBUFFER, savedFb[0]);
			gl.glBlitFramebuffer(0, 0, width, height, 0, 0, width, height, GL2.GL_COLOR_BUFFER_BIT, GL2.GL_NEAREST);

		} finally {
			gl.glFramebufferRenderbuffer(GL2.GL_FRAMEBUFFER, GL2.GL_COLOR_ATTACHMENT0, GL2.GL_RENDERBUFFER, 0);
			gl.glBindRenderbuffer(GL2.GL_RENDERBUFFER, 0);
			gl.glBindFramebuffer(GL2.GL_FRAMEBUFFER, savedFb[0]);
		}
	}

	public void antiAlias(int width, int height, Runnable operation) {
		antiAlias(width, height, Color.COLORLESS_TRANSPARENT, operation);
	}

}
