/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableInteger;
import ch.kuramo.javie.api.IAnimatableLayerReference;
import ch.kuramo.javie.api.IAnimatableString;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IAnimatableVec3d;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.core.AnimatableBoolean;
import ch.kuramo.javie.core.AnimatableColor;
import ch.kuramo.javie.core.AnimatableDouble;
import ch.kuramo.javie.core.AnimatableInteger;
import ch.kuramo.javie.core.AnimatableLayerReference;
import ch.kuramo.javie.core.AnimatableString;
import ch.kuramo.javie.core.AnimatableValue;
import ch.kuramo.javie.core.AnimatableVec2d;
import ch.kuramo.javie.core.AnimatableVec3d;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectDescriptor;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.PropertyDescriptor;
import ch.kuramo.javie.core.Util;

class EffectDescriptorImpl implements EffectDescriptor {

	private final String _type;

	private final Class<?> _interiorClass;

	private final Class<Effect> _exteriorClass;

	private final Class<?> _exprInterface;

	private final ResourceBundle _bundle;

	private final String _category;

	private final String _label;

	private final Map<String, PropertyDescriptor> _propertyDescriptorMap;

	private final List<PropertyDescriptor> _propertyDescriptorList;


	EffectDescriptorImpl(String type, Class<?> interiorClass, Class<Effect> exteriorClass, Class<?> exprInterface) {
		super();
		_type = type;
		_interiorClass = interiorClass;
		_exteriorClass = exteriorClass;
		_exprInterface = exprInterface;
		_bundle = getBundle();
		_category = interiorClass.getAnnotation(ch.kuramo.javie.api.annotations.Effect.class).category().trim();
		_label = getString("label");
		_propertyDescriptorMap = Collections.unmodifiableMap(createPropertyDescriptors());
		_propertyDescriptorList = Collections.unmodifiableList(Util.newList(_propertyDescriptorMap.values()));
	}

	private ResourceBundle getBundle() {
		try {
			// javassistでいじったクラスだからか _interiorClass.getPackage() が
			// nullを返すようなので、クラス名からパッケージ名を切り出す。
			String className = _interiorClass.getName();
			String bundleName = className.substring(0, className.lastIndexOf('.') + 1) + "messages";

			return ResourceBundle.getBundle(
					bundleName, Locale.getDefault(), _interiorClass.getClassLoader());

		} catch (MissingResourceException e) {
			return null;
		}
	}

	private String getString(String key) {
		String str = getString(key, null);
		return (str != null) ? str : ('!' + key + '!');
	}

	private String getString(String key, String defaultStr) {
		String str = null;
		if (_bundle != null) {
			try {
				str = _bundle.getString(String.format("%s.%s", _interiorClass.getSimpleName(), key));
			} catch (MissingResourceException e) {
				// ignore
			}
		}
		return (str != null) ? str : defaultStr;
	}

	private Map<String, PropertyDescriptor> createPropertyDescriptors() {
		Map<String, PropertyDescriptor> map = Util.newLinkedHashMap();

		for (Field f : _interiorClass.getDeclaredFields()) {
			Property anno = f.getAnnotation(Property.class);
			if (anno == null) {
				continue;
			}

			String name = f.getName();
			String value = anno.value();
			String min = anno.min();
			String max = anno.max();

			Class<?> type = f.getType();
			if (IAnimatableDouble.class.isAssignableFrom(type)) {
				map.put(name, new AnimatableDoublePropertyDescriptor(name, value, min, max));
			} else if (IAnimatableInteger.class.isAssignableFrom(type)) {
				map.put(name, new AnimatableIntegerPropertyDescriptor(name, value, min, max));
			} else if (IAnimatableVec2d.class.isAssignableFrom(type)) {
				map.put(name, new AnimatableVec2dPropertyDescriptor(name, value, min, max));
			} else if (IAnimatableVec3d.class.isAssignableFrom(type)) {
				map.put(name, new AnimatableVec3dPropertyDescriptor(name, value, min, max));
			} else if (IAnimatableColor.class.isAssignableFrom(type)) {
				map.put(name, new AnimatableColorPropertyDescriptor(name, value));
			} else if (IAnimatableString.class.isAssignableFrom(type)) {
				map.put(name, new AnimatableStringPropertyDescriptor(name, value));
			} else if (IAnimatableBoolean.class.isAssignableFrom(type)) {
				map.put(name, new AnimatableBooleanPropertyDescriptor(name, value));
			} else if (IAnimatableEnum.class.isAssignableFrom(type)) {
				map.put(name, new AnimatableEnumPropertyDescriptor(name, value));
			} else if (IAnimatableLayerReference.class.isAssignableFrom(type)) {
				map.put(name, new AnimatableLayerReferencePropertyDescriptor(name, value));
//			} else if (String.class.equals(type)) {
//				map.put(name, new StringPropertyDescriptor(name, value));
//			} else if (Enum.class.isAssignableFrom(type)) {
//				@SuppressWarnings("unchecked")
//				Class<Enum> enumType = (Class<Enum>) type;
//				map.put(name, new EnumPropertyDescriptor(name, value, enumType));
//			} else if (type.isPrimitive()) {
//				map.put(name, new PrimitivePropertyDescriptor(name, value, type));
			} else {
				// 非サポートの型
				// EffectRegistryImpl#processProperties で警告を出しているはずなので無視
			}
		}

		return map;
	}

	public boolean isUnknown() {
		return false;
	}

	public String getType() {
		return _type;
	}

	public Class<Effect> getEffectClass() {
		return _exteriorClass;
	}

	public Class<?> getExpressionInterface() {
		return _exprInterface;
	}

	public String getCategory() {
		return _category;
	}

	public String getLabel() {
		return _label;
	}

	public List<PropertyDescriptor> getPropertyDescriptors() {
		return _propertyDescriptorList;
	}

	public PropertyDescriptor getPropertyDescriptor(String name) {
		return _propertyDescriptorMap.get(name);
	}


	private <A extends AnimatableValue<?>> Class<A> getPropertyType(String propertyName) {
		try {
			String upperCamelCase = Character.toUpperCase(propertyName.charAt(0)) + propertyName.substring(1);
			@SuppressWarnings("unchecked")
			Class<A> propertyType = (Class<A>) _exteriorClass.getMethod("get" + upperCamelCase).getReturnType();
			return propertyType;
		} catch (NoSuchMethodException e) {
			throw new JavieRuntimeException(e);
		}
	}

	private abstract class AbstractPropertyDescriptor<T> implements PropertyDescriptor {

		protected final Class<T> type;

		protected final String name;

		protected final String label;

		protected final Method getter;

		protected final Method setter;

		private AbstractPropertyDescriptor(Class<T> type, String name) {
			super();
			this.type = type;
			this.name = name;
			this.label = EffectDescriptorImpl.this.getString(name + ".label");
			try {
				String upperCamelCase = Character.toUpperCase(name.charAt(0)) + name.substring(1);
				getter = _exteriorClass.getDeclaredMethod("get" + upperCamelCase);
				setter = _exteriorClass.getDeclaredMethod("set" + upperCamelCase, type);
			} catch (NoSuchMethodException e) {
				throw new JavieRuntimeException(e);
			}
		}

		public Class<T> getPropertyClass() {
			return type;
		}

		public String getName() {
			return name;
		}

		public String getLabel() {
			return label;
		}

		public String getString(String key) {
			return EffectDescriptorImpl.this.getString(name + "." + key, null);
		}

		public void afterDecode(Project p, Effect effect) throws ProjectDecodeException {
			// nothing to do
		}

		public T get(Effect effect) {
			try {
				@SuppressWarnings("unchecked")
				T value = (T) getter.invoke(effect);
				return value;
			} catch (IllegalAccessException e) {
				throw new JavieRuntimeException(e);
			} catch (InvocationTargetException e) {
				throw new JavieRuntimeException(e);
			}
		}

		protected void set(Effect effect, T value) {
			try {
				setter.invoke(effect, value);
			} catch (IllegalAccessException e) {
				throw new JavieRuntimeException(e);
			} catch (InvocationTargetException e) {
				throw new JavieRuntimeException(e);
			}
		}
	}

	private class AnimatableDoublePropertyDescriptor extends AbstractPropertyDescriptor<AnimatableDouble> {

		private final Double value;

		private final Double min;

		private final Double max;

		private AnimatableDoublePropertyDescriptor(String name, String value, String min, String max) {
			super(AnimatableDouble.class, name);
			this.min = toDouble(min, Double.NEGATIVE_INFINITY);
			this.max = Math.max(this.min, toDouble(max, Double.POSITIVE_INFINITY));
			this.value = Math.max(this.min, Math.min(this.max, toDouble(value, 0)));
		}

		public void initValue(Effect effect) {
			set(effect, new AnimatableDouble(value, min, max));
		}
	}

	private class AnimatableIntegerPropertyDescriptor extends AbstractPropertyDescriptor<AnimatableInteger> {

		private final Integer value;

		private final Integer min;

		private final Integer max;

		private AnimatableIntegerPropertyDescriptor(String name, String value, String min, String max) {
			super(AnimatableInteger.class, name);
			this.min = toInteger(min, Integer.MIN_VALUE);
			this.max = Math.max(this.min, toInteger(max, Integer.MAX_VALUE));
			this.value = Math.max(this.min, Math.min(this.max, toInteger(value, 0)));
		}

		public void initValue(Effect effect) {
			set(effect, new AnimatableInteger(value, min, max));
		}
	}

	private class AnimatableVec2dPropertyDescriptor extends AbstractPropertyDescriptor<AnimatableVec2d> {

		private final Vec2d value;

		private final Vec2d min;

		private final Vec2d max;

		private AnimatableVec2dPropertyDescriptor(String name, String value, String min, String max) {
			super(AnimatableVec2d.class, name);
			this.min = toVec2d(min, Vec2d.NEGATIVE_INFINITY);
			this.max = Vec2d.max(this.min, toVec2d(max, Vec2d.POSITIVE_INFINITY));
			this.value = Vec2d.max(this.min, Vec2d.min(this.max, toVec2d(value, Vec2d.ZERO)));
		}

		public void initValue(Effect effect) {
			set(effect, new AnimatableVec2d(value, min, max));
		}
	}

	private class AnimatableVec3dPropertyDescriptor extends AbstractPropertyDescriptor<AnimatableVec3d> {

		private final Vec3d value;

		private final Vec3d min;

		private final Vec3d max;

		private AnimatableVec3dPropertyDescriptor(String name, String value, String min, String max) {
			super(AnimatableVec3d.class, name);
			this.min = toVec3d(min, Vec3d.NEGATIVE_INFINITY);
			this.max = Vec3d.max(this.min, toVec3d(max, Vec3d.POSITIVE_INFINITY));
			this.value = Vec3d.max(this.min, Vec3d.min(this.max, toVec3d(value, Vec3d.ZERO)));
		}

		public void initValue(Effect effect) {
			set(effect, new AnimatableVec3d(value, min, max));
		}
	}

	private class AnimatableColorPropertyDescriptor extends AbstractPropertyDescriptor<AnimatableColor> {

		private final Color value;

		private AnimatableColorPropertyDescriptor(String name, String value) {
			super(AnimatableColor.class, name);
			this.value = toColor(value, Color.BLACK);
		}

		public void initValue(Effect effect) {
			set(effect, new AnimatableColor(value));
		}
	}

	private class AnimatableStringPropertyDescriptor extends AbstractPropertyDescriptor<AnimatableString> {

		private final String value;

		private AnimatableStringPropertyDescriptor(String name, String value) {
			super(AnimatableString.class, name);
			this.value = value;
		}

		public void initValue(Effect effect) {
			set(effect, new AnimatableString(value));
		}
	}

	private class AnimatableBooleanPropertyDescriptor extends AbstractPropertyDescriptor<AnimatableBoolean> {

		private final Boolean value;

		private AnimatableBooleanPropertyDescriptor(String name, String value) {
			super(AnimatableBoolean.class, name);
			this.value = toBoolean(value);
		}

		public void initValue(Effect effect) {
			set(effect, new AnimatableBoolean(value));
		}
	}

	@SuppressWarnings("unchecked")
	private class AnimatableEnumPropertyDescriptor extends AbstractPropertyDescriptor {

		private final Class<Enum> enumType;

		private final Constructor<?> constructor;

		private final Enum value;

		private AnimatableEnumPropertyDescriptor(String name, String value) {
			super(getPropertyType(name), name);
			try {
				ParameterizedType genericSuper = (ParameterizedType) type.getGenericSuperclass();
				enumType = (Class<Enum>) genericSuper.getActualTypeArguments()[0];
				constructor = type.getConstructor(enumType);
				this.value = toEnum(enumType, value);
			} catch (NoSuchMethodException e) {
				throw new JavieRuntimeException(e);
			}
		}

		public void initValue(Effect effect) {
			try {
				set(effect, constructor.newInstance(value));
			} catch (InstantiationException e) {
				throw new JavieRuntimeException(e);
			} catch (IllegalAccessException e) {
				throw new JavieRuntimeException(e);
			} catch (InvocationTargetException e) {
				throw new JavieRuntimeException(e);
			}
		}
	}

	private class AnimatableLayerReferencePropertyDescriptor extends AbstractPropertyDescriptor<AnimatableLayerReference> {

		private final String value;

		private AnimatableLayerReferencePropertyDescriptor(String name, String value) {
			super(AnimatableLayerReference.class, name);
			this.value = value;
		}

		public void initValue(Effect effect) {
			set(effect, new AnimatableLayerReference(value));
		}
	}

	@SuppressWarnings("unused")
	private class StringPropertyDescriptor extends AbstractPropertyDescriptor<String> {

		private final String value;

		private StringPropertyDescriptor(String name, String value) {
			super(String.class, name);
			this.value = value;
		}

		public void initValue(Effect effect) {
			set(effect, value);
		}
	}

	@SuppressWarnings({ "unchecked", "unused" })
	private class EnumPropertyDescriptor extends AbstractPropertyDescriptor<Enum> {

		private final Enum value;

		private EnumPropertyDescriptor(String name, String value, Class<Enum> enumType) {
			super(enumType, name);
			this.value = toEnum(enumType, value);
		}

		public void initValue(Effect effect) {
			set(effect, value);
		}
	}

	@SuppressWarnings({ "unchecked", "unused" })
	private class PrimitivePropertyDescriptor extends AbstractPropertyDescriptor {

		private final Object value;

		private PrimitivePropertyDescriptor(String name, String value, Class<?> primitiveType) {
			super(primitiveType, name);
			this.value = toPrimitive(primitiveType, value);
		}

		public void initValue(Effect effect) {
			set(effect, value);
		}
	}

	private static boolean toBoolean(String s) {
		return Boolean.parseBoolean(s.trim());
	}

	private static double toDouble(String s, double defaultValue) {
		if (s.length() > 0) {
			try {
				return Double.parseDouble(s.trim());
			} catch (NumberFormatException e) {
				// TODO ログに警告を出す
			}
		}
		return defaultValue;
	}

	private static float toFloat(String s, float defaultValue) {
		if (s.length() > 0) {
			try {
				return Float.parseFloat(s.trim());
			} catch (NumberFormatException e) {
				// TODO ログに警告を出す
			}
		}
		return defaultValue;
	}

	private static long toLong(String s, long defaultValue) {
		if (s.length() > 0) {
			try {
				return Long.parseLong(s.trim());
			} catch (NumberFormatException e) {
				// TODO ログに警告を出す
			}
		}
		return defaultValue;
	}

	private static int toInteger(String s, int defaultValue) {
		if (s.length() > 0) {
			try {
				return Integer.parseInt(s.trim());
			} catch (NumberFormatException e) {
				// TODO ログに警告を出す
			}
		}
		return defaultValue;
	}

	private static short toShort(String s, short defaultValue) {
		if (s.length() > 0) {
			try {
				return Short.parseShort(s.trim());
			} catch (NumberFormatException e) {
				// TODO ログに警告を出す
			}
		}
		return defaultValue;
	}

	private static char toCharacter(String s, char defaultValue) {
		if (s.length() > 0) {
			try {
				int intValue = Integer.parseInt(s.trim());
				if (intValue < Character.MIN_VALUE || intValue > Character.MAX_VALUE) {
					throw new NumberFormatException(s);
				}
				return (char) intValue;
			} catch (NumberFormatException e) {
				// TODO ログに警告を出す
			}
		}
		return defaultValue;
	}

	private static byte toByte(String s, byte defaultValue) {
		if (s.length() > 0) {
			try {
				int intValue = Integer.parseInt(s.trim());
				if (intValue < Byte.MIN_VALUE || intValue > Byte.MAX_VALUE) {
					throw new NumberFormatException(s);
				}
				return (byte) intValue;
			} catch (NumberFormatException e) {
				// TODO ログに警告を出す
			}
		}
		return defaultValue;
	}

	private static Vec2d toVec2d(String s, Vec2d defaultValue) {
		if (s.length() > 0) {
			try {
				double x, y;
				String[] array = s.trim().split("\\s*,\\s*");
				switch (array.length) {
					case 1:
						x = y = Double.parseDouble(array[0]);
						break;
					default:
						x = Double.parseDouble(array[0]);
						y = Double.parseDouble(array[1]);
						break;
				}
				return new Vec2d(x, y);
			} catch (NumberFormatException e) {
				// TODO ログに警告を出す
			}
		}
		return defaultValue;
	}


	private static Vec3d toVec3d(String s, Vec3d defaultValue) {
		if (s.length() > 0) {
			try {
				double x, y, z;
				String[] array = s.trim().split("\\s*,\\s*");
				switch (array.length) {
					case 1:
						x = y = z = Double.parseDouble(array[0]);
						break;
					case 2:
						x = Double.parseDouble(array[0]);
						y = Double.parseDouble(array[1]);
						z = 0;
						break;
					default:
						x = Double.parseDouble(array[0]);
						y = Double.parseDouble(array[1]);
						z = Double.parseDouble(array[2]);
						break;
				}
				return new Vec3d(x, y, z);
			} catch (NumberFormatException e) {
				// TODO ログに警告を出す
			}
		}
		return defaultValue;
	}

	private static Color toColor(String s, Color defaultValue) {
		if (s.length() > 0) {
			try {
				double r, g, b, a;
				String[] array = s.trim().split("\\s*,\\s*");
				switch (array.length) {
					case 1:
						r = g = b = Double.parseDouble(array[0]);
						a = 1;
						break;
					case 2:
						r = g = b = Double.parseDouble(array[0]);
						a = Double.parseDouble(array[1]);
						break;
					case 3:
						r = Double.parseDouble(array[0]);
						g = Double.parseDouble(array[1]);
						b = Double.parseDouble(array[2]);
						a = 1;
						break;
					default:
						r = Double.parseDouble(array[0]);
						g = Double.parseDouble(array[1]);
						b = Double.parseDouble(array[2]);
						a = Double.parseDouble(array[3]);
						break;
				}
				return new Color(r, g, b, a);
			} catch (NumberFormatException e) {
				// TODO ログに警告を出す
			}
		}
		return defaultValue;
	}

	private static <E extends Enum<E>> E toEnum(Class<E> enumType, String s) {
		try {
			return Enum.valueOf(enumType, s.trim());
		} catch (IllegalArgumentException e) {
			// TODO ログに警告を出す
		}
		try {
			@SuppressWarnings("unchecked")
			E first = (E) Array.get(enumType.getMethod("values").invoke(null), 0);
			return first;
		} catch (IllegalAccessException e) {
			throw new JavieRuntimeException(e);
		} catch (InvocationTargetException e) {
			throw new JavieRuntimeException(e);
		} catch (NoSuchMethodException e) {
			throw new JavieRuntimeException(e);
		}
	}

	private static Object toPrimitive(Class<?> primitiveType, String s) {
		if (primitiveType == boolean.class) {
			return toBoolean(s);
		} else if (primitiveType == double.class) {
			return toDouble(s, 0);
		} else if (primitiveType == float.class) {
			return toFloat(s, 0);
		} else if (primitiveType == long.class) {
			return toLong(s, 0);
		} else if (primitiveType == int.class) {
			return toInteger(s, 0);
		} else if (primitiveType == short.class) {
			return toShort(s, (short)0);
		} else if (primitiveType == char.class) {
			return toCharacter(s, (char)0);
		} else if (primitiveType == byte.class) {
			return toByte(s, (byte)0);
		} else {
			throw new JavieRuntimeException("unsupported type: " + primitiveType.getName());
		}
	}

}
