/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.io.File;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.core.MediaFileInput;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.internal.JavaImageInput;
import ch.kuramo.javie.core.internal.JavaSoundInput;
import ch.kuramo.javie.core.services.MediaFileInputFactory;

import com.google.inject.Inject;
import com.google.inject.Injector;

public class MediaFileInputFactoryImpl implements MediaFileInputFactory {

	private static final Logger _logger = LoggerFactory.getLogger(MediaFileInputFactoryImpl.class);


	private static class Entry {
		int counter = 1;
		final MediaFileInput input;
		Entry(MediaFileInput input) { this.input = input; }
	}


	private final List<Class<? extends MediaFileInput>> _inputClasses = Util.newList();

	private final Map<String, Entry> _entries = Util.newMap();

	@Inject
	private Injector _injector;


	public MediaFileInputFactoryImpl() {
		super();

		Class<MediaFileInput> clazz;

		if ((clazz = classForName("ch.kuramo.javie.core.internal.MacOSXQTCoreVideoInput")) != null) {
			_inputClasses.add(JavaImageInput.class);
			_inputClasses.add(clazz);
			_inputClasses.add(JavaSoundInput.class);

		} else if ((clazz = classForName("ch.kuramo.javie.core.internal.WindowsDirectShowInput")) != null) {
			_inputClasses.add(JavaImageInput.class);
			_inputClasses.add(JavaSoundInput.class);	// WindowsDirectShowInputでwavファイルを読み込むと
			_inputClasses.add(clazz);					// パフォーマンスに問題があるようなので、JavaSoundInputを前に入れておく。

		} else {
			_logger.warn("only JavaImageInput and JavaSoundInput are available");
			_inputClasses.add(JavaImageInput.class);
			_inputClasses.add(JavaSoundInput.class);
		}
	}

	private Class<MediaFileInput> classForName(String className) {
		try {
			@SuppressWarnings("unchecked")
			Class<MediaFileInput> clazz = (Class<MediaFileInput>) Class.forName(className);
			return clazz;
		} catch (Throwable t) {
			return null;
		}
	}

	public synchronized MediaFileInput get(String key, File file) {
		Entry entry = _entries.get(key);
		if (entry != null) {
			++entry.counter;
			return entry.input;
		}

		// TODO 複数の MediaFileInput 実装の優先順位をユーザーが決められるようにして、
		//		その順番で試行するようにする。

		Map<Class<? extends MediaFileInput>, Throwable> throwables = Util.newLinkedHashMap();

		for (Class<? extends MediaFileInput> clazz : _inputClasses) {
			MediaFileInput input;
			try {
				input = _injector.getInstance(clazz);
				if (!input.initialize(file)) {
					input.dispose();
					continue;
				}
			} catch (Throwable t) {
				throwables.put(clazz, t);
				continue;
			}

			_entries.put(key, new Entry(input));
			return input;
		}

		for (Map.Entry<Class<? extends MediaFileInput>, Throwable> e : throwables.entrySet()) {
			_logger.info(String.format("failed to create MediaFileInput (%s): %s: %s",
					e.getKey().getName(), file.getAbsolutePath(), e.getValue().toString()));
		}

		return null;
	}

	public synchronized <T extends MediaFileInput> T get(Class<T> inputClass, String key, File file) {
		Entry entry = _entries.get(key);
		if (entry != null) {
			++entry.counter;
			return inputClass.cast(entry.input);
		}

		try {
			T input = _injector.getInstance(inputClass);
			if (input.initialize(file)) {
				_entries.put(key, new Entry(input));
				return input;
			} else {
				input.dispose();
			}
		} catch (Throwable t) {
			_logger.info(String.format("failed to create MediaFileInput (%s): %s: %s",
					inputClass.getName(), file.getAbsolutePath(), t.toString()));
		}

		return null;
	}

	public synchronized void release(String key) {
		Entry entry = _entries.get(key);
		if (--entry.counter == 0) {
			_entries.remove(key);
			entry.input.dispose();
		}
	}

}
