/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.util.concurrent.ConcurrentLinkedQueue;

import javax.media.opengl.GL2;

import ch.kuramo.javie.core.services.RenderBufferPool;
import ch.kuramo.javie.core.services.RenderContext;

import com.google.inject.Inject;

public class RenderBufferPoolImpl implements RenderBufferPool {

	private final ConcurrentLinkedQueue<Integer> _pool = new ConcurrentLinkedQueue<Integer>();

	private final RenderContext _context;


	@Inject
	public RenderBufferPoolImpl(RenderContext context) {
		_context = context;
	}

	private int createRenderBuffer() {
		GL2 gl = _context.getGL().getGL2();

		int[] rb = new int[1];
		gl.glGenRenderbuffers(1, rb, 0);
		return rb[0];
	}

	private void reduceRenderBuffer(int renderBuffer) {
		GL2 gl = _context.getGL().getGL2();

		int[] binding = new int[1];
		gl.glGetIntegerv(GL2.GL_RENDERBUFFER_BINDING, binding, 0);
		try {
			gl.glBindRenderbuffer(GL2.GL_RENDERBUFFER, renderBuffer);
			gl.glRenderbufferStorage(GL2.GL_RENDERBUFFER, GL2.GL_RGBA2, 1, 1);
		} finally {
			gl.glBindRenderbuffer(GL2.GL_RENDERBUFFER, binding[0]);
		}
	}

	public int get() {
		Integer rb = _pool.peek();
		return (rb != null) ? rb : createRenderBuffer();
	}

	public void put(int renderBuffer) {
		reduceRenderBuffer(renderBuffer);
		_pool.offer(renderBuffer);
	}

}
