/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.util.Queue;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.SynchronousTaskException;
import ch.kuramo.javie.core.services.SynchronousTaskThread;

public class SynchronousTaskThreadImpl extends Thread implements SynchronousTaskThread {

	private static final Logger _logger = LoggerFactory.getLogger(SynchronousTaskThreadImpl.class);


	private final Object _lock = new Object();

	private final Queue<TaskRunner<?>> _queue = Util.newLinkedList();

	private boolean _exit = false;


	public SynchronousTaskThreadImpl() {
		super();
	}

	public SynchronousTaskThreadImpl(String name) {
		super(name);
	}

	public void exit() {
		synchronized (_lock) {
			if (!_exit) {
				_exit = true;
				_lock.notify();
			}
		}
	}

	public <T> T exit(Task<T> task) throws SynchronousTaskException {
		TaskRunner<T> runner = new TaskRunner<T>(task);

		synchronized (_lock) {
			if (_exit) {
				return null;
			}
			_queue.add(runner);
			_exit = true;
			_lock.notify();
		}

		return waitRunner(runner);
	}

	public <T> T invoke(Task<T> task) throws SynchronousTaskException, IllegalStateException {
		TaskRunner<T> runner = new TaskRunner<T>(task);

		synchronized (_lock) {
			if (_exit) {
				throw new IllegalStateException("this SynchronousTaskThread already exited.");
			}
			_queue.add(runner);
			_lock.notify();
		}

		return waitRunner(runner);
	}

	private <T> T waitRunner(TaskRunner<T> runner) throws SynchronousTaskException {
		synchronized (runner) {
			while (!runner.isFinished()) {
				try {
					runner.wait();
				} catch (InterruptedException e) {
					_logger.warn("unexpected interruption", e);
				}
			}
		}

		if (runner.getThrowable() != null) {
			throw new SynchronousTaskException(runner.getThrowable());
		}

		return runner.getResult();
	}

	@Override
	public void run() {
		while (true) {
			TaskRunner<?> runner;

			synchronized (_lock) {
				while ((runner = _queue.poll()) == null && !_exit) {
					try {
						_lock.wait();
					} catch (InterruptedException e) {
						_logger.warn("unexpected interruption", e);
					}
				}

				// _exitがtrueでもrunnerがnullでなければ実行する。
				// なお、ここでrunnerがnullなら_exitは必ずtrueなので、_exitのチェックは省略。
				if (/*_exit &&*/ runner == null) {
					break;
				}
			}

			runner.run();
		}
	}


	private static class TaskRunner<T> {

		private final Task<T> _task;

		private T _result;

		private Throwable _throwable;

		private boolean _finished;


		private TaskRunner(Task<T> task) {
			_task = task;
		}

		private void run() {
			try {
				_result = _task.run();
			} catch (Throwable t) {
				_throwable = t;
			}

			synchronized (this) {
				_finished = true;
				notify();
			}
		}

		private T getResult() {
			return _result;
		}

		private Throwable getThrowable() {
			return _throwable;
		}

		private boolean isFinished() {
			return _finished;
		}

	}

}
