/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.util.List;

import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.VideoBuffer;
import ch.kuramo.javie.core.VideoEffect;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.services.AudioRenderSupport;
import ch.kuramo.javie.core.services.RenderContext;
import ch.kuramo.javie.core.services.VideoEffectPipeline;
import ch.kuramo.javie.core.services.VideoRenderSupport;

import com.google.inject.Inject;

public class VideoEffectPipelineImpl implements VideoEffectPipeline {

	@Inject
	private RenderContext _context;

	@Inject
	private VideoRenderSupport _vrSupport;

	@Inject
	private AudioRenderSupport _arSupport;


	public VideoBounds getVideoBounds(List<Effect> effects,
			WrappedOperation<VideoBounds> inputBoundsOperation) {

		VideoEffectContextImpl impl = new VideoEffectContextImpl(
				enabledVideoEffectsOnly(effects), inputBoundsOperation, null);
		VideoEffectContextProxy.push(impl);
		try {
			return impl.getPreviousBounds();
		} finally {
			VideoEffectContextProxy.pop();
		}
	}

	public VideoBuffer doVideoEffects(List<Effect> effects,
			WrappedOperation<VideoBounds> inputBoundsOperation,
			WrappedOperation<VideoBuffer> inputBufferOperation) {

		VideoEffectContextImpl impl = new VideoEffectContextImpl(
				enabledVideoEffectsOnly(effects), inputBoundsOperation, inputBufferOperation);
		VideoEffectContextProxy.push(impl);
		try {
			return (VideoBuffer) impl.doPreviousEffect();
		} finally {
			VideoEffectContextProxy.pop();
		}
	}

	private List<VideoEffect> enabledVideoEffectsOnly(List<Effect> effects) {
		List<VideoEffect> videoEffects = Util.newList();
		for (Effect e : effects) {
			if (e instanceof VideoEffect && e.isEnabled()) {
				videoEffects.add((VideoEffect) e);
			}
		}
		return videoEffects;
	}


	private class VideoEffectContextImpl
			extends AbstractEffectContext implements IVideoEffectContext {

		private List<VideoEffect> effects;

		private final WrappedOperation<VideoBounds> inputBoundsOperation;

		private final WrappedOperation<VideoBuffer> inputBufferOperation;

		private boolean gettingBounds;


		private VideoEffectContextImpl(
				List<VideoEffect> effects,
				WrappedOperation<VideoBounds> inputBoundsOperation,
				WrappedOperation<VideoBuffer> inputBufferOperation) {

			super(_context, _vrSupport, _arSupport);

			this.effects = effects;
			this.inputBoundsOperation = inputBoundsOperation;
			this.inputBufferOperation = inputBufferOperation;
		}

		public VideoBounds getPreviousBounds() {
			if (inputBoundsOperation == null) {
				throw new IllegalStateException();
			}

			boolean savedGettingBounds = gettingBounds;
			try {
				gettingBounds = true;

				if (effects.isEmpty()) {
					return _context.saveAndExecute(inputBoundsOperation);
				}

				final VideoEffect prevEffect = effects.get(effects.size() - 1);

				return executePrevious(new WrappedOperation<VideoBounds>() {
					public VideoBounds execute() {
						VideoBounds bounds = prevEffect.getVideoBounds();
						return (bounds != null) ? bounds : getPreviousBounds();
					}
				});
			} finally {
				gettingBounds = savedGettingBounds;
			}
		}
		
		public IVideoBuffer doPreviousEffect() {
			if (inputBufferOperation == null || gettingBounds) {
				throw new IllegalStateException();
			}

			if (effects.isEmpty()) {
				return (IVideoBuffer) _context.saveAndExecute(inputBufferOperation);
			}

			final VideoEffect prevEffect = effects.get(effects.size() - 1);

			return executePrevious(new WrappedOperation<IVideoBuffer>() {
				public IVideoBuffer execute() {
					IVideoBuffer buffer = prevEffect.doVideoEffect();
					return (buffer != null) ? buffer : doPreviousEffect();
				}
			});
		}

		private <T> T executePrevious(final WrappedOperation<T> wop) {
			List<VideoEffect> savedEffects = effects;
			try {

				effects = effects.subList(0, effects.size() - 1);

				return _context.saveAndExecute(new WrappedOperation<T>() {
					public T execute() {
						return _vrSupport.pushMatrixAndExecute(wop);
					}
				});

			} finally {
				effects = savedEffects;
			}
		}


	}

}
