/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.nio.FloatBuffer;
import java.nio.ShortBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.concurrent.locks.ReentrantLock;

import javax.media.opengl.GL2;
import javax.media.opengl.glu.GLU;
import javax.vecmath.Point2d;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IAntiAliasSupport;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.core.BlendMode;
import ch.kuramo.javie.core.DepthBuffer;
import ch.kuramo.javie.core.TrackMatte;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.VideoBuffer;
import ch.kuramo.javie.core.VideoLayerBuffer;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.internal.DepthBufferImpl;
import ch.kuramo.javie.core.internal.VideoBufferImpl;
import ch.kuramo.javie.core.services.GLGlobal;
import ch.kuramo.javie.core.services.RenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;
import ch.kuramo.javie.core.shaders.AccumulationShaders;
import ch.kuramo.javie.core.shaders.BlendModeShaders;
import ch.kuramo.javie.core.shaders.TrackMatteShaders;

import com.google.inject.Inject;
import com.google.inject.Injector;

import ftgl.FTGL;
import ftgl.FTGLfont;

public class VideoRenderSupportImpl implements VideoRenderSupport {

	private static final Logger _logger = LoggerFactory.getLogger(VideoRenderSupportImpl.class);

	private static final float[] FLOAT0000 = new float[] { 0, 0, 0, 0 };


	@Inject
	private Injector _injector;

	@Inject
	private GLGlobal _glGlobal;

	@Inject
	private RenderContext _context;

	@Inject
	private IShaderRegistry _shaders;

	@Inject
	private IAntiAliasSupport _aaSupport;

	@Inject
	private IArrayPools _arrayPools;


	public VideoBuffer createVideoBuffer(ColorMode colorMode, Size2i size) {
		VideoBufferImpl vb = new VideoBufferImpl(colorMode, size);
		_injector.injectMembers(vb);
		return vb;
	}

	public VideoBuffer createVideoBuffer(ColorMode colorMode, VideoBounds bounds) {
		VideoBufferImpl vb = new VideoBufferImpl(colorMode, bounds);
		_injector.injectMembers(vb);
		return vb;
	}

	public DepthBuffer createDepthBuffer(Size2i size) {
		DepthBufferImpl db = new DepthBufferImpl(size);
		_injector.injectMembers(db);
		return db;
	}

	public int createTexture(ColorMode colorMode, Size2i textureSize) {
		GL2 gl = _context.getGL().getGL2();

		int[] tex = new int[1];
		gl.glGenTextures(1, tex, 0);
		gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, tex[0]);

		gl.glTexParameteri(GL2.GL_TEXTURE_RECTANGLE, GL2.GL_TEXTURE_MIN_FILTER, GL2.GL_LINEAR);
		gl.glTexParameteri(GL2.GL_TEXTURE_RECTANGLE, GL2.GL_TEXTURE_MAG_FILTER, GL2.GL_LINEAR);
		gl.glTexParameteri(GL2.GL_TEXTURE_RECTANGLE, GL2.GL_TEXTURE_WRAP_S, GL2.GL_CLAMP_TO_BORDER);
		gl.glTexParameteri(GL2.GL_TEXTURE_RECTANGLE, GL2.GL_TEXTURE_WRAP_T, GL2.GL_CLAMP_TO_BORDER);
		gl.glTexParameterfv(GL2.GL_TEXTURE_RECTANGLE, GL2.GL_TEXTURE_BORDER_COLOR, FLOAT0000, 0);
		gl.glTexImage2D(GL2.GL_TEXTURE_RECTANGLE, 0, colorMode.glInternalFormat,
				Math.max(1, textureSize.width), Math.max(1, textureSize.height), 0, GL2.GL_BGRA, colorMode.glDataType, null);

		gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);

		return tex[0];
	}

	public int createDepthTexture(Size2i textureSize) {
		GL2 gl = _context.getGL().getGL2();

		int[] tex = new int[1];
		gl.glGenTextures(1, tex, 0);
		gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, tex[0]);

		gl.glTexParameteri(GL2.GL_TEXTURE_RECTANGLE, GL2.GL_TEXTURE_MIN_FILTER, GL2.GL_NEAREST);
		gl.glTexParameteri(GL2.GL_TEXTURE_RECTANGLE, GL2.GL_TEXTURE_MAG_FILTER, GL2.GL_NEAREST);
		gl.glTexParameteri(GL2.GL_TEXTURE_RECTANGLE, GL2.GL_TEXTURE_WRAP_S, GL2.GL_CLAMP_TO_BORDER);
		gl.glTexParameteri(GL2.GL_TEXTURE_RECTANGLE, GL2.GL_TEXTURE_WRAP_T, GL2.GL_CLAMP_TO_BORDER);
		gl.glTexParameterfv(GL2.GL_TEXTURE_RECTANGLE, GL2.GL_TEXTURE_BORDER_COLOR, FLOAT0000, 0);
		gl.glTexImage2D(GL2.GL_TEXTURE_RECTANGLE, 0, GL2.GL_DEPTH_COMPONENT,	// TODO GL_DEPTH_COMPONENT32 等を明示的に指定する方が良い？
				Math.max(1, textureSize.width), Math.max(1, textureSize.height), 0, GL2.GL_DEPTH_COMPONENT, GL2.GL_FLOAT, null);

		gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);

		return tex[0];
	}

	public void deleteTexture(int texture) {
		GL2 gl = _context.getGL().getGL2();
		gl.glDeleteTextures(1, new int[] { texture }, 0);
	}

	public IArray<?> createArray(ColorMode colorMode, Size2i imageSize) {
		int arraySize = imageSize.width * imageSize.height * 4;

		switch (colorMode) {
			case RGBA8:
				return _arrayPools.getByteArray(arraySize);
			case RGBA16:
				return _arrayPools.getShortArray(arraySize);
			case RGBA16_FLOAT:
			case RGBA32_FLOAT:
				return _arrayPools.getFloatArray(arraySize);
		}

		// Never reaches here!
		throw new UnsupportedOperationException("ColorMode: " + colorMode.name());
	}

	public void copyArrayToTexture(Object array, int texture, ColorMode colorMode, Size2i imageSize) {
		ReentrantLock lock = _glGlobal.getGlobalLock();
		lock.lock();
		try {

			GL2 gl = _context.getGL().getGL2();

			int[] pbo = new int[1];
			gl.glGenBuffers(1, pbo, 0);
			gl.glBindBuffer(GL2.GL_PIXEL_UNPACK_BUFFER, pbo[0]);
			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, texture);

			int imageBytes = imageSize.width * imageSize.height * colorMode.javaPixelBytes;

			// 配列 -> PBO
			gl.glBufferData(GL2.GL_PIXEL_UNPACK_BUFFER, imageBytes, toNioBuffer(array), GL2.GL_STREAM_DRAW);

			// PBO -> テクスチャ
			gl.glTexSubImage2D(GL2.GL_TEXTURE_RECTANGLE, 0, 0, 0, imageSize.width, imageSize.height,
					GL2.GL_BGRA, colorMode.glDataType, 0);

			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);
			gl.glBindBuffer(GL2.GL_PIXEL_UNPACK_BUFFER, 0);
			gl.glDeleteBuffers(1, pbo, 0);

		} finally {
			lock.unlock();
		}
	}

	private Buffer toNioBuffer(Object array) {
		if (array instanceof byte[]) {
			return ByteBuffer.wrap((byte[]) array);
			
		} else if (array instanceof short[]) {
			return ShortBuffer.wrap((short[]) array);

		} else if (array instanceof float[]) {
			return FloatBuffer.wrap((float[]) array);

		} else {
			throw new IllegalArgumentException(
					"not array or unsupported array type: "
					+ array.getClass().getName());
		}
	}

	public void copyTextureToArray(int texture, Object array, ColorMode colorMode, Size2i imageSize) {
		ReentrantLock lock = _glGlobal.getGlobalLock();
		lock.lock();
		try {

			GL2 gl = _context.getGL().getGL2();

			int[] pbo = new int[1];
			gl.glGenBuffers(1, pbo, 0);
			gl.glBindBuffer(GL2.GL_PIXEL_PACK_BUFFER, pbo[0]);
			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, texture);

			int imageBytes = imageSize.width * imageSize.height * colorMode.javaPixelBytes;

			// PBOの領域を確保
			gl.glBufferData(GL2.GL_PIXEL_PACK_BUFFER, imageBytes, null, GL2.GL_STREAM_READ);

			// テクスチャをFBOに割当てる
			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, texture, 0);

			// テクスチャ -> PBO
			gl.glReadBuffer(GL2.GL_COLOR_ATTACHMENT0);
			gl.glReadPixels(0, 0, imageSize.width, imageSize.height, GL2.GL_BGRA, colorMode.glDataType, 0);

			// PBO -> 配列
			ByteBuffer mappedPBO = gl.glMapBuffer(GL2.GL_PIXEL_PACK_BUFFER, GL2.GL_READ_ONLY);
			copyPBOToArray(mappedPBO, array);
			gl.glUnmapBuffer(GL2.GL_PIXEL_PACK_BUFFER);

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

			gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);
			gl.glBindBuffer(GL2.GL_PIXEL_PACK_BUFFER, 0);
			gl.glDeleteBuffers(1, pbo, 0);

		} finally {
			lock.unlock();
		}
	}

	private void copyPBOToArray(ByteBuffer mappedPBO, Object array) {
		if (array instanceof byte[]) {
			mappedPBO.get((byte[]) array, 0, mappedPBO.capacity());
			
		} else if (array instanceof short[]) {
			mappedPBO.asShortBuffer().get((short[]) array, 0, mappedPBO.capacity() / 2);

		} else if (array instanceof float[]) {
			mappedPBO.asFloatBuffer().get((float[]) array, 0, mappedPBO.capacity() / 4);

		} else {
			throw new IllegalArgumentException(
					"not array or unsupported array type: "
					+ array.getClass().getName());
		}
	}

	public void clear(VideoBuffer videoBuffer) {
		if (videoBuffer.isArray()) {
			Size2i size = videoBuffer.getImageSize();
			clear(videoBuffer.getArray(), size.width * size.height * 4);
			return;
		}

		if (!videoBuffer.isAllocated()) {
			videoBuffer.allocateAsTexture();
		}

		fill(videoBuffer.getTexture(), Color.COLORLESS_TRANSPARENT);
	}

	private void clear(Object array, int length) {
		if (array instanceof byte[]) {
			Arrays.fill((byte[]) array, 0, length, (byte) 0);

		} else if (array instanceof short[]) {
			Arrays.fill((short[]) array, 0, length, (short) 0);

		} else if (array instanceof float[]) {
			Arrays.fill((float[]) array, 0, length, 0f);

		} else {
			throw new IllegalArgumentException(
					"not array or unsupported array type: "
					+ array.getClass().getName());
		}
	}

	public void fill(VideoBuffer videoBuffer, Color color) {
		// 配列として確保されている場合
		if (videoBuffer.isArray()) {
			fill(videoBuffer.getArray(), videoBuffer.getImageSize(), color);
			return;
		}

		if (!videoBuffer.isAllocated()) {
			videoBuffer.allocateAsTexture();
		}

		if (videoBuffer.getBounds().isEmpty()) {
			// サイズがゼロの場合でも1x1のテクスチャを確保しているので無色透明で塗っておく。
			fill(videoBuffer.getTexture(), Color.COLORLESS_TRANSPARENT);
		} else {
			fill(videoBuffer.getTexture(), color);
		}
	}

	private void fill(int texture, Color color) {
		GL2 gl = _context.getGL().getGL2();

		gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
				GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, texture, 0);
		gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

		float a = (float)color.a;
		float r = (float)color.r * a; 
		float g = (float)color.g * a; 
		float b = (float)color.b * a; 
		gl.glClearColor(r, g, b, a);
		gl.glClear(GL2.GL_COLOR_BUFFER_BIT);

		gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
				GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);
	}

	private void fill(Object array, Size2i imageSize, Color color) {
		// TODO
		throw new UnsupportedOperationException("not implemented");
	}

	public void fillRectCR(
			VideoBuffer resultBuffer,
			final double[] mvMatrix, final double[] prjMatrix,
			final Size2i size, final Color color) {

		final int resultTexture = resultBuffer.getTexture();
		final Size2i resultImageSize = resultBuffer.getImageSize();


		pushMatrixAndExecute(new WrappedOperation<Object>() {
			public Object execute() {
				final GL2 gl = _context.getGL().getGL2();

				gl.glViewport(0, 0, resultImageSize.width, resultImageSize.height);
				gl.glMatrixMode(GL2.GL_PROJECTION);
				gl.glLoadMatrixd(prjMatrix, 0);
				gl.glMatrixMode(GL2.GL_MODELVIEW);
				gl.glLoadMatrixd(mvMatrix, 0);


				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, resultTexture, 0);
				gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

				float[] savedColor = new float[4];
				gl.glGetFloatv(GL2.GL_CURRENT_COLOR, savedColor, 0);

				float a = (float)color.a;
				float r = (float)color.r * a;
				float g = (float)color.g * a;
				float b = (float)color.b * a;
				gl.glColor4f(r, g, b, a);

				_aaSupport.antiAlias(resultImageSize.width, resultImageSize.height, new Runnable() {
					public void run() {
						gl.glBegin(GL2.GL_QUADS);
						gl.glVertex2i(0, 0);
						gl.glVertex2i(size.width, 0);
						gl.glVertex2i(size.width, size.height);
						gl.glVertex2i(0, size.height);
						gl.glEnd();
					}
				});

				gl.glColor4fv(savedColor, 0);

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

				return null;
			}
		});
	}

	private void ortho2D(GL2 gl, GLU glu, Size2i imageSize) {
		ortho2D(gl, glu, imageSize.width, imageSize.height);
	}

	private void ortho2D(GL2 gl, GLU glu, int width, int height) {
		gl.glViewport(0, 0, width, height);

		gl.glMatrixMode(GL2.GL_PROJECTION);
		gl.glLoadIdentity();
		glu.gluOrtho2D(0, width, 0, height);

		gl.glMatrixMode(GL2.GL_MODELVIEW);
		gl.glLoadIdentity();
	}

	public void trackMatte(
			VideoBuffer matteBuffer, VideoBuffer fillBuffer, VideoBuffer resultBuffer,
			final TrackMatte trackMatte, final double matteOpacity) {

		if (trackMatte == TrackMatte.NONE) {
			throw new IllegalArgumentException();
		}

		final int matteTexture = matteBuffer.getTexture();
		final int fillTexture = fillBuffer.getTexture();
		final int resultTexture = resultBuffer.getTexture();
		final Size2i imageSize = resultBuffer.getImageSize();

		pushMatrixAndExecute(new WrappedOperation<Object>() {
			public Object execute() {
				final GL2 gl = _context.getGL().getGL2();

				ortho2D(gl, _context.getGLU(), imageSize);

				// レンダリング先を resultTexture に設定
				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, resultTexture, 0);
				gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

				// レンダリングのソースとなるテクスチャユニットの設定
				gl.glActiveTexture(GL2.GL_TEXTURE0);
				gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, fillTexture);
				gl.glActiveTexture(GL2.GL_TEXTURE1);
				gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, matteTexture);

				final IShaderProgram program = _shaders.getProgram(TrackMatteShaders.class, trackMatte.name());
				program.useProgram(new Runnable() {
					public void run() {
						gl.glUniform1i(program.getUniformLocation("texFill"), 0);
						gl.glUniform1i(program.getUniformLocation("texMatte"), 1);
						gl.glUniform1f(program.getUniformLocation("matteOpacity"), (float)matteOpacity);

						gl.glBegin(GL2.GL_QUADS);
						gl.glTexCoord2f(0, 0);
						gl.glVertex2f(0, 0);
						gl.glTexCoord2f(imageSize.width, 0);
						gl.glVertex2f(imageSize.width, 0);
						gl.glTexCoord2f(imageSize.width, imageSize.height);
						gl.glVertex2f(imageSize.width, imageSize.height);
						gl.glTexCoord2f(0, imageSize.height);
						gl.glVertex2f(0, imageSize.height);
						gl.glEnd();
					}
				});

				gl.glActiveTexture(GL2.GL_TEXTURE1);
				gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);
				gl.glActiveTexture(GL2.GL_TEXTURE0);
				gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

				return null;
			}
		});
	}

	public void blend(
			VideoBuffer srcBuffer, VideoBuffer dstBuffer, VideoBuffer resultBuffer,
			final BlendMode blendMode, final double opacity, final int dissolveSeed) {

		final int srcTexture = srcBuffer.getTexture();
		final int dstTexture = dstBuffer.getTexture();
		final int resultTexture = resultBuffer.getTexture();
		final Size2i imageSize = resultBuffer.getImageSize();

		pushMatrixAndExecute(new WrappedOperation<Object>() {
			public Object execute() {
				final GL2 gl = _context.getGL().getGL2();

				ortho2D(gl, _context.getGLU(), imageSize);

				// レンダリング先を resultTexture に設定
				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, resultTexture, 0);
				gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

				// レンダリングのソースとなるテクスチャユニットの設定
				gl.glActiveTexture(GL2.GL_TEXTURE0);
				gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, dstTexture);
				gl.glActiveTexture(GL2.GL_TEXTURE1);
				gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, srcTexture);

				final IShaderProgram program = _shaders.getProgram(BlendModeShaders.class, blendMode.name());
				program.useProgram(new Runnable() {
					public void run() {
						gl.glUniform1i(program.getUniformLocation("texDst"), 0);
						gl.glUniform1i(program.getUniformLocation("texSrc"), 1);
						gl.glUniform1f(program.getUniformLocation("opacity"), (float)opacity);

						switch (blendMode) {
							case DISSOLVE:
								gl.glUniform1f(program.getUniformLocation("dissolveSeed"), 100f*dissolveSeed/Integer.MAX_VALUE);
								break;
							case DANCING_DISSOLVE:
								gl.glUniform1f(program.getUniformLocation("dissolveSeed"), 100f*dissolveSeed/Integer.MAX_VALUE + (float)_context.getTime().toSecond());
								break;
						}

						gl.glBegin(GL2.GL_QUADS);
						gl.glTexCoord2f(0, 0);
						gl.glVertex2f(0, 0);
						gl.glTexCoord2f(imageSize.width, 0);
						gl.glVertex2f(imageSize.width, 0);
						gl.glTexCoord2f(imageSize.width, imageSize.height);
						gl.glVertex2f(imageSize.width, imageSize.height);
						gl.glTexCoord2f(0, imageSize.height);
						gl.glVertex2f(0, imageSize.height);
						gl.glEnd();
					}
				});

				gl.glActiveTexture(GL2.GL_TEXTURE1);
				gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);
				gl.glActiveTexture(GL2.GL_TEXTURE0);
				gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

				return null;
			}
		});
	}

	public void blend(final Collection<VideoLayerBuffer> srcBuffers, final VideoBuffer dstBuffer, VideoBuffer resultBuffer) {

		final int resultTexture = resultBuffer.getTexture();
		final Size2i imageSize = resultBuffer.getImageSize();

		pushMatrixAndExecute(new WrappedOperation<Object>() {
			public Object execute() {
				final GL2 gl = _context.getGL().getGL2();

				ortho2D(gl, _context.getGLU(), imageSize);

				final int maxLayers = BlendModeShaders.getMaxIntersectingLayers();

				if (srcBuffers.size() > maxLayers) {
					_logger.warn(String.format("number of layers in a inersection group exceeds limit: "
							+ "limit=%d, layers=%d", maxLayers, srcBuffers.size()));
				}
				final int nLayers = Math.min(srcBuffers.size(), maxLayers);

				final int[] blendMode = new int[nLayers];
				final float[] opacity = new float[maxLayers];
				final float[] dissolveSeed = new float[maxLayers];
				float time = (float)_context.getTime().toSecond();

				// レンダリング先を resultTexture に設定
				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, resultTexture, 0);
				gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

				// テクスチャユニットの設定
				gl.glActiveTexture(GL2.GL_TEXTURE0);
				gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, dstBuffer.getTexture());

				int i = 0;
				for (VideoLayerBuffer vlb : srcBuffers) {
					gl.glActiveTexture(GL2.GL_TEXTURE0 + i + 1);
					gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, vlb.getVideoBuffer().getTexture());
					gl.glActiveTexture(GL2.GL_TEXTURE0 + i + 1 + maxLayers);
					gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, vlb.getDepthBuffer().getTexture());
					blendMode[i] = vlb.getBlendMode().ordinal();
					opacity[i] = (float)vlb.getOpacity();

					switch (vlb.getBlendMode()) {
						case DISSOLVE:
							dissolveSeed[i] = 100f*vlb.getDissolveSeed()/Integer.MAX_VALUE;
							break;
						case DANCING_DISSOLVE:
							dissolveSeed[i] = 100f*vlb.getDissolveSeed()/Integer.MAX_VALUE + time;
							break;
					}

					++i;
					if (i >= nLayers) break;
				}

				final IShaderProgram program = _shaders.getProgram(
						BlendModeShaders.getIntersectingBlendSourceFactory(blendMode));
				program.useProgram(new Runnable() {
					public void run() {
						gl.glUniform1i(program.getUniformLocation("nLayers"), nLayers);
						gl.glUniform1i(program.getUniformLocation("texDst"), 0);
						for (int i = 0; i < maxLayers; ++i) {
							gl.glUniform1i(program.getUniformLocation("texSrc" + i), i + 1);
							gl.glUniform1i(program.getUniformLocation("texDep" + i), i + 1 + maxLayers);
						}
						for (int i = 0; i < nLayers; ++i) {
							gl.glUniform1i(program.getUniformLocation("blendMode" + i), blendMode[i]);
						}
						gl.glUniform1fv(program.getUniformLocation("opacity[0]"), maxLayers, opacity, 0);
						gl.glUniform1fv(program.getUniformLocation("dissolveSeed[0]"), maxLayers, dissolveSeed, 0);

						gl.glBegin(GL2.GL_QUADS);
						gl.glTexCoord2f(0, 0);
						gl.glVertex2f(0, 0);
						gl.glTexCoord2f(imageSize.width, 0);
						gl.glVertex2f(imageSize.width, 0);
						gl.glTexCoord2f(imageSize.width, imageSize.height);
						gl.glVertex2f(imageSize.width, imageSize.height);
						gl.glTexCoord2f(0, imageSize.height);
						gl.glVertex2f(0, imageSize.height);
						gl.glEnd();
					}
				});

				for (int j = 0, n = maxLayers * 2 + 1; j < n; ++j) {
					gl.glActiveTexture(GL2.GL_TEXTURE0 + j);
					gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);
				}
				gl.glActiveTexture(GL2.GL_TEXTURE0);

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

				return null;
			}
		});
	}

	public void premultiply(final VideoBuffer vb) {
		pushMatrixAndExecute(new WrappedOperation<Object>() {
			public Object execute() {
				GL2 gl = _context.getGL().getGL2();
				GLU glu = _context.getGLU();

				VideoBounds bounds = vb.getBounds();
				ortho2D(gl, glu, bounds.width, bounds.height);

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, vb.getTexture(), 0);
				gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

				gl.glEnable(GL2.GL_BLEND);
				gl.glBlendFuncSeparate(GL2.GL_ZERO, GL2.GL_DST_ALPHA, GL2.GL_ZERO, GL2.GL_ONE);

				gl.glBegin(GL2.GL_QUADS);
				gl.glVertex2f(0, 0);
				gl.glVertex2f(bounds.width, 0);
				gl.glVertex2f(bounds.width, bounds.height);
				gl.glVertex2f(0, bounds.height);
				gl.glEnd();

				gl.glDisable(GL2.GL_BLEND);

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

				return null;
			}
		});
	}

	public void copy(VideoBuffer src, VideoBuffer dst) {
		final int srcTexture = src.getTexture();
		final VideoBounds srcBounds = src.getBounds();

		final int dstTexture = dst.getTexture();
		final VideoBounds dstBounds = dst.getBounds();


		pushMatrixAndExecute(new WrappedOperation<Object>() {
			public Object execute() {
				GL2 gl = _context.getGL().getGL2();

				ortho2D(gl, _context.getGLU(), dstBounds.width, dstBounds.height);
				gl.glTranslatef((float) -dstBounds.x, (float) -dstBounds.y, 0);


				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, dstTexture, 0);
				gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

				gl.glActiveTexture(GL2.GL_TEXTURE0);
				gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, srcTexture);
				gl.glEnable(GL2.GL_TEXTURE_RECTANGLE);

				float[] savedColor = new float[4];
				gl.glGetFloatv(GL2.GL_CURRENT_COLOR, savedColor, 0);

				gl.glColor4f(1, 1, 1, 1);

				gl.glBegin(GL2.GL_QUADS);
				gl.glTexCoord2f(0, 0);
				gl.glVertex2f((float)srcBounds.x, (float)srcBounds.y);
				gl.glTexCoord2f(srcBounds.width, 0);
				gl.glVertex2f((float)(srcBounds.x+srcBounds.width), (float)srcBounds.y);
				gl.glTexCoord2f(srcBounds.width, srcBounds.height);
				gl.glVertex2f((float)(srcBounds.x+srcBounds.width), (float)(srcBounds.y+srcBounds.height));
				gl.glTexCoord2f(0, srcBounds.height);
				gl.glVertex2f((float)srcBounds.x, (float)(srcBounds.y+srcBounds.height));
				gl.glEnd();

				gl.glColor4fv(savedColor, 0);

				gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);
				gl.glDisable(GL2.GL_TEXTURE_RECTANGLE);
				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

				return null;
			}
		});
	}

	public <T> T pushMatrixAndExecute(WrappedOperation<T> operation) {
		GL2 gl = _context.getGL().getGL2();

		int[] viewport = new int[4];
		gl.glGetIntegerv(GL2.GL_VIEWPORT, viewport, 0);

		int[] matrixMode = new int[1];
		gl.glGetIntegerv(GL2.GL_MATRIX_MODE, matrixMode, 0);

		double[] mvMatrix = new double[16];
		double[] prjMatrix = new double[16];
		getMatrix(mvMatrix, prjMatrix);

		try {

			return operation.execute();

		} finally {
			setMatrix(mvMatrix, prjMatrix);
			gl.glMatrixMode(matrixMode[0]);
			gl.glViewport(viewport[0], viewport[1], viewport[2], viewport[3]);
		}
	}

	public void multModelViewMatrix(
			Vec3d anchorPoint, Vec3d scale, Vec3d orientation, Vec3d rotate, Vec3d position) {

		GL2 gl = _context.getGL().getGL2();
		gl.glMatrixMode(GL2.GL_MODELVIEW);

		// 位置
		gl.glTranslatef((float) position.x, (float) position.y, (float) position.z);

		// 方向
		gl.glRotatef((float) orientation.x, 1, 0, 0);
		gl.glRotatef((float) orientation.y, 0, 1, 0);
		gl.glRotatef((float) orientation.z, 0, 0, 1);

		// 回転
		gl.glRotatef((float) rotate.x, 1, 0, 0);
		gl.glRotatef((float) rotate.y, 0, 1, 0);
		gl.glRotatef((float) rotate.z, 0, 0, 1);

		// 拡大縮小
		gl.glScalef((float) (scale.x/100.0), (float) (scale.y/100.0), (float) (scale.z/100.0));

		// アンカーポイント
		gl.glTranslatef((float) -anchorPoint.x, (float) -anchorPoint.y, (float) -anchorPoint.z);
	}

	public void multModelViewMatrixInverse(
			Vec3d anchorPoint, Vec3d scale, Vec3d orientation, Vec3d rotate, Vec3d position) {

		GL2 gl = _context.getGL().getGL2();
		gl.glMatrixMode(GL2.GL_MODELVIEW);

		// アンカーポイント
		gl.glTranslatef((float) anchorPoint.x, (float) anchorPoint.y, (float) anchorPoint.z);

		// 拡大縮小
		gl.glScalef((float) (100.0/scale.x), (float) (100.0/scale.y), (float) (100.0/scale.z));

		// 回転
		gl.glRotatef((float) -rotate.z, 0, 0, 1);
		gl.glRotatef((float) -rotate.y, 0, 1, 0);
		gl.glRotatef((float) -rotate.x, 1, 0, 0);

		// 方向
		gl.glRotatef((float) -orientation.z, 0, 0, 1);
		gl.glRotatef((float) -orientation.y, 0, 1, 0);
		gl.glRotatef((float) -orientation.x, 1, 0, 0);

		// 位置
		gl.glTranslatef((float) -position.x, (float) -position.y, (float) -position.z);
	}

	public void multCameraMatrix(
			Vec3d orientation, Vec3d rotate, Vec3d position, Vec3d pointOfInterest, boolean flipZ) {

		double dx = pointOfInterest.x - position.x;
		double dy = pointOfInterest.y - position.y;
		double dz = pointOfInterest.z - position.z;

		float _1 = flipZ ? -1 : 1;

		GL2 gl = _context.getGL().getGL2();
		gl.glMatrixMode(GL2.GL_MODELVIEW);

		// 回転
		gl.glRotatef((float) -rotate.z,  0,  0,  1);
		gl.glRotatef((float) -rotate.y,  0, _1,  0);
		gl.glRotatef((float) -rotate.x, _1,  0,  0);

		// 方向
		gl.glRotatef((float) -orientation.z,  0,  0,  1);
		gl.glRotatef((float) -orientation.y,  0, _1,  0);
		gl.glRotatef((float) -orientation.x, _1,  0,  0);

		// 目標点
		gl.glRotatef((float) Math.toDegrees(Math.atan2(dy, Math.sqrt(dx*dx+dz*dz))), _1, 0, 0);
		gl.glRotatef((float) Math.toDegrees(-Math.atan2(dx, dz)), 0, _1, 0);

		// 位置
		gl.glTranslatef((float) -position.x, (float) -position.y, (float) -position.z * _1);

		if (flipZ) {
			gl.glScalef(1, 1, -1);
		}
	}

	public void multCameraMatrixInverse(
			final Vec3d orientation, final Vec3d rotate, final Vec3d position, final Vec3d pointOfInterest) {

		double dx = pointOfInterest.x - position.x;
		double dy = pointOfInterest.y - position.y;
		double dz = pointOfInterest.z - position.z;

		GL2 gl = _context.getGL().getGL2();
		gl.glMatrixMode(GL2.GL_MODELVIEW);

		// 位置
		gl.glTranslatef((float) position.x, (float) position.y, (float) position.z);

		// 目標点
		gl.glRotatef((float) Math.toDegrees(Math.atan2(dx, dz)), 0, 1, 0);
		gl.glRotatef((float) Math.toDegrees(-Math.atan2(dy, Math.sqrt(dx*dx+dz*dz))), 1, 0, 0);

		// 方向
		gl.glRotatef((float) orientation.x, 1, 0, 0);
		gl.glRotatef((float) orientation.y, 0, 1, 0);
		gl.glRotatef((float) orientation.z, 0, 0, 1);

		// 回転
		gl.glRotatef((float) rotate.x, 1, 0, 0);
		gl.glRotatef((float) rotate.y, 0, 1, 0);
		gl.glRotatef((float) rotate.z, 0, 0, 1);
	}

	public void multModelViewMatrix(double[] mvMatrix) {
		GL2 gl = _context.getGL().getGL2();
		gl.glMatrixMode(GL2.GL_MODELVIEW);
		gl.glMultMatrixd(mvMatrix, 0);
	}

	public void getMatrix(double[] mvMatrix, double[] prjMatrix) {
		GL2 gl = _context.getGL().getGL2();
		if (mvMatrix != null) {
			gl.glGetDoublev(GL2.GL_MODELVIEW_MATRIX, mvMatrix, 0);
		}
		if (prjMatrix != null) {
			gl.glGetDoublev(GL2.GL_PROJECTION_MATRIX, prjMatrix, 0);
		}
	}

	public void setMatrix(double[] mvMatrix, double[] prjMatrix) {
		GL2 gl = _context.getGL().getGL2();
		if (prjMatrix != null) {
			gl.glMatrixMode(GL2.GL_PROJECTION);
			gl.glLoadMatrixd(prjMatrix, 0);
		}
		if (mvMatrix != null) {
			gl.glMatrixMode(GL2.GL_MODELVIEW);
			gl.glLoadMatrixd(mvMatrix, 0);
		}
	}

	public void resetMatrix() {
		GL2 gl = _context.getGL().getGL2();
		gl.glMatrixMode(GL2.GL_PROJECTION);
		gl.glLoadIdentity();
		gl.glMatrixMode(GL2.GL_MODELVIEW);
		gl.glLoadIdentity();
	}

	public Vec3d project(Vec3d point, double[] mvMatrix, double[] prjMatrix, Size2i viewportSize) {
		int[] viewport = new int[] { 0, 0, viewportSize.width, viewportSize.height };
		double[] result = new double[3];
		_context.getGLU().gluProject(
				point.x, point.y, point.z,
				mvMatrix, 0, prjMatrix, 0, viewport, 0,
				result, 0);
		return new Vec3d(result[0], result[1], result[2]);
	}

	public void transform(
			VideoBuffer srcBuffer, VideoBuffer resultBuffer, DepthBuffer depthBuffer,
			final double[] mvMatrix, final double[] prjMatrix) {

		final int srcTexture = srcBuffer.getTexture();
		final Size2i srcImageSize = srcBuffer.getImageSize();

		final int resultTexture = resultBuffer.getTexture();
		final Size2i resultImageSize = resultBuffer.getImageSize();

		final int depthTexture = (depthBuffer != null) ? depthBuffer.getTexture() : 0;


		pushMatrixAndExecute(new WrappedOperation<Object>() {
			public Object execute() {
				GL2 gl = _context.getGL().getGL2();

				gl.glViewport(0, 0, resultImageSize.width, resultImageSize.height);
				gl.glMatrixMode(GL2.GL_PROJECTION);
				gl.glLoadMatrixd(prjMatrix, 0);
				gl.glMatrixMode(GL2.GL_MODELVIEW);
				gl.glLoadMatrixd(mvMatrix, 0);


				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, resultTexture, 0);
				gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);
				gl.glActiveTexture(GL2.GL_TEXTURE0);
				gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, srcTexture);
				gl.glEnable(GL2.GL_TEXTURE_RECTANGLE);

				if (depthTexture != 0) {
					gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
							GL2.GL_DEPTH_ATTACHMENT, GL2.GL_TEXTURE_RECTANGLE, depthTexture, 0);
					gl.glEnable(GL2.GL_DEPTH_TEST);
					gl.glClearDepth(1.0);
					gl.glClear(GL2.GL_DEPTH_BUFFER_BIT);
				}

				float[] savedColor = new float[4];
				gl.glGetFloatv(GL2.GL_CURRENT_COLOR, savedColor, 0);

				gl.glColor4f(1, 1, 1, 1);

				gl.glBegin(GL2.GL_QUADS);
				gl.glTexCoord2f(0, 0);
				gl.glVertex2f(0, 0);
				gl.glTexCoord2f(srcImageSize.width, 0);
				gl.glVertex2f(srcImageSize.width, 0);
				gl.glTexCoord2f(srcImageSize.width, srcImageSize.height);
				gl.glVertex2f(srcImageSize.width, srcImageSize.height);
				gl.glTexCoord2f(0, srcImageSize.height);
				gl.glVertex2f(0, srcImageSize.height);
				gl.glEnd();

				gl.glColor4fv(savedColor, 0);

				if (depthTexture != 0) {
					gl.glDisable(GL2.GL_DEPTH_TEST);
					gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
							GL2.GL_DEPTH_ATTACHMENT, GL2.GL_TEXTURE_RECTANGLE, 0, 0);
				}

				gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);
				gl.glDisable(GL2.GL_TEXTURE_RECTANGLE);
				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

				return null;
			}
		});
	}

	public void scaleAndFlipVertical(VideoBuffer srcBuffer, VideoBuffer resultBuffer, final double scale) {
		final int srcTexture = srcBuffer.getTexture();
		final Size2i srcImageSize = srcBuffer.getImageSize();

		final int resultTexture = resultBuffer.getTexture();
		final Size2i resultImageSize = resultBuffer.getImageSize();

		pushMatrixAndExecute(new WrappedOperation<Object>() {
			public Object execute() {
				GL2 gl = _context.getGL().getGL2();
				GLU glu = _context.getGLU();

				gl.glViewport(0, 0, resultImageSize.width, resultImageSize.height);

				gl.glMatrixMode(GL2.GL_PROJECTION);
				gl.glLoadIdentity();
				glu.gluOrtho2D(0, resultImageSize.width, resultImageSize.height, 0);

				gl.glMatrixMode(GL2.GL_MODELVIEW);
				gl.glLoadIdentity();
				gl.glScalef((float) scale, (float) scale, 1);

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, resultTexture, 0);
				gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);
				gl.glActiveTexture(GL2.GL_TEXTURE0);
				gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, srcTexture);
				gl.glEnable(GL2.GL_TEXTURE_RECTANGLE);

				float[] savedColor = new float[4];
				gl.glGetFloatv(GL2.GL_CURRENT_COLOR, savedColor, 0);

				gl.glColor4f(1, 1, 1, 1);

				gl.glBegin(GL2.GL_QUADS);
				gl.glTexCoord2f(0, 0);
				gl.glVertex2f(0, 0);
				gl.glTexCoord2f(srcImageSize.width, 0);
				gl.glVertex2f(srcImageSize.width, 0);
				gl.glTexCoord2f(srcImageSize.width, srcImageSize.height);
				gl.glVertex2f(srcImageSize.width, srcImageSize.height);
				gl.glTexCoord2f(0, srcImageSize.height);
				gl.glVertex2f(0, srcImageSize.height);
				gl.glEnd();

				gl.glColor4fv(savedColor, 0);

				gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);
				gl.glDisable(GL2.GL_TEXTURE_RECTANGLE);
				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

				return null;
			}
		});
	}

	public void accumulate(final List<VideoBuffer> srcBuffers, List<Double> weights, VideoBuffer dstBuffer) {
		final int nUnits = srcBuffers.size();
		final int maxUnits = _glGlobal.getMaxTextureImageUnits();
		if (nUnits > maxUnits) {
			throw new IllegalArgumentException();
		}

		final float[] weightsArray = new float[nUnits];
		for (int i = 0; i < nUnits; ++i) {
			weightsArray[i] = weights.get(i).floatValue();
		}

		final int dstTexture = dstBuffer.getTexture();
		final VideoBounds bounds = dstBuffer.getBounds();

		pushMatrixAndExecute(new WrappedOperation<Object>() {
			public Object execute() {
				final GL2 gl = _context.getGL().getGL2();

				ortho2D(gl, _context.getGLU(), bounds.width, bounds.height);

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, dstTexture, 0);
				gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

				gl.glPushAttrib(GL2.GL_ENABLE_BIT | GL2.GL_COLOR_BUFFER_BIT | GL2.GL_TEXTURE_BIT);

				gl.glEnable(GL2.GL_BLEND);
				gl.glBlendFunc(GL2.GL_ONE, GL2.GL_ONE);
				gl.glBlendEquation(GL2.GL_FUNC_ADD);

				for (int i = 0; i < maxUnits; ++i) {
					gl.glActiveTexture(GL2.GL_TEXTURE0 + i);
					gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, (i < nUnits) ? srcBuffers.get(i).getTexture() : 0);
				}

				final IShaderProgram program = _shaders.getProgram(AccumulationShaders.class, "accumulation");
				program.useProgram(new Runnable() {
					public void run() {
						gl.glUniform1fv(program.getUniformLocation("weights[0]"), nUnits, weightsArray, 0);
						gl.glUniform1i(program.getUniformLocation("nUnits"), nUnits);
						for (int i = 0; i < maxUnits; ++i) {
							gl.glUniform1i(program.getUniformLocation("texSrc" + i), i);
						}

						gl.glBegin(GL2.GL_QUADS);
						gl.glTexCoord2f(0, 0);
						gl.glVertex2f(0, 0);
						gl.glTexCoord2f(bounds.width, 0);
						gl.glVertex2f(bounds.width, 0);
						gl.glTexCoord2f(bounds.width, bounds.height);
						gl.glVertex2f(bounds.width, bounds.height);
						gl.glTexCoord2f(0, bounds.height);
						gl.glVertex2f(0, bounds.height);
						gl.glEnd();
					}
				});

				gl.glPopAttrib();

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

				return null;
			}
		});
	}

	public void accumulate(final List<VideoBuffer> srcBuffers, double weight, VideoBuffer dstBuffer) {
		accumulate(srcBuffers, Collections.nCopies(srcBuffers.size(), weight), dstBuffer);
	}

	private static final int[][] _frustumEdgeTable = {
		{ 0, 1 },		// 0
		{ 0, 2 },		// 1
		{ 0, 6 },		// 2
		{ 1, 3 },		// 3
		{ 1, 7 },		// 4
		{ 2, 3 },		// 5
		{ 2, 4 },		// 6
		{ 3, 5 },		// 7
		{ 4, 5 },		// 8
		{ 4, 6 },		// 9
		{ 5, 7 },		// 10
		{ 6, 7 }		// 11
	};

	private static final int[][] _frustumClipSearchTable = {
		{ 1, 3, 5, 2, 4, 11 },		// 0
		{ 0, 3, 5, 2, 6, 9 },		// 1
		{ 1, 6, 9, 0, 4, 11 },		// 2
		{ 0, 1, 5, 4, 7, 10 },		// 3
		{ 0, 2, 11, 3, 7, 10 },		// 4
		{ 0, 1, 3, 6, 7, 8 },		// 5
		{ 1, 2, 9, 5, 7, 8 },		// 6
		{ 3, 4, 10, 5, 6, 8 },		// 7
		{ 5, 6, 7, 9, 10, 11 },		// 8
		{ 1, 2, 6, 8, 10, 11 },		// 9
		{ 3, 4, 7, 8, 9, 11 },		// 10
		{ 0, 2, 4, 8, 9, 10 }		// 11
	};

	public void fillDepth(DepthBuffer depthBuffer, final double[] mvMatrix, final double[] prjMatrix) {

		Size2i size = depthBuffer.getImageSize();
		final int w = size.width;
		final int h = size.height;
		int[] viewport = new int[] { 0, 0, w, h };
		double[][] unprj = new double[8][3];

		GLU glu = _context.getGLU();
		glu.gluUnProject(0, 0, 0, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[0], 0);
		glu.gluUnProject(0, 0, 1, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[1], 0);
		glu.gluUnProject(w, 0, 0, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[2], 0);
		glu.gluUnProject(w, 0, 1, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[3], 0);
		glu.gluUnProject(w, h, 0, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[4], 0);
		glu.gluUnProject(w, h, 1, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[5], 0);
		glu.gluUnProject(0, h, 0, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[6], 0);
		glu.gluUnProject(0, h, 1, mvMatrix, 0, prjMatrix, 0, viewport, 0, unprj[7], 0);

		Point3d[] frustumVertices = new Point3d[8];
		for (int i = 0; i < 8; ++i) {
			frustumVertices[i] = new Point3d(unprj[i]);
		}

		Point3d[][] frustumEdges = new Point3d[12][2];
		for (int i = 0; i < 12; ++i) {
			frustumEdges[i][0] = frustumVertices[_frustumEdgeTable[i][0]];
			frustumEdges[i][1] = frustumVertices[_frustumEdgeTable[i][1]];
		}

		final Set<Point2d> set = Util.newLinkedHashSet();
		Vector3d v = new Vector3d();

		for (int i = 0; i < 12; ++i) {
			Point3d p0 = frustumEdges[i][0];
			Point3d p1 = frustumEdges[i][1];

			v.sub(p1, p0);

			Point2d p = new Point2d(
					p0.x - v.x/v.z*p0.z,
					p0.y - v.y/v.z*p0.z);

			boolean x = (p0.x <= p.x && p.x <= p1.x) || (p1.x <= p.x && p.x <= p0.x);
			boolean y = (p0.y <= p.y && p.y <= p1.y) || (p1.y <= p.y && p.y <= p0.y);
			if (x && y) {
				set.add(p);

				FRUSTUM_CLIP_SEARCH:
				while (true) {
					for (int j = 0; j < 6; ++j) {
						int k = _frustumClipSearchTable[i][j];
						p0 = frustumEdges[k][0];
						p1 = frustumEdges[k][1];

						v.sub(p1, p0);
	
						p = new Point2d(
								p0.x - v.x/v.z*p0.z,
								p0.y - v.y/v.z*p0.z);
	
						x = (p0.x <= p.x && p.x <= p1.x) || (p1.x <= p.x && p.x <= p0.x);
						y = (p0.y <= p.y && p.y <= p1.y) || (p1.y <= p.y && p.y <= p0.y);
						if (x && y && !set.contains(p)) {
							set.add(p);
							i = k;
							continue FRUSTUM_CLIP_SEARCH;
						}
					}
					break;
				}
				break;
			}
		}


		VideoBuffer tmpBuffer = createVideoBuffer(ColorMode.RGBA8, size);
		try {
			final int tmpTexture = tmpBuffer.getTexture();
			final int depthTexture = depthBuffer.getTexture();

			pushMatrixAndExecute(new WrappedOperation<Object>() {
				public Object execute() {
					GL2 gl = _context.getGL().getGL2();

					gl.glViewport(0, 0, w, h);
					gl.glMatrixMode(GL2.GL_PROJECTION);
					gl.glLoadMatrixd(prjMatrix, 0);
					gl.glMatrixMode(GL2.GL_MODELVIEW);
					gl.glLoadMatrixd(mvMatrix, 0);

					gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
							GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, tmpTexture, 0);
					gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

					gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
							GL2.GL_DEPTH_ATTACHMENT, GL2.GL_TEXTURE_RECTANGLE, depthTexture, 0);
					gl.glEnable(GL2.GL_DEPTH_TEST);
					gl.glClearDepth(1.0);
					gl.glClear(GL2.GL_DEPTH_BUFFER_BIT);

					if (set.size() >= 3) {
						gl.glBegin(GL2.GL_POLYGON);
						for (Point2d pt : set) {
							gl.glVertex2d(pt.x, pt.y);
						}
						gl.glEnd();
					}

					gl.glDisable(GL2.GL_DEPTH_TEST);
					gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
							GL2.GL_DEPTH_ATTACHMENT, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

					gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
							GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

					return null;
				}
			});
		} finally {
			tmpBuffer.dispose();
		}
	}

	public void renderText(
			final FTGLfont font, final Color fillColor,
			final String[] texts, final double[][] offsets,
			VideoBuffer resultBuffer) {

		final VideoBounds resultBounds = resultBuffer.getBounds();
		if (resultBounds.isEmpty()) {
			return;
		}

		final int resultTexture = resultBuffer.getTexture();

		pushMatrixAndExecute(new WrappedOperation<Object>() {
			public Object execute() {
				GL2 gl = _context.getGL().getGL2();
				GLU glu = _context.getGLU();

				float scale = (float) _context.getVideoResolution().scale;

				ortho2D(gl, glu, resultBounds.width, resultBounds.height);
				gl.glTranslatef((float) -resultBounds.x, (float) -resultBounds.y, 0);

				double[] mvMatrix = new double[16];
				getMatrix(mvMatrix, null);

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, resultTexture, 0);
				gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

				// これをしておかないと ftglRenderFont 内の glEnable(GL_TEXTURE_2D) でエラー1282になる。
				gl.glActiveTexture(GL2.GL_TEXTURE0);

				float[] savedColor = new float[4];
				gl.glGetFloatv(GL2.GL_CURRENT_COLOR, savedColor, 0);

				gl.glColor4f((float)fillColor.r, (float)fillColor.g, (float)fillColor.b, (float)fillColor.a);

				// オリジナルのFTGLは内部でアルファブレンドの設定を行っているが、Javieはそれを無効にした修正版FTGLを使用している。
				// Javieには次のアルファブレンドが必要なため。
				gl.glEnable(GL2.GL_BLEND);
				gl.glBlendFuncSeparate(GL2.GL_SRC_ALPHA, GL2.GL_ONE_MINUS_SRC_ALPHA, GL2.GL_ONE, GL2.GL_ONE_MINUS_SRC_ALPHA);

				for (int i = 0; i < texts.length; ++i) {
					setMatrix(mvMatrix, null);
					gl.glTranslatef((float)offsets[i][0], (float)offsets[i][1], 0);
					gl.glScalef(scale, -scale, scale);
					FTGL.ftglRenderFont(font, texts[i], FTGL.FTGL_RENDER_ALL);
				}

				gl.glDisable(GL2.GL_BLEND);

				gl.glColor4fv(savedColor, 0);

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

				return null;
			}
		});
	}

	public void renderText(
			final FTGLfont font, final Color fillColor, final String[] texts, final double[][] offsets,
			VideoBuffer resultBuffer, final double[] mvMatrix, final double[] prjMatrix) {

		final int resultTexture = resultBuffer.getTexture();
		final Size2i resultImageSize = resultBuffer.getImageSize();


		pushMatrixAndExecute(new WrappedOperation<Object>() {
			public Object execute() {
				final GL2 gl = _context.getGL().getGL2();

				final float scale = (float) _context.getVideoResolution().scale;

				gl.glViewport(0, 0, resultImageSize.width, resultImageSize.height);
				gl.glMatrixMode(GL2.GL_PROJECTION);
				gl.glLoadMatrixd(prjMatrix, 0);
				gl.glMatrixMode(GL2.GL_MODELVIEW);
				gl.glLoadMatrixd(mvMatrix, 0);

				final double[] mvMatrix = new double[16];
				getMatrix(mvMatrix, null);


				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, resultTexture, 0);
				gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

				// これをしておかないと ftglRenderFont 内の glEnable(GL_TEXTURE_2D) でエラー1282になる。
				gl.glActiveTexture(GL2.GL_TEXTURE0);

				float[] savedColor = new float[4];
				gl.glGetFloatv(GL2.GL_CURRENT_COLOR, savedColor, 0);

				float a = (float)fillColor.a;
				float r = (float)fillColor.r * a;
				float g = (float)fillColor.g * a;
				float b = (float)fillColor.b * a;
				gl.glColor4f(r, g, b, a);

				_aaSupport.antiAlias(resultImageSize.width, resultImageSize.height, new Runnable() {
					public void run() {
						for (int i = 0; i < texts.length; ++i) {
							setMatrix(mvMatrix, null);
							gl.glTranslatef((float)offsets[i][0], (float)offsets[i][1], 0);
							gl.glScalef(scale, -scale, scale);
							FTGL.ftglRenderFont(font, texts[i], FTGL.FTGL_RENDER_ALL);
						}
					}
				});

				gl.glColor4fv(savedColor, 0);

				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);

				return null;
			}
		});
	}

}
