/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.output;

import java.awt.Transparency;
import java.awt.color.ColorSpace;
import java.awt.image.BufferedImage;
import java.awt.image.ComponentColorModel;
import java.awt.image.DataBuffer;
import java.awt.image.DataBufferByte;
import java.awt.image.DataBufferUShort;
import java.awt.image.Raster;
import java.awt.image.WritableRaster;
import java.io.File;
import java.io.IOException;

import javax.imageio.ImageIO;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.core.AudioBuffer;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.VideoBuffer;

import com.google.inject.Inject;

public class JavaImageSequenceOutput extends PushSourceOutput {

	private File parentFile;

	private boolean png;

	private String filenameFormat;

	@Inject
	private IArrayPools _arrayPools;


	@Override
	protected void initialize(Composition comp, AudioMode audioMode, File file) {
		parentFile = file.getParentFile();

		png = (comp.getColorMode() == ColorMode.RGBA16
				|| file.getName().toLowerCase().endsWith(".png"));

		Time frameDuration = comp.getFrameDuration();
		long numFrames = comp.getDuration().toFrameNumber(frameDuration);
		filenameFormat = createFilenameFormat(file.getName(), numFrames);
	}

	private String createFilenameFormat(String baseFilename, long numFrames) {
		int lastDot = baseFilename.lastIndexOf('.');
		if (lastDot != -1) {
			baseFilename = baseFilename.substring(0, lastDot);
		}
		return baseFilename.replaceAll("%", "%%")
				+ " %0" + String.valueOf(numFrames-1).length() + "d."
				+ (png ? "png" : "bmp");
	}

	@Override
	protected void finish() {
		// nothing to do
	}

	@Override
	protected boolean isVideoOutputting() {
		return true;
	}

	@Override
	protected boolean isAudioOutputting() {
		return false;
	}

	@Override
	protected void writeVideo(long frameNumber, Time time, VideoBuffer vb) {
		try {
			writeToFile(vb, String.format(filenameFormat, frameNumber));
		} catch (IOException e) {
			throw new JavieRuntimeException(e);
		}
	}

	@Override
	protected void writeAudio(long frameNumber, Time time, AudioBuffer ab) {
		throw new UnsupportedOperationException();
	}

	private void writeToFile(VideoBuffer vb, String filename) throws IOException {
		File file = new File(parentFile, filename);

		Size2i size = vb.getImageSize();
		int width = size.width;
		int height = size.height;

		switch (vb.getColorMode()) {
			case RGBA8: {
				byte[] array = (byte[]) vb.getArray();

				if (png) {
					WritableRaster wr = Raster.createInterleavedRaster(
							new DataBufferByte(array, width * height * 4),
							width, height, width*4, 4, new int[] { 2, 1, 0, 3 }, null);
					ComponentColorModel cm = new ComponentColorModel(
							ColorSpace.getInstance(ColorSpace.CS_sRGB), true, false,
							Transparency.TRANSLUCENT, DataBuffer.TYPE_BYTE);
					BufferedImage image = new BufferedImage(cm, wr, false, null);
					ImageIO.write(image, "png", file);

				} else {
					IArray<byte[]> pa = _arrayPools.getByteArray(width*height*3);
					byte[] array2 = pa.getArray();
					try {
						for (int i = 0, n = width*height; i < n; ++i) {
							array2[i*3  ] = array[i*4  ];
							array2[i*3+1] = array[i*4+1];
							array2[i*3+2] = array[i*4+2];
						}
						WritableRaster wr = Raster.createInterleavedRaster(
								new DataBufferByte(array2, width * height * 3),
								width, height, width*3, 3, new int[] { 2, 1, 0 }, null);
						ComponentColorModel cm = new ComponentColorModel(
								ColorSpace.getInstance(ColorSpace.CS_sRGB), false, false,
								Transparency.OPAQUE, DataBuffer.TYPE_BYTE);
						BufferedImage image = new BufferedImage(cm, wr, false, null);
						ImageIO.write(image, "bmp", file);
					} finally {
						pa.release();
					}
				}
				break;
			}
			case RGBA16: {
				short[] array = (short[]) vb.getArray();
				WritableRaster wr = Raster.createInterleavedRaster(
						new DataBufferUShort(array, width * height * 4),
						width, height, width*4, 4, new int[] { 2, 1, 0, 3 }, null);
				ComponentColorModel cm = new ComponentColorModel(
						ColorSpace.getInstance(ColorSpace.CS_sRGB), true, false,
						Transparency.TRANSLUCENT, DataBuffer.TYPE_USHORT);
				BufferedImage image = new BufferedImage(cm, wr, false, null);
				ImageIO.write(image, "png", file);
				break;
			}
			default:
				throw new IllegalArgumentException("unsupported ColorMode: " + vb.getColorMode());
		}
	}

}
