/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.output;

import java.io.File;
import java.io.IOException;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

import javax.sound.sampled.AudioFileFormat;
import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.core.AudioBuffer;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.VideoBuffer;

import com.google.inject.Inject;

public class JavaSoundWaveOutput extends PushSourceOutput {

	private PipedOutputStream outputStream;

	private WaveOutputThread outputThread;

	@Inject
	private IArrayPools _arrayPools;

	@Override
	protected void initialize(Composition comp, AudioMode audioMode, File file) {
		// TODO writeAudio(AudioBuffer)メソッドの処理をちゃんと書けば STEREO_48KHZ_INT16 以外も対応できるはず。
		if (audioMode != AudioMode.STEREO_48KHZ_INT16) {
			throw new UnsupportedOperationException("STEREO_48KHZ_INT16 is only supported.");
		}

		Time duration = comp.getDuration();
		long totalFrames = duration.toFrameNumber(audioMode.sampleDuration);

		outputStream = new PipedOutputStream();
		try {
			outputThread = new WaveOutputThread(outputStream, audioMode, totalFrames, file);
		} catch (IOException e) {
			throw new JavieRuntimeException(e);
		}
		outputThread.start();
	}

	@Override
	protected void finish() {
		try {
			outputStream.close();
		} catch (IOException e) {
			throw new JavieRuntimeException(e);
		}

		try {
			outputThread.join();
		} catch (InterruptedException e) {
			throw new JavieRuntimeException(e);
		}

		Exception e = outputThread.exception;
		if (e != null) {
			throw new JavieRuntimeException(e);
		}
	}

	@Override
	protected boolean isVideoOutputting() {
		return false;
	}

	@Override
	protected boolean isAudioOutputting() {
		return true;
	}

	@Override
	protected void writeVideo(long frameNumber, Time time, VideoBuffer vb) {
		throw new UnsupportedOperationException();
	}

	@Override
	protected void writeAudio(long frameNumber, Time time, AudioBuffer ab) {
		try {
			writeAudio(ab);
		} catch (IOException e) {
			throw new JavieRuntimeException(e);
		}
	}

	private void writeAudio(AudioBuffer ab) throws IOException {
		AudioMode audioMode = ab.getAudioMode();
		int dataLen = ab.getDataLength();
		int dataLenInBytes = ab.getDataLengthInBytes();
		IArray<byte[]> pa = _arrayPools.getByteArray(dataLenInBytes);
		byte[] outBuf = pa.getArray();

		switch (audioMode.dataType) {
			case SHORT:
				ByteBuffer.wrap(outBuf, 0, dataLenInBytes).order(ByteOrder.nativeOrder())
						.asShortBuffer().put((short[]) ab.getData(), 0, dataLen);
				break;
			case INT: {
				int[] data = (int[]) ab.getData();
				for (int i = 0; i < dataLen; ++i) {
					outBuf[i*2  ] = (byte) ((data[i] >>> 16) & 0xff);
					outBuf[i*2+1] = (byte) ((data[i] >>> 24) & 0xff);
				}
				break;
			}
			case FLOAT: {
				float[] data = (float[]) ab.getData();
				for (int i = 0; i < dataLen; ++i) {
					short shortVal = (short) (Math.min(Math.max(data[i], -1.0), 1.0) * Short.MAX_VALUE);
					outBuf[i*2  ] = (byte) ((shortVal      ) & 0xff);
					outBuf[i*2+1] = (byte) ((shortVal >>> 8) & 0xff);
				}
				break;
			}
			default:
				throw new UnsupportedOperationException(
						"unsupported AudioMode.DataType: " + audioMode.dataType);
		}

		outputStream.write(outBuf, 0, dataLenInBytes);
		pa.release();
	}

	private static class WaveOutputThread extends Thread {

		private final PipedInputStream inputStream;

		private final AudioMode audioMode;

		private final long totalFrames;

		private final File file;

		private Exception exception;


		private WaveOutputThread(PipedOutputStream src, AudioMode audioMode, long totalFrames, File file) throws IOException {
			inputStream = new PipedInputStream(src);
			this.audioMode = audioMode;
			this.totalFrames = totalFrames;
			this.file = file;
		}

		public void run() {
			AudioFormat format = new AudioFormat(
					audioMode.sampleRate, audioMode.sampleSize*8, audioMode.channels, true,
					ByteOrder.nativeOrder().equals(ByteOrder.BIG_ENDIAN));
			AudioInputStream ais = new AudioInputStream(inputStream, format, audioMode.frameSize*totalFrames);
			try {
				AudioSystem.write(ais, AudioFileFormat.Type.WAVE, file);
			} catch (IOException e) {
				exception = e;
			} finally {
				try {
					ais.close();
				} catch (IOException e) {
					exception = e;
				}
			}
		}

	}

}
