/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.output;

import java.io.File;
import java.util.List;

import ch.kuramo.javie.api.AudioMode;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.AudioMode.DataType;
import ch.kuramo.javie.core.AudioBuffer;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.VideoBuffer;

public class WindowsDirectShowOutput extends PushSourceOutput {

	private static final int DS_TIME_SCALE = 10000000;


	private VideoCompressorSettings _vcompSettings;

	private long _nativeObjPointer;

	private Time _frameDuration;

	private volatile Time _videoTime;


	public VideoCompressorSettings getVideoCompressorSettings() {
		return _vcompSettings;
	}

	public void setVideoCompressorSettings(VideoCompressorSettings vcompSettings) {
		_vcompSettings = vcompSettings;
	}

	@Override
	protected void initialize(Composition comp, AudioMode audioMode, File file) {
		if (_vcompSettings == null) throw new IllegalStateException("VideoCompressorSettings is not set");

		Size2i size = comp.getSize();
		Time frameDuration = comp.getFrameDuration();
		long dsFrameDuration = (long)(DS_TIME_SCALE*frameDuration.toSecond());

		String vcName = _vcompSettings.moniker;
		double vcQuality = _vcompSettings.quality;
		int vcKeyFrameRate = _vcompSettings.keyFrameRate;
		int vcPFramesPerKey = _vcompSettings.PFramesPerKey;
		long vcBitRate = _vcompSettings.bitRate;
		long vcWindowSize = _vcompSettings.windowSize;
		byte[] vcConfig = _vcompSettings.config;

		long[] nativeObjPointer = new long[1];
		int hr = initialize(nativeObjPointer, file.getAbsolutePath(),
				size.width, size.height, dsFrameDuration, true,				// TODO アルファチャンネル有無の指定
				vcName, vcQuality, vcKeyFrameRate, vcPFramesPerKey,
				vcBitRate, vcWindowSize, vcConfig,
				audioMode.channels, audioMode.sampleRate, audioMode.sampleSize,
				(audioMode.dataType == DataType.FLOAT));
		if (hr != 0) {
			throw new JavieRuntimeException("initialize: hr=" + hr);
		}
		_nativeObjPointer = nativeObjPointer[0];
		_frameDuration = frameDuration;
	}

	@Override
	protected void finish() {
		finish(_nativeObjPointer);
		_nativeObjPointer = 0;
		_frameDuration = null;
	}

	@Override
	protected boolean isVideoOutputting() {
		return true;
	}

	@Override
	protected boolean isAudioOutputting() {
		return true;
	}

	@Override
	protected void videoLoop(Composition comp, AudioMode audioMode, ProgressMonitor monitor) {
		super.videoLoop(comp, audioMode, monitor);
		_videoTime = comp.getDuration();
	}

	@Override
	protected void writeVideo(long frameNumber, Time time, VideoBuffer vb) {
		long dsTimeStart = (long)(DS_TIME_SCALE*time.toSecond());
		long dsTimeEnd = (long)(DS_TIME_SCALE*time.add(_frameDuration).toSecond());
		writeVideo(_nativeObjPointer, dsTimeStart, dsTimeEnd, (byte[])vb.getArray());
		_videoTime = time;
	}

	@Override
	protected void audioLoop(Composition comp, AudioMode audioMode, ProgressMonitor monitor) {
		// オーディオの処理は別スレッドで行われるので、
		// COMの初期化と終了処理を前後に行う必要がある。
		coInitializeEx();
		super.audioLoop(comp, audioMode, monitor);
		coUninitialize();
	}

	@Override
	protected void writeAudio(long frameNumber, Time time, AudioBuffer ab) {
		// インターリーブの設定をするとなぜか書き出し途中で固まる。
		// オーディオの書き出しに適当なスリープを挟んでやるとひとまず回避できるようなので今はそうしている。
		// （でも、問題が再現しにくくなっているだけだと思うが...）

		Time videoTime;
		while ((videoTime = _videoTime) == null || videoTime.before(time)) {
			try { Thread.sleep(100); } catch (InterruptedException e) { }
		}

		long dsTime = (long)(DS_TIME_SCALE*time.toSecond());
		writeAudio(_nativeObjPointer, dsTime, ab.getFrameCount(), ab.getData());
	}


	static { System.loadLibrary("DirectShowOutput"); }

	private native static int initialize(
			long[] nativeObjPointer, String filename,
			int vWidth, int vHeight, long vFrameDuration, boolean vHasAlpha,
			String vcName, double vcQuality, int vcKeyFrameRate, int vcPFramesPerKey,
			long vcBitRate, long vcWindowSize, byte[] vcConfig,
			int aChannels, int aSampleRate, int aSampleSize, boolean aFloat);

	private native static int finish(long nativeObjPointer);

	private native static int writeVideo(long nativeObjPointer,
			long timeStart, long timeEnd, byte[] buffer);

	private native static int writeAudio(long nativeObjPointer,
			long time, int audioFrameCount, Object buffer);

	private native static int coInitializeEx();

	private native static int coUninitialize();



	public static class VideoCompressorDescriptor {
		public final String moniker;
		public final String friendlyName;

		public final int capabilities;
		public final boolean canQuality;
		public final boolean canKeyFrame;
		public final boolean canBFrame;
		public final boolean canCrunch;
		public final boolean canWindow;
		public final boolean canConfigDialog;
		public final boolean canAboutDialog;

		public final double defaultQuality;
		public final int defaultKeyFrameRate;
		public final int defaultPFramesPerKey;
		public final long defaultBitRate;

		public VideoCompressorDescriptor(String moniker, String friendlyName,
				int capabilities, boolean canConfigDialog, boolean canAboutDialog,
				double defaultQuality, int defaultKeyFrameRate, int defaultPFramesPerKey, long defaultBitRate) {

			this.moniker = moniker;
			this.friendlyName = friendlyName;

			this.capabilities = capabilities;
			this.canQuality  = ((capabilities & 0x01) != 0);
			this.canKeyFrame = ((capabilities & 0x04) != 0);
			this.canBFrame   = ((capabilities & 0x08) != 0);
			this.canCrunch   = ((capabilities & 0x02) != 0);
			this.canWindow   = ((capabilities & 0x10) != 0);
			this.canConfigDialog = canConfigDialog;
			this.canAboutDialog = canAboutDialog;

			this.defaultQuality = defaultQuality;
			this.defaultKeyFrameRate = defaultKeyFrameRate;
			this.defaultPFramesPerKey = defaultPFramesPerKey;
			this.defaultBitRate = defaultBitRate;
		}
	}

	public static class VideoCompressorSettings implements Cloneable {
		public final String moniker;
		public double quality;
		public int keyFrameRate;
		public int PFramesPerKey;
		public long bitRate;
		public long windowSize;
		public byte[] config;

		public VideoCompressorSettings(VideoCompressorDescriptor desc) {
			this.moniker = desc.moniker;
			this.quality = desc.defaultQuality;
			this.keyFrameRate = desc.defaultKeyFrameRate;
			this.PFramesPerKey = desc.defaultPFramesPerKey;
			this.bitRate = desc.defaultBitRate;
			this.windowSize = 1;
		}

		public VideoCompressorSettings clone() {
			try {
				return (VideoCompressorSettings) super.clone();
			} catch (CloneNotSupportedException e) {
				throw new Error(e);
			}
		}
	}

	public static List<VideoCompressorDescriptor> listVideoCompressors(Composition comp) {
		Size2i size = comp.getSize();
		Time frameDuration = comp.getFrameDuration();
		long dsFrameDuration = (long)(DS_TIME_SCALE*frameDuration.toSecond());

		List<VideoCompressorDescriptor> list = Util.newList();
		int hr = getVideoCompressors(list, size.width, size.height, dsFrameDuration, true);		// TODO アルファチャンネル有無の指定
		if (hr != 0) {
			throw new JavieRuntimeException("getVideoCompressors: hr=" + hr);
		}
		return list;
	}

	public static void openVideoCompressorConfigDialog(
			Composition comp, VideoCompressorDescriptor desc,
			VideoCompressorSettings settings, long hwnd) {

		if (!desc.moniker.equals(settings.moniker)) {
			throw new IllegalArgumentException();
		}
		if (!desc.canConfigDialog) {
			throw new IllegalArgumentException();
		}

		Size2i size = comp.getSize();
		Time frameDuration = comp.getFrameDuration();
		long dsFrameDuration = (long)(DS_TIME_SCALE*frameDuration.toSecond());

		byte[][] ioConfig = new byte[][] { settings.config };
		int hr = openVideoCompressorConfigDialog(
				size.width, size.height, dsFrameDuration, true,		// TODO アルファチャンネル有無の指定
				desc.moniker, ioConfig, hwnd);
		if (hr != 0) {
			throw new JavieRuntimeException("openVideoCompressorConfigDialog: hr=" + hr);
		}
		settings.config = ioConfig[0];
	}

	public static void openVideoCompressorAboutDialog(
			VideoCompressorDescriptor desc, long hwnd) {

		if (!desc.canAboutDialog) {
			throw new IllegalArgumentException();
		}

		int hr = openVideoCompressorAboutDialog(desc.moniker, hwnd);
		if (hr != 0) {
			throw new JavieRuntimeException("openVideoCompressorAboutDialog: hr=" + hr);
		}
	}

	private native static int getVideoCompressors(
			List<VideoCompressorDescriptor> list,
			int width, int height, long frameDuration, boolean hasAlpha);

	private native static int openVideoCompressorConfigDialog(
			int width, int height, long frameDuration, boolean hasAlpha,
			String moniker, byte[][] ioConfig, long hwnd);

	private native static int openVideoCompressorAboutDialog(String moniker, long hwnd);

}
