/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.shaders;

import java.util.Arrays;
import java.util.List;
import java.util.Set;

import ch.kuramo.javie.api.IShaderSourceFactory;
import ch.kuramo.javie.api.ShaderType;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.GLGlobal;

import com.google.inject.Inject;

public class BlendModeShaders {

	private static final String[] createProgramSource(String name, boolean dissolve) {
		return new String[] {
								"uniform sampler2DRect texDst;",
								"uniform sampler2DRect texSrc;",
								"uniform float opacity;",
				dissolve ?		"uniform float dissolveSeed;" : "",
								"",
				String.format(	"vec4 blend_%s(vec4 pDst, vec4 pSrc, float opacity%s);", name, dissolve ? ", float dissolveSeed" : ""),
								"",
								"void main(void)",
								"{",
								"	vec2 texCoord = gl_TexCoord[0].st;",
								"	vec4 dst = texture2DRect(texDst, texCoord);",
								"	vec4 src = texture2DRect(texSrc, texCoord);",
				String.format(	"	dst = blend_%s(vec4(dst.rgb*dst.a, dst.a), src, opacity%s);", name, dissolve ? ", dissolveSeed" : ""),
								"	gl_FragColor = (dst.a != 0.0) ? vec4(dst.rgb/dst.a, dst.a) : vec4(0.0);",
								"}"
		};
	}

	private static final String[] createProgramSource(String name) {
		return createProgramSource(name, false);
	}


	@ShaderSource(program=false)
	public static final String blend_functions = "blend_functions.frag";


	@ShaderSource(attach="blend_functions")
	public static final String[] NORMAL = createProgramSource("normal");

	@ShaderSource(attach="blend_functions")
	public static final String[] DISSOLVE = createProgramSource("dissolve", true);

	@ShaderSource(attach="blend_functions")
	public static final String[] DANCING_DISSOLVE = DISSOLVE;

	@ShaderSource(attach="blend_functions")
	public static final String[] DARKEN = createProgramSource("darken");

	@ShaderSource(attach="blend_functions")
	public static final String[] MULTIPLY = createProgramSource("multiply");

	@ShaderSource(attach="blend_functions")
	public static final String[] COLOR_BURN = createProgramSource("color_burn");

	@ShaderSource(attach="blend_functions")
	public static final String[] LINEAR_BURN = createProgramSource("linear_burn");

	@ShaderSource(attach="blend_functions")
	public static final String[] DARKER_COLOR = createProgramSource("darker_color");

	@ShaderSource(attach="blend_functions")
	public static final String[] ADD = createProgramSource("add");

	@ShaderSource(attach="blend_functions")
	public static final String[] LIGHTEN = createProgramSource("lighten");

	@ShaderSource(attach="blend_functions")
	public static final String[] SCREEN = createProgramSource("screen");

	@ShaderSource(attach="blend_functions")
	public static final String[] COLOR_DODGE = createProgramSource("color_dodge");

	@ShaderSource(attach="blend_functions")
	public static final String[] LINEAR_DODGE = createProgramSource("linear_dodge");

	@ShaderSource(attach="blend_functions")
	public static final String[] LIGHTER_COLOR = createProgramSource("lighter_color");

	@ShaderSource(attach="blend_functions")
	public static final String[] OVERLAY = createProgramSource("overlay");

	@ShaderSource(attach="blend_functions")
	public static final String[] SOFT_LIGHT = createProgramSource("soft_light");

	@ShaderSource(attach="blend_functions")
	public static final String[] HARD_LIGHT = createProgramSource("hard_light");

	@ShaderSource(attach="blend_functions")
	public static final String[] LINEAR_LIGHT = createProgramSource("linear_light");

	@ShaderSource(attach="blend_functions")
	public static final String[] VIVID_LIGHT = createProgramSource("vivid_light");

	@ShaderSource(attach="blend_functions")
	public static final String[] PIN_LIGHT = createProgramSource("pin_light");

	@ShaderSource(attach="blend_functions")
	public static final String[] HARD_MIX = createProgramSource("hard_mix");

	@ShaderSource(attach="blend_functions")
	public static final String[] DIFFERENCE = createProgramSource("difference");

	@ShaderSource(attach="blend_functions")
	public static final String[] EXCLUSION = createProgramSource("exclusion");

	@ShaderSource(attach="blend_functions")
	public static final String[] HUE = createProgramSource("hue");

	@ShaderSource(attach="blend_functions")
	public static final String[] SATURATION = createProgramSource("saturation");

	@ShaderSource(attach="blend_functions")
	public static final String[] COLOR = createProgramSource("color");

	@ShaderSource(attach="blend_functions")
	public static final String[] LUMINOSITY = createProgramSource("luminosity");

	@ShaderSource(attach="blend_functions")
	public static final String[] STENCIL_ALPHA = createProgramSource("stencil_alpha");

	@ShaderSource(attach="blend_functions")
	public static final String[] STENCIL_LUMA = createProgramSource("stencil_luma");

	@ShaderSource(attach="blend_functions")
	public static final String[] SILHOUETTE_ALPHA = createProgramSource("silhouette_alpha");

	@ShaderSource(attach="blend_functions")
	public static final String[] SILHOUETTE_LUMA = createProgramSource("silhouette_luma");



	@ShaderSource(program=false)
	public final String[] intersecting_blend_main;


	private static int _maxIntersectingLayers;

	public static int getMaxIntersectingLayers() {
		return _maxIntersectingLayers;
	}


	private String[] createIntersectingBlendMain() {
		List<String> source = Util.newList();

		// 以下のような変な書き方をしているのは、普通に書いたら動かなかったから。
		// (Macではsampler2DRectの配列をuniform変数にしても、配列のインデックスを定数にしないと動かない)
		// (GMAではint配列のuniform変数に何を渡しても、値が0になっている)

		// あと、シェーダ内で配列を使うと遅くなるようなので、ord0-ordNは配列にしていない。
		// TODO texture2DRectでサンプリングした値は変数に保持しておいた方がいいかも。
		//      ただし配列ではなくord0-ordNと同じような形で。


		/********************
		const int maxLayers = N;

		uniform int nLayers;
		uniform sampler2DRect texDst;
		uniform sampler2DRect texSrc0;
		uniform sampler2DRect texDep0;
		uniform int blendMode0;
		...
		uniform sampler2DRect texSrcN;
		uniform sampler2DRect texDepN;
		uniform int blendModeN;
		uniform float opacity[maxLayers];
		uniform float dissolveSeed[maxLayers];

		vec2 texCoord = gl_TexCoord[0].st;
		********************/

		source.add(String.format("const int maxLayers = %d;", _maxIntersectingLayers));

		source.add("uniform int nLayers;");
		source.add("uniform sampler2DRect texDst;");

		for (int i = 0; i < _maxIntersectingLayers; ++i) {
			source.add(String.format("uniform sampler2DRect texSrc%d;", i));
			source.add(String.format("uniform sampler2DRect texDep%d;", i));
			source.add(String.format("uniform int blendMode%d;", i));
		}

		source.add("uniform float opacity[maxLayers];");
		source.add("uniform float dissolveSeed[maxLayers];");

		source.add("vec2 texCoord = gl_TexCoord[0].st;");


		/********************
		vec4 getSrc(int i) {
			return
				(i == 0) ? texture2DRect(texSrc0, texCoord) :
				...
				(i == N) ? texture2DRect(texSrcN, texCoord) :
				vec4(0.0);
		}
		********************/

		source.add("vec4 getSrc(int i) {");
		source.add("	return");
		for (int i = 0; i < _maxIntersectingLayers; ++i) {
			source.add(String.format("		(i == %1$d) ? texture2DRect(texSrc%1$d, texCoord) :", i));
		}
		source.add("		vec4(0.0);");
		source.add("}");


		/********************
		float getDep(int i) {
			return
				(i == 0) ? texture2DRect(texDep0, texCoord).r :
				...
				(i == N) ? texture2DRect(texDepN, texCoord).r :
				-1.0;
		}
		********************/

		source.add("float getDep(int i) {");
		source.add("	return");
		for (int i = 0; i < _maxIntersectingLayers; ++i) {
			source.add(String.format("		(i == %1$d) ? texture2DRect(texDep%1$d, texCoord).r :", i));
		}
		source.add("		-1.0;");
		source.add("}");


		/********************
		int getBlendMode(int i) {
			return
				(i == 0) ? blendMode0 :
				...
				(i == N) ? blendModeN :
				0;
		}
		********************/

		source.add("int getBlendMode(int i) {");
		source.add("	return");
		for (int i = 0; i < _maxIntersectingLayers; ++i) {
			source.add(String.format("		(i == %1$d) ? blendMode%1$d :", i));
		}
		source.add("		0;");
		source.add("}");


		/********************
		int ord0 = 0;
		...
		int ordN = N;
		********************/

		for (int i = 0; i < _maxIntersectingLayers; ++i) {
			source.add(String.format("int ord%1$d = %1$d;", i));
		}


		/********************
		int getOrd(int i) {
			return
				(i == 0) ? ord0 :
				...
				(i == N) ? ordN :
				-1;
		}
		********************/

		source.add("int getOrd(int i) {");
		source.add("	return");
		for (int i = 0; i < _maxIntersectingLayers; ++i) {
			source.add(String.format("		(i == %1$d) ? ord%1$d :", i));
		}
		source.add("		-1;");
		source.add("}");


		/********************
		void setOrd(int i, int ord) {
			if (i == 0) ord0 = ord; else
			...
			if (i == N) ordN = ord;
		}
		********************/

		source.add("void setOrd(int i, int ord) {");
		for (int i = 0; i < _maxIntersectingLayers-1; ++i) {
			source.add(String.format("	if (i == %1$d) ord%1$d = ord; else ", i));
		}
		source.add(String.format("	if (i == %1$d) ord%1$d = ord;", _maxIntersectingLayers-1));
		source.add("}");


		/********************/

		source.addAll(Arrays.asList(new String[] {
				"vec4 blend(int mode, vec4 pDst, vec4 pSrc, float opacity, float dissolveSeed);",
				"",
				"void main(void)",
				"{",
				"	vec4 dst = texture2DRect(texDst, texCoord);",
				"",
				// texSrcNのアルファ値が全てゼロ(nzaCount == 0)の場合、texDstの値をそのまま使えばよい。
				// texSrcNのうちひとつだけがゼロ以外のアルファを持つ(nzaCount == 1)場合、そのテクスチャだけをtexDstとブレンドすればよい。
				"	int nzaCount = 0;",
				"	int nzaLayer;",
				"	vec4 nzaColor;",
				"	for (int i = 0; i < nLayers; ++i) {",
				"		vec4 s = getSrc(i);",
				"		if (s.a > 0.0) {",
				"			if (++nzaCount > 1) {",
				"				break;",
				"			}",
				"			nzaLayer = i;",
				"			nzaColor = s;",
				"		}",
				"	}",
				"	if (nzaCount == 1) {",
				"		dst = vec4(dst.rgb*dst.a, dst.a);",
				"		dst = blend(getBlendMode(nzaLayer), dst, nzaColor, opacity[nzaLayer], dissolveSeed[nzaLayer]);",
				"		dst = (dst.a != 0.0) ? vec4(dst.rgb/dst.a, dst.a) : vec4(0.0);",
				"",
				"	} else if (nzaCount > 1) {",
				"		dst = vec4(dst.rgb*dst.a, dst.a);",
				"		",
				"		for (int i = nLayers - 2; i >= 0; --i) {",
				"			int ord = getOrd(i);",
				"			float dep = getDep(ord);",
				"			int j, ord2;",
				"			for (j = i + 1; j < nLayers && getDep(ord2 = getOrd(j)) > dep; ++j) {",
				"				setOrd(j - 1, ord2);",
				"			}",
				"			setOrd(j - 1, ord);",
				"		}",
				"		",
				"		for (int i = 0; i < nLayers; ++i) {",
				"			int ord = getOrd(i);",
				"			dst = blend(getBlendMode(ord), dst, getSrc(ord), opacity[ord], dissolveSeed[ord]);",
				"		}",
				"		dst = (dst.a != 0.0) ? vec4(dst.rgb/dst.a, dst.a) : vec4(0.0);",
				"	}",
				"",
				"	gl_FragColor = dst;",
				"}"
		}));

		return source.toArray(new String[source.size()]);
	}

	@Inject
	public BlendModeShaders(GLGlobal glGlobal) {
		// TODO 32個より多くのテクスチャユニットを扱えるのかどうか？
		//
		// GL_TEXTURE* は GL_TEXTURE0 から GL_TEXTURE31 の32個しか定義されていない。
		// OpenGLの仕様上、テクスチャユニットは32個までということなのだろうか？
		// とりあえず32個が上限と仮定して制限しておく。
		// （あまり多く交差するのはパフォーマンス的な問題もあるし）
		//
		_maxIntersectingLayers = (Math.min(32, glGlobal.getMaxTextureImageUnits()) - 1) / 2;

		intersecting_blend_main = createIntersectingBlendMain();
	}

	public static IShaderSourceFactory getIntersectingBlendSourceFactory(int[] blendModes) {
		final Set<Integer> set = Util.newSet();
		for (int i = 0; i < blendModes.length; ++i) {
			set.add(blendModes[i]);
		}
		Integer[] array = set.toArray(new Integer[set.size()]);
		Arrays.sort(array);

		StringBuilder sb = new StringBuilder(BlendModeShaders.class.getName() + ".INTERSECTING_BLEND");
		for (Integer i : array) {
			sb.append(":").append(i);
		}
		final String name = sb.toString();

		final String[] attach = {
				BlendModeShaders.class.getName() + ".blend_functions",
				BlendModeShaders.class.getName() + ".intersecting_blend_main"
		};


		return new IShaderSourceFactory() {
			public String getName()		{ return name; }
			public ShaderType getType()	{ return ShaderType.FRAGMENT_SHADER; }
			public boolean isProgram()	{ return true; }
			public String[] getAttach()	{ return attach; }

			public String[] getSource() {
				return new String[] {
										"vec4 blend_normal(vec4 pDst, vec4 pSrc, float opacity);",
					 set.contains(1) ||
					 set.contains(2)  ? "vec4 blend_dissolve(vec4 pDst, vec4 pSrc, float opacity, float seed);" : "",

					 set.contains(3)  ? "vec4 blend_darken(vec4 pDst, vec4 pSrc, float opacity);" : "",
					 set.contains(4)  ? "vec4 blend_multiply(vec4 pDst, vec4 pSrc, float opacity);" : "",
					 set.contains(5)  ? "vec4 blend_color_burn(vec4 pDst, vec4 pSrc, float opacity);" : "",
					 set.contains(6)  ? "vec4 blend_linear_burn(vec4 pDst, vec4 pSrc, float opacity);" : "",
					 set.contains(7)  ? "vec4 blend_darker_color(vec4 pDst, vec4 pSrc, float opacity);" : "",

					 set.contains(8)  ? "vec4 blend_add(vec4 pDst, vec4 pSrc, float opacity);" : "",
					 set.contains(9)  ? "vec4 blend_lighten(vec4 pDst, vec4 pSrc, float opacity);" : "",
					 set.contains(10) ? "vec4 blend_screen(vec4 pDst, vec4 pSrc, float opacity);" : "",
					 set.contains(11) ? "vec4 blend_color_dodge(vec4 pDst, vec4 pSrc, float opacity);" : "",
					 set.contains(12) ? "vec4 blend_linear_dodge(vec4 pDst, vec4 pSrc, float opacity);" : "",
					 set.contains(13) ? "vec4 blend_lighter_color(vec4 pDst, vec4 pSrc, float opacity);" : "",

					 set.contains(14) ? "vec4 blend_overlay(vec4 pDst, vec4 pSrc, float opacity);" : "",
					 set.contains(15) ? "vec4 blend_soft_light(vec4 pDst, vec4 pSrc, float opacity);" : "",
					 set.contains(16) ? "vec4 blend_hard_light(vec4 pDst, vec4 pSrc, float opacity);" : "",
					 set.contains(17) ? "vec4 blend_linear_light(vec4 pDst, vec4 pSrc, float opacity);" : "",
					 set.contains(18) ? "vec4 blend_vivid_light(vec4 pDst, vec4 pSrc, float opacity);" : "",
					 set.contains(19) ? "vec4 blend_pin_light(vec4 pDst, vec4 pSrc, float opacity);" : "",
					 set.contains(20) ? "vec4 blend_hard_mix(vec4 pDst, vec4 pSrc, float opacity);" : "",

					 set.contains(21) ? "vec4 blend_difference(vec4 pDst, vec4 pSrc, float opacity);" : "",
					 set.contains(22) ? "vec4 blend_exclusion(vec4 pDst, vec4 pSrc, float opacity);" : "",

					 set.contains(23) ? "vec4 blend_hue(vec4 pDst, vec4 pSrc, float opacity);" : "",
					 set.contains(24) ? "vec4 blend_saturation(vec4 pDst, vec4 pSrc, float opacity);" : "",
					 set.contains(25) ? "vec4 blend_color(vec4 pDst, vec4 pSrc, float opacity);" : "",
					 set.contains(26) ? "vec4 blend_luminosity(vec4 pDst, vec4 pSrc, float opacity);" : "",

//					 set.contains(27) ? "vec4 blend_stencil_alpha(vec4 pDst, vec4 pSrc, float opacity);" : "",
//					 set.contains(28) ? "vec4 blend_stencil_luma(vec4 pDst, vec4 pSrc, float opacity);" : "",
					 set.contains(29) ? "vec4 blend_silhouette_alpha(vec4 pDst, vec4 pSrc, float opacity);" : "",
					 set.contains(30) ? "vec4 blend_silhouette_luma(vec4 pDst, vec4 pSrc, float opacity);" : "",

										"vec4 blend(int mode, vec4 pDst, vec4 pSrc, float opacity, float dissolveSeed)",
										"{",
										"	return",
					 set.contains(0)  ? "		(mode == 0) ? blend_normal(pDst, pSrc, opacity) :" : "",
					 set.contains(1)  ? "		(mode == 1) ? blend_dissolve(pDst, pSrc, opacity, dissolveSeed) :" : "",
					 set.contains(2)  ? "		(mode == 2) ? blend_dissolve(pDst, pSrc, opacity, dissolveSeed) :" : "",

					 set.contains(3)  ? "		(mode == 3) ? blend_darken(pDst, pSrc, opacity) :" : "",
					 set.contains(4)  ? "		(mode == 4) ? blend_multiply(pDst, pSrc, opacity) :" : "",
					 set.contains(5)  ? "		(mode == 5) ? blend_color_burn(pDst, pSrc, opacity) :" : "",
					 set.contains(6)  ? "		(mode == 6) ? blend_linear_burn(pDst, pSrc, opacity) :" : "",
					 set.contains(7)  ? "		(mode == 7) ? blend_darker_color(pDst, pSrc, opacity) :" : "",

					 set.contains(8)  ? "		(mode == 8) ? blend_add(pDst, pSrc, opacity) :" : "",
					 set.contains(9)  ? "		(mode == 9) ? blend_lighten(pDst, pSrc, opacity) :" : "",
					 set.contains(10) ? "		(mode == 10) ? blend_screen(pDst, pSrc, opacity) :" : "",
					 set.contains(11) ? "		(mode == 11) ? blend_color_dodge(pDst, pSrc, opacity) :" : "",
					 set.contains(12) ? "		(mode == 12) ? blend_linear_dodge(pDst, pSrc, opacity) :" : "",
					 set.contains(13) ? "		(mode == 13) ? blend_lighter_color(pDst, pSrc, opacity) :" : "",

					 set.contains(14) ? "		(mode == 14) ? blend_overlay(pDst, pSrc, opacity) :" : "",
					 set.contains(15) ? "		(mode == 15) ? blend_soft_light(pDst, pSrc, opacity) :" : "",
					 set.contains(16) ? "		(mode == 16) ? blend_hard_light(pDst, pSrc, opacity) :" : "",
					 set.contains(17) ? "		(mode == 17) ? blend_linear_light(pDst, pSrc, opacity) :" : "",
					 set.contains(18) ? "		(mode == 18) ? blend_vivid_light(pDst, pSrc, opacity) :" : "",
					 set.contains(19) ? "		(mode == 19) ? blend_pin_light(pDst, pSrc, opacity) :" : "",
					 set.contains(20) ? "		(mode == 20) ? blend_hard_mix(pDst, pSrc, opacity) :" : "",

					 set.contains(21) ? "		(mode == 21) ? blend_difference(pDst, pSrc, opacity) :" : "",
					 set.contains(22) ? "		(mode == 22) ? blend_exclusion(pDst, pSrc, opacity) :" : "",

					 set.contains(23) ? "		(mode == 23) ? blend_hue(pDst, pSrc, opacity) :" : "",
					 set.contains(24) ? "		(mode == 24) ? blend_saturation(pDst, pSrc, opacity) :" : "",
					 set.contains(25) ? "		(mode == 25) ? blend_color(pDst, pSrc, opacity) :" : "",
					 set.contains(26) ? "		(mode == 26) ? blend_luminosity(pDst, pSrc, opacity) :" : "",

//TODO ステンシルアルファとステンシルルミナンスは、次のふたつの理由で3Dレイヤーでは正しく動作しない。
//      1. レイヤー矩形の外側はアルファ値がゼロなので、ノンゼロアルファ(nza)のテストで落とされてしまう（これはシェーダの修正だけで解決可能）
//      2. レイヤー矩形の外側はデプス値が最大なので、ソートしたときに一番奥に配置されてしまう（これはシェーダの修正だけでは解決できない）
//
//					 set.contains(27) ? "		(mode == 27) ? blend_stencil_alpha(pDst, pSrc, opacity) :" : "",
//					 set.contains(28) ? "		(mode == 28) ? blend_stencil_luma(pDst, pSrc, opacity) :" : "",
					 set.contains(29) ? "		(mode == 29) ? blend_silhouette_alpha(pDst, pSrc, opacity) :" : "",
					 set.contains(30) ? "		(mode == 30) ? blend_silhouette_luma(pDst, pSrc, opacity) :" : "",

										"					   blend_normal(pDst, pSrc, opacity);",
										"}"
				};
			}
		};
	}

}
