/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects;

import java.util.HashMap;
import java.util.Map;

import javax.media.opengl.GL2;

import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;

public class BlendModeShaders {

	public static final String BLEND_FUNCTIONS = "ch.kuramo.javie.core.shaders.BlendModeShaders.blend_functions";


	public static class Premult {

		private static String[] createSource(String name) {
			return new String[] {
									"uniform sampler2DRect texDst;",
									"uniform sampler2DRect texSrc;",
									"uniform float opacity;",
									"",
					String.format(	"vec4 blend_%s(vec4 pDst, vec4 pSrc, float opacity);", name),
									"",
									"void main(void)",
									"{",
									"	vec2 texCoord = gl_TexCoord[0].st;",
									"	vec4 dst = texture2DRect(texDst, texCoord);",
									"	vec4 src = texture2DRect(texSrc, texCoord);",
					String.format(	"	gl_FragColor = blend_%s(dst, src, opacity);", name),
									"}"
			};
		}

		@ShaderSource
		public static final String[] NONE = {
			"uniform sampler2DRect texSrc;",
			"uniform float opacity;",
			"",
			"void main(void)",
			"{",
			"	vec4 src = texture2DRect(texSrc, gl_TexCoord[0].st);",
			"	gl_FragColor = src * opacity;",
			"}"
		};

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] NORMAL = createSource("normal");

//		@ShaderSource(attach=BLEND_FUNCTIONS)
//		public static final String[] DISSOLVE = createSource("dissolve");
//
//		@ShaderSource(attach=BLEND_FUNCTIONS)
//		public static final String[] DANCING_DISSOLVE = createSource("dancing_dissolve");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DARKEN = createSource("darken");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] MULTIPLY = createSource("multiply");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] COLOR_BURN = createSource("color_burn");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LINEAR_BURN = createSource("linear_burn");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DARKER_COLOR = createSource("darker_color");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] ADD = createSource("add");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LIGHTEN = createSource("lighten");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SCREEN = createSource("screen");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] COLOR_DODGE = createSource("color_dodge");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LINEAR_DODGE = createSource("linear_dodge");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LIGHTER_COLOR = createSource("lighter_color");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] OVERLAY = createSource("overlay");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SOFT_LIGHT = createSource("soft_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] HARD_LIGHT = createSource("hard_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LINEAR_LIGHT = createSource("linear_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] VIVID_LIGHT = createSource("vivid_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] PIN_LIGHT = createSource("pin_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] HARD_MIX = createSource("hard_mix");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DIFFERENCE = createSource("difference");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] EXCLUSION = createSource("exclusion");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] HUE = createSource("hue");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SATURATION = createSource("saturation");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] COLOR = createSource("color");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LUMINOSITY = createSource("luminosity");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] STENCIL_ALPHA = createSource("stencil_alpha");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] STENCIL_LUMA = createSource("stencil_luma");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SILHOUETTE_ALPHA = createSource("silhouette_alpha");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SILHOUETTE_LUMA = createSource("silhouette_luma");

	}


	public static class SrcUnmult {

		private static String[] createSource(String name) {
			return new String[] {
									"uniform sampler2DRect texDst;",
									"uniform sampler2DRect texSrc;",
									"uniform float opacity;",
									"",
					String.format(	"vec4 blend_%s(vec4 pDst, vec4 pSrc, float opacity);", name),
									"",
									"void main(void)",
									"{",
									"	vec2 texCoord = gl_TexCoord[0].st;",
									"	vec4 dst = texture2DRect(texDst, texCoord);",
									"	vec4 src = texture2DRect(texSrc, texCoord);",
					String.format(	"	gl_FragColor = blend_%s(dst, vec4(src.rgb*src.a, src.a), opacity);", name),
									"}"
			};
		}

		@ShaderSource
		public static final String[] NONE = {
			"uniform sampler2DRect texSrc;",
			"uniform float opacity;",
			"",
			"void main(void)",
			"{",
			"	vec4 src = texture2DRect(texSrc, gl_TexCoord[0].st);",
			"	gl_FragColor = vec4(src.rgb*src.a, src.a) * opacity;",
			"}"
		};

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] NORMAL = createSource("normal");

//		@ShaderSource(attach=BLEND_FUNCTIONS)
//		public static final String[] DISSOLVE = createSource("dissolve");
//
//		@ShaderSource(attach=BLEND_FUNCTIONS)
//		public static final String[] DANCING_DISSOLVE = createSource("dancing_dissolve");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DARKEN = createSource("darken");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] MULTIPLY = createSource("multiply");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] COLOR_BURN = createSource("color_burn");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LINEAR_BURN = createSource("linear_burn");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DARKER_COLOR = createSource("darker_color");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] ADD = createSource("add");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LIGHTEN = createSource("lighten");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SCREEN = createSource("screen");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] COLOR_DODGE = createSource("color_dodge");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LINEAR_DODGE = createSource("linear_dodge");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LIGHTER_COLOR = createSource("lighter_color");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] OVERLAY = createSource("overlay");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SOFT_LIGHT = createSource("soft_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] HARD_LIGHT = createSource("hard_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LINEAR_LIGHT = createSource("linear_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] VIVID_LIGHT = createSource("vivid_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] PIN_LIGHT = createSource("pin_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] HARD_MIX = createSource("hard_mix");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DIFFERENCE = createSource("difference");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] EXCLUSION = createSource("exclusion");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] HUE = createSource("hue");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SATURATION = createSource("saturation");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] COLOR = createSource("color");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LUMINOSITY = createSource("luminosity");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] STENCIL_ALPHA = createSource("stencil_alpha");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] STENCIL_LUMA = createSource("stencil_luma");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SILHOUETTE_ALPHA = createSource("silhouette_alpha");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SILHOUETTE_LUMA = createSource("silhouette_luma");

	}


	public static BlendModeShaders forPremult(IVideoEffectContext context, IShaderRegistry shaders) {
		return new BlendModeShaders(context, shaders, Premult.class);
	}

	public static BlendModeShaders forSrcUnmult(IVideoEffectContext context, IShaderRegistry shaders) {
		return new BlendModeShaders(context, shaders, SrcUnmult.class);
	}


	private final IVideoEffectContext context;

	private final IShaderRegistry shaders;

	private final Class<?> shadersClass;

	private final Map<BlendMode, IShaderProgram> programs = new HashMap<BlendMode, IShaderProgram>();


	public BlendModeShaders(IVideoEffectContext context, IShaderRegistry shaders, Class<?> shadersClass) {
		this.context = context;
		this.shaders = shaders;
		this.shadersClass = shadersClass;

		for (BlendMode blendMode : BlendMode.values()) {
			IShaderProgram program = shaders.getProgram(shadersClass, blendMode.name());
			if (program == null) {
				throw new IllegalArgumentException(String.format(
						"no shader program found for BlendMode %s in class %s", blendMode.name(), shadersClass.getName()));
			}
			programs.put(blendMode, program);
		}
	}

	public void replace(BlendMode blendMode, IShaderProgram program) {
		if (program == null) {
			program = shaders.getProgram(shadersClass, blendMode.name());
		}
		programs.put(blendMode, program);
	}

	public IVideoBuffer blend(IVideoBuffer src, IVideoBuffer dst, final BlendMode blendMode, final double opacity) {
		IVideoBuffer buffer = null;
		try {
			VideoBounds bounds = src.getBounds();
			buffer = context.createVideoBuffer(bounds);

			final GL2 gl = context.getGL().getGL2();
			final int w = bounds.width;
			final int h = bounds.height;
			VideoEffectUtil.ortho2D(gl, context.getGLU(), w, h);

			gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
					GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, buffer.getTexture(), 0);
			gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

			gl.glPushAttrib(GL2.GL_TEXTURE_BIT);
			try {
				if (blendMode != BlendMode.NONE) {
					gl.glActiveTexture(GL2.GL_TEXTURE0);
					gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, dst.getTexture());
				}
				gl.glActiveTexture(GL2.GL_TEXTURE1);
				gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, src.getTexture());

				final IShaderProgram program = programs.get(blendMode);
				program.useProgram(new Runnable() {
					public void run() {
						if (blendMode != BlendMode.NONE) {
							gl.glUniform1i(program.getUniformLocation("texDst"), 0);
						}
						gl.glUniform1i(program.getUniformLocation("texSrc"), 1);
						gl.glUniform1f(program.getUniformLocation("opacity"), (float)opacity);

						gl.glBegin(GL2.GL_QUADS);
						gl.glTexCoord2f(0, 0);
						gl.glVertex2f(0, 0);
						gl.glTexCoord2f(w, 0);
						gl.glVertex2f(w, 0);
						gl.glTexCoord2f(w, h);
						gl.glVertex2f(w, h);
						gl.glTexCoord2f(0, h);
						gl.glVertex2f(0, h);
						gl.glEnd();
					}
				});

			} finally {
				gl.glPopAttrib();
				
				gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
						GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);
			}

			IVideoBuffer result = buffer;
			buffer = null;
			return result;

		} finally {
			if (buffer != null) buffer.dispose();
		}
	}

}
