/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects;

import javax.media.opengl.GL2;
import javax.media.opengl.glu.GLU;

import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.services.IShaderRegistry;

public class VideoEffectUtil {

	public static void ortho2D(GL2 gl, GLU glu, int width, int height) {
		gl.glViewport(0, 0, width, height);

		gl.glMatrixMode(GL2.GL_PROJECTION);
		gl.glLoadIdentity();
		glu.gluOrtho2D(0, width, 0, height);

		gl.glMatrixMode(GL2.GL_MODELVIEW);
		gl.glLoadIdentity();
	}

	private static final String[] createConvolution1DSource(boolean h) {
		return new String[] {
				"uniform sampler2DRect texture;",
				"uniform int radius;",
				"uniform float kernel[101];",
				"",
				"void main(void)",
				"{",
				"	vec2 texCoord = gl_TexCoord[0].st;",
				"	vec4 sum = vec4(0.0);",
				"	for (int i = -radius; i <= radius; ++i) {",
			h ?	"		vec4 src = texture2DRect(texture, vec2(texCoord.s+float(i), texCoord.t));"
			  :	"		vec4 src = texture2DRect(texture, vec2(texCoord.s, texCoord.t+float(i)));",
				"		sum += kernel[radius+i] * src;",
				"	}",
				"	gl_FragColor = sum;",
				"}"
		};		
	}

	@ShaderSource
	public static final String[] CONVOLUTION_1D_H = createConvolution1DSource(true);

	@ShaderSource
	public static final String[] CONVOLUTION_1D_V = createConvolution1DSource(false);


	public static void convolution1D(
			IVideoBuffer src, IVideoBuffer dst,
			boolean horizontal, final int radius, final float[] kernel,
			final GL2 gl, GLU glu, IShaderRegistry shaders) {

		VideoBounds srcBounds = src.getBounds();
		VideoBounds dstBounds = dst.getBounds();
		final float x1 = (float) (dstBounds.x - srcBounds.x);
		final float y1 = (float) (dstBounds.y - srcBounds.y);
		final float x2 = x1 + dstBounds.width;
		final float y2 = y1 + dstBounds.height;

		ortho2D(gl, glu, dstBounds.width, dstBounds.height);
		gl.glTranslatef(-x1, -y1, 0);

		gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
				GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, dst.getTexture(), 0);
		gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

		gl.glActiveTexture(GL2.GL_TEXTURE0);
		gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, src.getTexture());

		final IShaderProgram program = shaders.getProgram(
				VideoEffectUtil.class, horizontal ? "CONVOLUTION_1D_H" : "CONVOLUTION_1D_V");
		program.useProgram(new Runnable() {
			public void run() {
				gl.glUniform1i(program.getUniformLocation("texture"), 0);
				gl.glUniform1i(program.getUniformLocation("radius"), radius);
				gl.glUniform1fv(program.getUniformLocation("kernel[0]"), radius*2+1, kernel, 0);

				gl.glBegin(GL2.GL_QUADS);
				gl.glTexCoord2f(x1, y1);
				gl.glVertex2f(x1, y1);
				gl.glTexCoord2f(x2, y1);
				gl.glVertex2f(x2, y1);
				gl.glTexCoord2f(x2, y2);
				gl.glVertex2f(x2, y2);
				gl.glTexCoord2f(x1, y2);
				gl.glVertex2f(x1, y2);
				gl.glEnd();
			}
		});

		gl.glActiveTexture(GL2.GL_TEXTURE0);
		gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);

		gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
				GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);
	}

	@ShaderSource
	public static final String[] CONVOLUTION = {
		"uniform sampler2DRect texture;",
		"uniform int ksize;",
		"uniform float kernel[49];",
		"uniform vec2 offset[49];",
		"",
		"void main(void)",
		"{",
		"	vec2 texCoord = gl_TexCoord[0].st;",
		"	vec4 sum = vec4(0.0);",
		"	for (int i = 0; i < ksize; ++i) {",
		"		vec4 src = texture2DRect(texture, texCoord + offset[i]);",
		"		sum += kernel[i] * src;",
		"	}",
		"	gl_FragColor = sum;",
		"}"
	};

	public static void convolution(
			IVideoBuffer src, IVideoBuffer dst,
			final int ksize, final float[] kernel, final float[] offset,
			final GL2 gl, GLU glu, IShaderRegistry shaders) {

		VideoBounds srcBounds = src.getBounds();
		VideoBounds dstBounds = dst.getBounds();
		final float x1 = (float) (dstBounds.x - srcBounds.x);
		final float y1 = (float) (dstBounds.y - srcBounds.y);
		final float x2 = x1 + dstBounds.width;
		final float y2 = y1 + dstBounds.height;

		ortho2D(gl, glu, dstBounds.width, dstBounds.height);
		gl.glTranslatef(-x1, -y1, 0);

		gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
				GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, dst.getTexture(), 0);
		gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

		gl.glActiveTexture(GL2.GL_TEXTURE0);
		gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, src.getTexture());

		final IShaderProgram program = shaders.getProgram(VideoEffectUtil.class, "CONVOLUTION");
		program.useProgram(new Runnable() {
			public void run() {
				gl.glUniform1i(program.getUniformLocation("texture"), 0);
				gl.glUniform1i(program.getUniformLocation("ksize"), ksize);
				gl.glUniform1fv(program.getUniformLocation("kernel[0]"), ksize, kernel, 0);
				gl.glUniform2fv(program.getUniformLocation("offset[0]"), ksize, offset, 0);

				gl.glBegin(GL2.GL_QUADS);
				gl.glTexCoord2f(x1, y1);
				gl.glVertex2f(x1, y1);
				gl.glTexCoord2f(x2, y1);
				gl.glVertex2f(x2, y1);
				gl.glTexCoord2f(x2, y2);
				gl.glVertex2f(x2, y2);
				gl.glTexCoord2f(x1, y2);
				gl.glVertex2f(x1, y2);
				gl.glEnd();
			}
		});

		gl.glActiveTexture(GL2.GL_TEXTURE0);
		gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);

		gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
				GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);
	}

	public static VideoBounds expandBounds(VideoBounds bounds, int expand, boolean h, boolean v) {
		int dx = 0, dy = 0, dw = 0, dh = 0;

		if (h) {
			dx = -expand;
			dw = expand*2;
		}

		if (v) {
			dy = -expand;
			dh = expand*2;
		}

		return new VideoBounds(
				bounds.x + dx, bounds.y + dy,
				bounds.width + dw, bounds.height + dh);
	}

	public static void setNearest(IVideoBuffer vb, GL2 gl) {
		gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, vb.getTexture());
		gl.glTexParameteri(GL2.GL_TEXTURE_RECTANGLE, GL2.GL_TEXTURE_MIN_FILTER, GL2.GL_NEAREST);
		gl.glTexParameteri(GL2.GL_TEXTURE_RECTANGLE, GL2.GL_TEXTURE_MAG_FILTER, GL2.GL_NEAREST);
		gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);
	}

	public static void setLinear(IVideoBuffer vb, GL2 gl) {
		gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, vb.getTexture());
		gl.glTexParameteri(GL2.GL_TEXTURE_RECTANGLE, GL2.GL_TEXTURE_MIN_FILTER, GL2.GL_LINEAR);
		gl.glTexParameteri(GL2.GL_TEXTURE_RECTANGLE, GL2.GL_TEXTURE_MAG_FILTER, GL2.GL_LINEAR);
		gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);
	}

	public static void setClampToEdge(IVideoBuffer vb, GL2 gl) {
		gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, vb.getTexture());
		gl.glTexParameteri(GL2.GL_TEXTURE_RECTANGLE, GL2.GL_TEXTURE_WRAP_S, GL2.GL_CLAMP_TO_EDGE);
		gl.glTexParameteri(GL2.GL_TEXTURE_RECTANGLE, GL2.GL_TEXTURE_WRAP_T, GL2.GL_CLAMP_TO_EDGE);
		gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);
	}

	public static void setClampToBorder(IVideoBuffer vb, GL2 gl) {
		gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, vb.getTexture());
		gl.glTexParameteri(GL2.GL_TEXTURE_RECTANGLE, GL2.GL_TEXTURE_WRAP_S, GL2.GL_CLAMP_TO_BORDER);
		gl.glTexParameteri(GL2.GL_TEXTURE_RECTANGLE, GL2.GL_TEXTURE_WRAP_T, GL2.GL_CLAMP_TO_BORDER);
		gl.glBindTexture(GL2.GL_TEXTURE_RECTANGLE, 0);
	}

	public static void clearTexture(IVideoBuffer vb, GL2 gl) {
		if (vb.isArray()) {
			throw new IllegalArgumentException();
		}

		gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
				GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, vb.getTexture(), 0);
		gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

		gl.glClearColor(0, 0, 0, 0);
		gl.glClear(GL2.GL_COLOR_BUFFER_BIT);

		gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
				GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);
	}

	// TODO VideoRenderSupportImplにもほぼ同じものがあるので整理する。
	public static void premultiply(IVideoBuffer vb, GL2 gl, GLU glu) {
		VideoBounds bounds = vb.getBounds();
		ortho2D(gl, glu, bounds.width, bounds.height);

		gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
				GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, vb.getTexture(), 0);
		gl.glDrawBuffer(GL2.GL_COLOR_ATTACHMENT0);

		gl.glEnable(GL2.GL_BLEND);
		gl.glBlendFuncSeparate(GL2.GL_ZERO, GL2.GL_DST_ALPHA, GL2.GL_ZERO, GL2.GL_ONE);

		gl.glBegin(GL2.GL_QUADS);
		gl.glVertex2f(0, 0);
		gl.glVertex2f(bounds.width, 0);
		gl.glVertex2f(bounds.width, bounds.height);
		gl.glVertex2f(0, bounds.height);
		gl.glEnd();

		gl.glDisable(GL2.GL_BLEND);

		gl.glFramebufferTexture2D(GL2.GL_FRAMEBUFFER,
				GL2.GL_COLOR_ATTACHMENT0, GL2.GL_TEXTURE_RECTANGLE, 0, 0);
	}


	private VideoEffectUtil() { }

}
